module Main where


import Control.Monad (when)
import Control.Monad.State
import Data.Bits (shiftL)
import qualified Data.Map.Strict as Map
import System.Exit (exitSuccess)
import System.IO

import Bestmove
import Bitboard
import Fen

--  https://backscattering.de/chess/uci/
-- Chess engine using fen notation and bitboards demo 
-- #################################################################
uciPrint :: String
uciPrint =
  unlines
    $ [ "id name HaskellChess 1.0"
      , "id author nik"
      , ""
      , "option name Debug Log File type string default"
      , "option name Threads type spin default 1 min 1 max 16" -- can set to more maybe --> threads are at compile time, but this is sparks 
      , "uciok" -- pass this at the end for uci
      ]

data EngineConfig = EngineConfig
  { name :: String
  , author :: String
  , debugMode :: Int
  , threads :: Int
  , fen :: FEN
  }

instance Show EngineConfig where
  show config =
    unlines
      $ [ "id name " ++ name config
        , "id author " ++ author config
        , ""
        , "option name Debug " ++ show (debugMode config)
        , "option name Threads " ++ show (threads config)
        , "starting position " ++ show (fen config)
        ]

setConfig :: Int -> Int -> FEN -> EngineConfig
setConfig t d f =
  EngineConfig
    { name = "HaskellChess 1.0"
    , author = "nik"
    , threads = t
    , debugMode = d
    , fen = f
    }

applyMove :: Board -> Int -> Int -> Board
applyMove board fromSquare toSquare =
  let fromBit = 1 `shiftL` (fromSquare)
      toBit = 1 `shiftL` (toSquare)
      clearBoard = clearSquare board toSquare
   in case findPiece board fromSquare of
        Just ("pawnsWhite", bb) ->
          clearBoard {pawnsWhite = updateBitboard bb fromBit toBit}
        Just ("pawnsBlack", bb) ->
          clearBoard {pawnsBlack = updateBitboard bb fromBit toBit}
        Just ("knightsWhite", bb) ->
          clearBoard {knightsWhite = updateBitboard bb fromBit toBit}
        Just ("knightsBlack", bb) ->
          clearBoard {knightsBlack = updateBitboard bb fromBit toBit}
        Just ("bishopsWhite", bb) ->
          clearBoard {bishopsWhite = updateBitboard bb fromBit toBit}
        Just ("bishopsBlack", bb) ->
          clearBoard {bishopsBlack = updateBitboard bb fromBit toBit}
        Just ("rooksWhite", bb) ->
          clearBoard {rooksWhite = updateBitboard bb fromBit toBit}
        Just ("rooksBlack", bb) ->
          clearBoard {rooksBlack = updateBitboard bb fromBit toBit}
        Just ("queensWhite", bb) ->
          clearBoard {queensWhite = updateBitboard bb fromBit toBit}
        Just ("queensBlack", bb) ->
          clearBoard {queensBlack = updateBitboard bb fromBit toBit}
        Just ("kingsWhite", bb) ->
          clearBoard {kingsWhite = updateBitboard bb fromBit toBit}
        Just ("kingsBlack", bb) ->
          clearBoard {kingsBlack = updateBitboard bb fromBit toBit}
        Nothing -> error "No piece found"
        _ -> error "No piece found"

  -- sum each layer of the board with corresponding piece value 
  -- use popcount on each Bitboard 

-- minimax best move solver
-- board position, color to move, 
-- this should be in the
runEngine :: Int -> Int -> FEN -> Board -> IO ()
runEngine threads dMode fenConfig board = do
  -- let config = setConfig threads dMode fenConfig
  guiIn <- getLine
  case words guiIn of
    -- set up the configurations of the engine 
    ["quit"] -> exitSuccess
    ["uci"] -> putStr uciPrint >> runEngine threads dMode fenConfig board
    -- option setting
    ["debug", val] ->
      case val of
        "on" -> runEngine threads 1 fenConfig board
        "off" -> runEngine threads 0 fenConfig board
        _ -> runEngine threads 0 fenConfig board
    -- set starting positions
    ["position", "startpos"] -> do
      when (dMode == 1) (putStrLn "Recieved starting position without moves")
      runEngine threads dMode setDefaultFen board
    -- need to change this to handle list of moves
    "position":"startpos":"moves":moves -> do
      when
        (dMode == 1)
        (putStrLn
           ("Recieved starting position with moves "
              ++ unwords moves
              ++ ". Adding the last move "
              ++ last moves
              ++ " to the state."))
      let (fromSquare, toSquare) = splitAt 2 (last moves) -- is monadic, doesn't need an in 
          newBoard =
            applyMove board (squareLookup fromSquare) (squareLookup toSquare)
      -- when (dMode == 1) (putStrLn (show newBoard))
      runEngine threads dMode setDefaultFen newBoard
    -- start a new game
    ["ucinewgame"] -> runEngine 1 1 setDefaultFen board
    -- ask the engine if it is available
    ["isready"] -> putStrLn "readyok" >> runEngine threads dMode fenConfig board
    -- start calculating best move based on fen position
    ["go", "infinite"] -> do
      let color =
            if (activeColor fenConfig) == 'b'
              then 1
              else -1
      -- let color = -1
      let dep = 4
      putStrLn
        ("Starting minimax for cpu on color: "
           ++ show color
           ++ ". DEPTH: "
           ++ show dep)
      let (bestMove, _) = runState (chooseBestMovePar color board dep) Map.empty
      let moveStr = showMove (extractMove bestMove) color
      -- let bestMove1 = chooseBestMove color board 3
      -- let moveStr = showMove (bestMove1) color

      putStrLn ("move: " ++ moveStr)
      let (fromSquare, toSquare) = splitAt 2 moveStr
          newBoard =
            applyMove board (squareLookup fromSquare) (squareLookup toSquare)
      putStrLn fromSquare
      putStrLn toSquare
      -- when (dMode == 1) (putStrLn (show newBoard))
      putStrLn ("bestmove " ++ moveStr)
      runEngine threads dMode fenConfig newBoard
    ["stop"] -> exitSuccess -- todo exit whatever calculation is currently ongong kill current calculation --> always start camcluation in separate thread and then kill when appropriate
    -- not 100% sure what this does 
    ["ponderhit"] -> runEngine threads dMode fenConfig board -- can ignore this 
    _ -> runEngine threads dMode fenConfig board

main :: IO ()
main = do
  hSetBuffering stdout NoBuffering
  hSetBuffering stdin NoBuffering
  runEngine 1 1 setDefaultFen startpos
