{-# LANGUAGE DeriveGeneric #-}
{-# LANGUAGE DeriveAnyClass #-}

module Bitboard where

import Control.DeepSeq (NFData)
import Data.Bits
  ( (.&.)
  , (.|.)
  , complement
  , shiftL
  , shiftR
  , testBit
  , xor
  )
import Data.Char (intToDigit)
import Data.Word (Word64)
import GHC.Generics (Generic)
import Numeric (showIntAtBase)

-- Move type that encodes each move
-- Doesn't compute the capture here, does that in evaluate board
-- #################################################################
data Move = Move
  { moveFrom :: Int -- save as a bitboard
  , moveTo :: Int -- also save as a bitboard
  , promotion :: String
  , isCapture :: Int
  , isCastling :: Int
  , isEnPassant :: Int
  } deriving (Show, Eq, Generic, NFData)

-- instance NFData Move where
--   rnf (Move f t p c e) = rnf f `seq` rnf t `seq` rnf p `seq` rnf c `seq` rnf e
movesGrid :: [Move] -> String
movesGrid moves = showGrid combinedBitboard
  where
    combinedBitboard = foldr combineMove 0 moves
    combineMove move acc = acc .|. 1 `shiftL` moveTo move

showMove :: Move -> Int -> String
showMove move color =
  squareToAlgebraic (moveFrom move) color
    ++ squareToAlgebraic (moveTo move) color

squareToAlgebraic :: Int -> Int -> String
squareToAlgebraic square color =
  let file =
        if color == 1
          then ['A' .. 'H'] !! (square `mod` 8)
          else ['a' .. 'h'] !! (square `mod` 8)
      rank = show (1 + (square `div` 8))
   in [file] ++ rank

-- #################################################################
type Bitboard = Word64 -- 8*8 bits

data Board = Board
  { pawnsWhite :: Bitboard
  , pawnsBlack :: Bitboard
  , knightsBlack :: Bitboard
  , knightsWhite :: Bitboard
  , bishopsBlack :: Bitboard
  , bishopsWhite :: Bitboard
  , rooksBlack :: Bitboard
  , rooksWhite :: Bitboard
  , queensWhite :: Bitboard
  , queensBlack :: Bitboard
  , kingsWhite :: Bitboard
  , kingsBlack :: Bitboard
  }

instance Show Board where
  show board =
    unlines
      [ "Pawns (White):   " ++ helper (pawnsWhite board)
      , "Pawns (Black):   " ++ helper (pawnsBlack board)
      , "Knights (White): " ++ helper (knightsWhite board)
      , "Knights (Black): " ++ helper (knightsBlack board)
      , "Bishops (White): " ++ helper (bishopsWhite board)
      , "Bishops (Black): " ++ helper (bishopsBlack board)
      , "Rooks (White):   " ++ helper (rooksWhite board)
      , "Rooks (Black):   " ++ helper (rooksBlack board)
      -- , "Castles (White): " ++ helper (castlesWhite board)
      -- , "Castles (Black): " ++ helper (castlesBlack board)
      , "Queens (White):  " ++ helper (queensWhite board)
      , "Queens (Black):  " ++ helper (queensBlack board)
      , "Kings (White):   " ++ helper (kingsWhite board)
      , "Kings (Black):   " ++ helper (kingsBlack board)
      ]
    where
      helper :: Word64 -> String
      helper bitboard =
        let x = showIntAtBase 16 intToDigit bitboard ""
         in "0x" ++ replicate (16 - length x) '0' ++ x

showPieceGrid :: Board -> String -> String
showPieceGrid board pieceType =
  case pieceType of
    "PawnsWhite" -> showGrid (pawnsWhite board)
    "PawnsBlack" -> showGrid (pawnsBlack board)
    "KnightsWhite" -> showGrid (knightsWhite board)
    "KnightsBlack" -> showGrid (knightsBlack board)
    "BishopsWhite" -> showGrid (bishopsWhite board)
    "BishopsBlack" -> showGrid (bishopsBlack board)
    "RooksWhite" -> showGrid (rooksWhite board)
    "RooksBlack" -> showGrid (rooksBlack board)
    "QueensWhite" -> showGrid (queensWhite board)
    "QueensBlack" -> showGrid (queensBlack board)
    "KingsWhite" -> showGrid (kingsWhite board)
    "KingsBlack" -> showGrid (kingsBlack board)
    _ -> "Invalid piece type"

showGrid :: Bitboard -> String
showGrid bitboard = unlines $ map showRow [7,6 .. 0]
  where
    -- Create an 8x8 binary grid representation
    showRow row = unwords $ map (showSquare row) [0 .. 7]
    showSquare row col =
      if testBit bitboard (row * 8 + col)
        then "1"
        else "0"

showBin :: Bitboard -> String
showBin bitboard =
  let x = showIntAtBase 2 intToDigit bitboard ""
   in replicate (64 - length x) '0' ++ x

-- hesadecimal instaed of binary. Each row corresponds to two hex values 
-- each piece type has it's own board --> can or all together for the entire board (64 bits) 
-- in the lookup table below, we need to check for each piece type if the desired square is occupied
startpos :: Board
startpos =
  Board
    { pawnsWhite = 0x000000000000FF00 -- [a,h]2 entire row 
    , pawnsBlack = 0x00FF000000000000 -- [a,h]7 entire row
    , knightsWhite = 0x0000000000000042 -- b1 (bit 1) and g1 (bit 6)
    , knightsBlack = 0x4200000000000000 -- b8 (bit 57) and g8 (bit 62)
    , bishopsWhite = 0x0000000000000024 -- c1 (bit 2) and f1 (bit 5)
    , bishopsBlack = 0x2400000000000000 -- c8 (bit 58) and f8 (bit 61)
    , rooksWhite = 0x0000000000000081 -- a1 (bit 0) and h1 (bit 7)
    , rooksBlack = 0x8100000000000000 -- a8 (bit 56) and h8 (bit 63)
    , queensWhite = 0x0000000000000008 -- d1 (bit 3)
    , queensBlack = 0x0800000000000000 -- d8 (bit 59)
    , kingsWhite = 0x0000000000000010 -- e1 (bit 4)
    , kingsBlack = 0x1000000000000000 -- e8 (bit 60)
    }

squareLookup :: String -> Int
squareLookup square =
  let (file, rank) = splitAt 1 square
      x = fromEnum (head file) - fromEnum 'a'
      y = read rank - 1
   in y * 8 + x

-- and with a negative mask to clear. Safe to do over whole board for all pieces since only one piece can occupy a square. saves tedious checking
clearSquare :: Board -> Int -> Board
clearSquare board square =
  let mask = complement (1 `shiftL` square) -- Mask to clear the square
   in board
        { pawnsWhite = pawnsWhite board .&. mask
        , pawnsBlack = pawnsBlack board .&. mask
        , knightsWhite = knightsWhite board .&. mask
        , knightsBlack = knightsBlack board .&. mask
        , bishopsWhite = bishopsWhite board .&. mask
        , bishopsBlack = bishopsBlack board .&. mask
        , rooksWhite = rooksWhite board .&. mask
        , rooksBlack = rooksBlack board .&. mask
        , queensWhite = queensWhite board .&. mask
        , queensBlack = queensBlack board .&. mask
        , kingsWhite = kingsWhite board .&. mask
        , kingsBlack = kingsBlack board .&. mask
        }

findPiece :: Board -> Int -> Maybe (String, Bitboard)
findPiece board square =
  let bit = 1 `shiftL` square
   in case () of
        _
          | pawnsWhite board .&. bit /= 0 ->
            Just ("pawnsWhite", pawnsWhite board)
          | pawnsBlack board .&. bit /= 0 ->
            Just ("pawnsBlack", pawnsBlack board)
          | knightsWhite board .&. bit /= 0 ->
            Just ("knightsWhite", knightsWhite board)
          | knightsBlack board .&. bit /= 0 ->
            Just ("knightsBlack", knightsBlack board)
          | bishopsWhite board .&. bit /= 0 ->
            Just ("bishopsWhite", bishopsWhite board)
          | bishopsBlack board .&. bit /= 0 ->
            Just ("bishopsBlack", bishopsBlack board)
          | rooksWhite board .&. bit /= 0 ->
            Just ("rooksWhite", rooksWhite board)
          | rooksBlack board .&. bit /= 0 ->
            Just ("rooksBlack", rooksBlack board)
          | queensWhite board .&. bit /= 0 ->
            Just ("queensWhite", queensWhite board)
          | queensBlack board .&. bit /= 0 ->
            Just ("queensBlack", queensBlack board)
          | kingsWhite board .&. bit /= 0 ->
            Just ("kingsWhite", kingsWhite board)
          | kingsBlack board .&. bit /= 0 ->
            Just ("kingsBlack", kingsBlack board)
          | otherwise -> Nothing

-- xor the frombit with the board to get rid of it, or the tobit to write to that spot 
updateBitboard :: Bitboard -> Bitboard -> Bitboard -> Bitboard
updateBitboard bitboard fromBit toBit = (bitboard `xor` fromBit) .|. toBit

-- extract active squares from bitboard
-- #################################################################
bitboardSquares :: Bitboard -> [Int]
bitboardSquares bitboard =
  [i | i <- [0 .. 63], bitboard .&. (1 `shiftL` i) /= 0]

-- add a function to uppercase if white else lowercase
-- occupancy is the entire and of each board
-- add castling to king piece
generateAllMovesWhite :: Board -> [Move]
generateAllMovesWhite board =
  let whites =
        (pawnsWhite board)
          .|. (knightsWhite board)
          .|. (bishopsWhite board)
          .|. (rooksWhite board)
          .|. (queensWhite board)
          .|. (kingsWhite board)
      blacks =
        (pawnsBlack board)
          .|. (knightsBlack board)
          .|. (bishopsBlack board)
          .|. (rooksBlack board)
          .|. (queensBlack board)
          .|. (kingsBlack board)
      occupancy = blacks .|. whites
      captureable = blacks
   in (generateAllPawnMovesW (pawnsWhite board) occupancy captureable)
        ++ (generateAllKnightMoves (knightsWhite board) occupancy captureable)
        ++ (generateAllBishopMoves (bishopsWhite board) occupancy captureable)
        ++ (generateAllRookMoves (rooksWhite board) occupancy captureable)
        ++ (generateAllQueenMoves (queensWhite board) occupancy captureable)
        ++ (generateAllKingMoves (kingsWhite board) occupancy captureable)

generateAllMovesBlack :: Board -> [Move]
generateAllMovesBlack board =
  let whites =
        (knightsWhite board)
          .|. (bishopsWhite board)
          .|. (rooksWhite board)
          .|. (queensWhite board)
          .|. (kingsWhite board)
          .|. (pawnsWhite board)
      blacks =
        (knightsBlack board)
          .|. (bishopsBlack board)
          .|. (rooksBlack board)
          .|. (queensBlack board)
          .|. (kingsBlack board)
          .|. (pawnsBlack board)
      occupancy = blacks .|. whites
      captureable = whites
   in (generateAllPawnMovesB (pawnsBlack board) occupancy captureable) 
        ++ (generateAllBishopMoves (bishopsBlack board) occupancy captureable)
        ++ (generateAllRookMoves (rooksBlack board) occupancy captureable)
        ++ (generateAllQueenMoves (queensBlack board) occupancy captureable)
        ++ (generateAllKingMoves (kingsBlack board) occupancy captureable)
        ++ (generateAllKnightMoves (knightsBlack board) occupancy captureable)

-- castle 
-- understand en passant 
-- do a board eval
{-
Here is the api for moves. I follow largely the same structure for each piece even though under the hood generation is quite different .
All piece move generators can be called through generateAllXMoves and take the same arguments, making piece generation simpler.
-}
-- NOTE : I SWITCHED LEFT AND RIGHT 
-- LEFT IS 7 SHIFTL AND RIGHT IS 9 NOT THE OTHER WAY AROUND. IT DOESN'T MAKE A DIFFERENCE THOUGH
fileA, fileH, fileAB, fileGH :: Bitboard
fileA = 0x0101010101010101

fileH = 0x8080808080808080

fileAB = fileA .|. (fileA `shiftL` 1)

fileGH = fileH .|. (fileH `shiftR` 1)

-- on the fly computed movement masks for pieces with fixed patterns
-- #################################################################
-- Knight
-- movement pattern
knightMasks :: [Bitboard]
knightMasks = [knightMoves (1 `shiftL` square) | square <- [0 .. 63]]

knightMoves :: Bitboard -> Bitboard
knightMoves knightBitboard =
  let move1 = (knightBitboard `shiftL` 15) .&. complement fileH
      move2 = (knightBitboard `shiftL` 17) .&. complement fileA
      move3 = (knightBitboard `shiftL` 10) .&. complement fileAB
      move4 = (knightBitboard `shiftL` 6) .&. complement fileGH
      move5 = (knightBitboard `shiftR` 15) .&. complement fileA
      move6 = (knightBitboard `shiftR` 17) .&. complement fileH
      move7 = (knightBitboard `shiftR` 10) .&. complement fileGH
      move8 = (knightBitboard `shiftR` 6) .&. complement fileAB
   in move1 .|. move2 .|. move3 .|. move4 .|. move5 .|. move6 .|. move7 .|. move8

generateKnightMoves :: Int -> Bitboard -> Bitboard -> Bitboard
generateKnightMoves square occupancy _ =
  let attacks = knightMasks !! square
   in attacks .&. complement occupancy

generateAllKnightMoves :: Bitboard -> Bitboard -> Bitboard -> [Move]
generateAllKnightMoves knightBoard occupancy captureable =
  [ Move
    square
    targetSquare
    ""
    (if targetBit .&. captureable /= 0
       then 1
       else 0)
    0
    0
  | square <- bitboardSquares knightBoard
  , targetSquare <-
      bitboardSquares (generateKnightMoves square occupancy captureable)
  , let targetBit = 1 `shiftL` targetSquare
  ]

-- precomputed move masks for all the sliding pieces
-- #################################################################
-- Rook 
-- start at the square then shift by 8 for vertical and 1 for horizontal 
generateRookMoves :: Int -> Bitboard -> Bitboard -> Bitboard
generateRookMoves square occupancy captureable =
  let position = 1 `shiftL` (square)
      up = generateLine position occupancy captureable 8 8 -- by 8 because we shift straight up so need to skip to next row 
      down = generateLine position occupancy captureable (-8) 8
      right = generateLine position occupancy captureable 1 8
      left = generateLine position occupancy captureable (-1) 8
   in up .|. down .|. right .|. left

-- get the occupancy 
generateAllRookMoves :: Bitboard -> Bitboard -> Bitboard -> [Move]
generateAllRookMoves rookBoard captureable occupancy -- after hitting a capturable block all cells behind it for sliding pieces
 =
  [ Move
    square
    targetSquare
    ""
    (if targetBit .&. captureable /= 0
       then 1
       else 0)
    0
    0
  | square <- bitboardSquares rookBoard
  , targetSquare <-
      bitboardSquares (generateRookMoves square occupancy captureable)
  , let targetBit = 1 `shiftL` targetSquare
  ]

-- bishop - same as rook but shift by 15 squares
-- #################################################################
generateBishopMoves :: Int -> Bitboard -> Bitboard -> Bitboard
generateBishopMoves square occupancy captureable =
  let position = 1 `shiftL` (square)
      upLeft = generateLine position occupancy captureable 9 8 -- by 8 because we shift straight up so need to skip to next row 
      upRight = generateLine position occupancy captureable 7 8
      downRight = generateLine position occupancy captureable (-9) 8
      downLeft = generateLine position occupancy captureable (-7) 8
   in upLeft .|. upRight .|. downRight .|. downLeft

-- get the occupancy 
generateAllBishopMoves :: Bitboard -> Bitboard -> Bitboard -> [Move]
generateAllBishopMoves bishopBoard captureable occupancy -- after hitting a capturable block all cells behind it for sliding pieces
 =
  [ Move
    square
    targetSquare
    ""
    (if targetBit .&. captureable /= 0
       then 1
       else 0)
    0
    0
  | square <- bitboardSquares bishopBoard
  , targetSquare <-
      bitboardSquares (generateBishopMoves square occupancy captureable)
  , let targetBit = 1 `shiftL` targetSquare
  ]

-- Queen
-- #################################################################
generateQueenMoves :: Int -> Bitboard -> Bitboard -> Bitboard
generateQueenMoves square occupancy captureable =
  let position = 1 `shiftL` square
      -- bishop moves
      upLeft = generateLine position occupancy captureable 9 8
      upRight = generateLine position occupancy captureable 7 8
      downRight = generateLine position occupancy captureable (-9) 8
      downLeft = generateLine position occupancy captureable (-7) 8
      -- rook moves 
      up = generateLine position occupancy captureable 8 8 -- by 8 because we shift straight up so need to skip to next row 
      down = generateLine position occupancy captureable (-8) 8
      right = generateLine position occupancy captureable 1 8
      left = generateLine position occupancy captureable (-1) 8
   in upLeft
        .|. upRight
        .|. downRight
        .|. downLeft
        .|. up
        .|. down
        .|. right
        .|. left

generateAllQueenMoves :: Bitboard -> Bitboard -> Bitboard -> [Move]
generateAllQueenMoves queenBoard occupancy captureable =
  [ Move
    square
    targetSquare
    ""
    (if targetBit .&. captureable /= 0
       then 1
       else 0)
    0
    0
  | square <- bitboardSquares queenBoard
  , targetSquare <-
      bitboardSquares (generateQueenMoves square occupancy captureable)
  , let targetBit = 1 `shiftL` targetSquare
  ]

generateAllQueenMovesT :: Bitboard -> Bitboard -> Bitboard -> Bitboard
generateAllQueenMovesT queenBoard occupancy captureable =
  foldr (.|.) 0 [generateQueenMoves square occupancy captureable | square <- bitboardSquares queenBoard]


-- King
-- #################################################################
generateKingMoves :: Int -> Bitboard -> Bitboard -> Bitboard
generateKingMoves square occupancy captureable =
  let position = 1 `shiftL` square
      -- bishop moves
      upLeft = generateLine position occupancy captureable 9 1
      upRight = generateLine position occupancy captureable 7 1
      downRight = generateLine position occupancy captureable (-9) 1
      downLeft = generateLine position occupancy captureable (-7) 1
      -- rook moves 
      up = generateLine position occupancy captureable 8 1
      down = generateLine position occupancy captureable (-8) 1
      right = generateLine position occupancy captureable 1 1
      left = generateLine position occupancy captureable (-1) 1
   in upLeft
        .|. upRight
        .|. downRight
        .|. downLeft
        .|. up
        .|. down
        .|. right
        .|. left

generateAllKingMoves :: Bitboard -> Bitboard -> Bitboard -> [Move]
generateAllKingMoves kingBoard occupancy captureable =
  [ Move
    square
    targetSquare
    ""
    (if targetBit .&. captureable /= 0
       then 1
       else 0)
    0
    0
  | square <- bitboardSquares kingBoard
  , targetSquare <-
      bitboardSquares (generateKingMoves square occupancy captureable)
  , let targetBit = 1 `shiftL` targetSquare
  ]

-- pawn - capture logic is handled in here
-- #################################################################
generatePawnMovesW :: Int -> Bitboard -> Bitboard -> Bitboard
generatePawnMovesW square occupancy captureable =
  let singleStepTarget = square + 8
      canMoveOneStep =
        singleStepTarget < 64
          && ((occupancy `shiftR` singleStepTarget) .&. 1) == 0
          && ((captureable `shiftR` singleStepTarget) .&. 1) == 0
      oneStepMask =
        if canMoveOneStep
          then (1 `shiftL` singleStepTarget)
          else 0
      doubleStepTarget = square + 16
      canMoveTwoSteps =
        canMoveOneStep
          && (square `div` 8 == 1)
          && doubleStepTarget < 64
          && ((occupancy `shiftR` doubleStepTarget) .&. 1) == 0
      twoStepMask =
        if canMoveTwoSteps
          then (1 `shiftL` doubleStepTarget)
          else 0
      file = square `mod` 8
      leftCaptureTarget = square + 7
      canCaptureLeft =
        file /= 0
          && leftCaptureTarget < 64
          && ((captureable `shiftR` leftCaptureTarget) .&. 1) == 1
      leftCaptureMask =
        if canCaptureLeft
          then (1 `shiftL` leftCaptureTarget)
          else 0
      rightCaptureTarget = square + 9
      canCaptureRight =
        file /= 7
          && rightCaptureTarget < 64
          && ((captureable `shiftR` rightCaptureTarget) .&. 1) == 1
      rightCaptureMask =
        if canCaptureRight
          then (1 `shiftL` rightCaptureTarget)
          else 0
   in oneStepMask .|. twoStepMask .|. leftCaptureMask .|. rightCaptureMask

generateAllPawnMovesW :: Bitboard -> Bitboard -> Bitboard -> [Move]
generateAllPawnMovesW pawnBoard occupancy captureable =
  [ Move
    square
    targetSquare
    ""
    (if targetBit .&. captureable /= 0
       then 1
       else 0)
    0
    0
  | square <- bitboardSquares pawnBoard
  , targetSquare <-
      bitboardSquares (generatePawnMovesW square occupancy captureable)
  , let targetBit = 1 `shiftL` targetSquare
  ]

-- #################################################################
generatePawnMovesB :: Int -> Bitboard -> Bitboard -> Bitboard
generatePawnMovesB square occupancy captureable =
  let file = square `mod` 8
      singleStepTarget = square - 8
      canMoveOneStep =
        singleStepTarget >= 0
          && ((occupancy `shiftR` singleStepTarget) .&. 1) == 0
      oneStepMask =
        if canMoveOneStep
          then (1 `shiftL` singleStepTarget)
          else 0
      doubleStepTarget = square - 16
      canMoveTwoSteps =
        canMoveOneStep
          && (square `div` 8 == 6)
          && doubleStepTarget >= 0
          && ((occupancy `shiftR` doubleStepTarget) .&. 1) == 0
      twoStepMask =
        if canMoveTwoSteps
          then (1 `shiftL` doubleStepTarget)
          else 0
      leftCaptureTarget = square - 9
      canCaptureLeft =
        file /= 0
          && leftCaptureTarget >= 0
          && ((captureable `shiftR` leftCaptureTarget) .&. 1) == 1
      leftCaptureMask =
        if canCaptureLeft
          then (1 `shiftL` leftCaptureTarget)
          else 0
      rightCaptureTarget = square - 7
      canCaptureRight =
        file /= 7
          && rightCaptureTarget >= 0
          && ((captureable `shiftR` rightCaptureTarget) .&. 1) == 1
      rightCaptureMask =
        if canCaptureRight
          then (1 `shiftL` rightCaptureTarget)
          else 0
   in oneStepMask .|. twoStepMask .|. leftCaptureMask .|. rightCaptureMask

generateAllPawnMovesB :: Bitboard -> Bitboard -> Bitboard -> [Move]
generateAllPawnMovesB pawnBoard occupancy captureable =
  [ Move
    square
    targetSquare
    ""
    (if targetBit .&. captureable /= 0
       then 1
       else 0)
    0
    0
  | square <- bitboardSquares pawnBoard
  , targetSquare <-
      bitboardSquares (generatePawnMovesB square occupancy captureable)
  , let targetBit = 1 `shiftL` targetSquare
  ]

-- Helper functions for sliding piece 
-- #################################################################
generateLine :: Bitboard -> Bitboard -> Bitboard -> Int -> Int -> Bitboard
generateLine position occupancy captureable step limit =
  let next =
        if step > 0
          then position `shiftL` step
          else position `shiftR` (-step) -- shiftL only trakes positive
   in if next == 0
           || invalidShift position step
           || canCapture position captureable
           || isBlocked position occupancy step
           || limit == 0
        then 0
        else next .|. generateLine next occupancy captureable step (limit - 1)

canCapture :: Bitboard -> Bitboard -> Bool
canCapture position captureable = (position .&. captureable) /= 0

isBlocked :: Bitboard -> Bitboard -> Int -> Bool
isBlocked position occupancy step =
  (step > 0 && (position `shiftL` step) .&. occupancy /= 0)
    || (step < 0 && (position `shiftR` (-step)) .&. occupancy /= 0)

-- check if we are overshooting the columns
invalidShift :: Bitboard -> Int -> Bool
invalidShift position step =
  (step > 0
     && (position `shiftL` step == 0
           || crossesLeftBoundary position step fileA
           || crossesDiagonalBoundary position step fileA fileH))
    || (step < 0
          && (position `shiftR` (-step) == 0
                || crossesRightBoundary position (-step) fileH
                || crossesDiagonalBoundary position step fileA fileH))

-- Helper functions for boundary checks
crossesLeftBoundary :: Bitboard -> Int -> Bitboard -> Bool
crossesLeftBoundary position step _ = step == 1 && (position .&. fileH /= 0)

crossesRightBoundary :: Bitboard -> Int -> Bitboard -> Bool
crossesRightBoundary position step _ =
  step == -1 && (position .&. fileA /= 0)

crossesDiagonalBoundary :: Bitboard -> Int -> Bitboard -> Bitboard -> Bool
crossesDiagonalBoundary position step fileAl fileHl
  | step == 9 || step == -7 = position .&. fileHl /= 0 -- Moving left and wrapping to file H
  | step == 7 || step == -9 = position .&. fileAl /= 0 -- Moving right and wrapping to file A
  | otherwise = False

-- demo :: IO ()
-- demo = do
--   let board = startpos
--   let kboard = 0x0000000000000042
--   let occupancy = 0x0000000000140008
--   let captureable = 0x0000000000030000
--   putStrLn
--     (unlines
--        (map
--           (\move -> showMove move 1)
--           (generateAllKnightMoves kboard occupancy captureable)))
--   putStrLn "All white board moves"
--   putStrLn (movesGrid (generateAllMovesWhite board))
--   putStrLn "All black board moves"
--   putStrLn (movesGrid (generateAllMovesBlack board))
--   putStrLn "Cap"
--   putStrLn (showGrid captureable)
--   putStrLn "moves for sq 9 pawn"
--   putStrLn (showGrid (generatePawnMovesW 9 occupancy captureable)) -- illegal pawn captures 
--   -- -- sample boards
--   let rookBoard = 0x000000000000081 -- a1 and h1
--   let bishopBoard = 0x000000000000024 -- a3, a5
--   let knightBoard = 0x0000000000000042 -- b1 and g1
--   let occupancy = 0x0000000000140008
--   let captureable = 0x0000100180001000
--   putStrLn "occupancy (stop before):"
--   putStrLn (showGrid (occupancy))
--   putStrLn "captureable (stop on):"
--   putStrLn (showGrid (captureable))
--   putStrLn "Queen moves from a1, h1"
--   putStrLn (showGrid (generateAllQueenMovesT rookBoard occupancy captureable))
--   -- putStrLn "Bishop Moves from c1 and f1:"
--   -- -- putStrLn (showGrid (generateAllBishopMoves bishopBoard occupancy captureable))
--   -- putStrLn "Knight Moves from b1 and g1:"
--   -- -- putStrLn
--   -- --   (unlines
--   -- --      (map showMove (generateAllKnightMoves knightBoard occupancy captureable) 1))
--   -- -- putStrLn (showBin (generateRookMoves 45 occupancy captureable))
--   -- -- putStrLn (showGrid (generateAllRookMoves rookBoard occupancy occupancy))
--   -- -- putStrLn (showGrid ((rookMasks occupancy captureable) !! 1)) -- attacks in generate rookmoves
--   -- -- putStrLn "Knight Grid:"
--   -- -- putStrLn (showGrid knightBoard)
--   -- -- putStrLn "occupancy:"
--   -- putStrLn (showGrid occupancy)
--   -- putStrLn "Knight Moves:"
--   -- putStrLn (showGrid (generateAllKnightMoves knightBoard occupancy))
