module MainSpec (tests) where

import Test.HUnit
import qualified Data.List as L
import qualified Data.Vector as V
import Lib
  ( ExpressionTree(..)
  , Clause(..)
  , Literal(..)
  , dpll
  )

-- Helper function: de-clutter solver evaluation
evaluateSeqDpllSolver :: [String] -> Int -> IO Bool
evaluateSeqDpllSolver clauses numVars = do
  let cnf = listToTree clauses
      initialAssignment = V.replicate numVars Nothing
  dpll cnf initialAssignment

-- Helper function: de-clutter solver evaluation
listToTree :: [String] -> ExpressionTree
listToTree x = Expr $ map (constructClause) x
  where
    constructClause  y = let w = L.words y
                         in Clause $ map (constructLiteral) w
    constructLiteral z =
                         if (head z) == '-'
                         then LiteralNeg ((read (tail z) :: Int) - 1)
                         else LiteralPos ((read z :: Int) - 1)

-- Unit tests for dpll
tests :: Test
tests = TestList
  [ sat1
  , sat2
  , sat3
  , sat4
  , sat5
  , unsat1
  , unsat2
  , unsat3
  , unsat4
  , unsat5
  , trivialUnsat
  , emptyFormula
  ]

sat1 :: Test
sat1 = TestCase $ do
  -- Evaluates CNF: (x0 v x1 v x2) ∧ (¬x0 v x1) ∧ (¬x1 v x2)
  result <- evaluateSeqDpllSolver ["1 2 3","-1 2","-2 3"] 3
  assertBool "Expected SAT" result

sat2 :: Test
sat2 = TestCase $ do
  result <- evaluateSeqDpllSolver ["-1 -2","-2 -3","-1 -3"] 3
  assertBool "Expected SAT" result

sat3 :: Test
sat3 = TestCase $ do
  result <- evaluateSeqDpllSolver ["1 2", "1 -3", "-1 3"] 3
  assertBool "Expected SAT" result

sat4 :: Test
sat4 = TestCase $ do
  result <- evaluateSeqDpllSolver ["1 2 3", "1 -2", "-1 4", "2 4", "3 -4"] 4
  assertBool "Expected SAT" result

sat5 :: Test
sat5 = TestCase $ do
  result <- evaluateSeqDpllSolver ["1 2 3", "-1 4", "-2 -4", "-3", "2 -3"] 4
  assertBool "Expected SAT" result

unsat1 :: Test
unsat1 = TestCase $ do
  result <- evaluateSeqDpllSolver ["3", "-1 2", "-2", "-3 1"] 3
  assertBool "Expected UNSAT" (not result)

unsat2 :: Test
unsat2 = TestCase $ do
  result <- evaluateSeqDpllSolver ["-1 2 3", "1 3 4", "1 3 -4", "1 -3 4", "1 -3 -4", "-2 -3 4", "-1 2 -3", "-1 -2 3"] 4
  assertBool "Expected SAT" result

unsat3 :: Test
unsat3 = TestCase $ do
  result <- evaluateSeqDpllSolver ["1 2", "-1", "-2"] 3
  assertBool "Expected UNSAT" (not result)

unsat4 :: Test
unsat4 = TestCase $ do
  result <- evaluateSeqDpllSolver ["1 2", "-1 3", "-2 3", "4", "-4", "2 -3", "3 -2"] 4
  assertBool "Expected UNSAT" (not result)

unsat5 :: Test
unsat5 = TestCase $ do
  result <- evaluateSeqDpllSolver ["-1 -2 -3", "-1 -2 3", "-1 2 -3", "-1 2 3", "1 -2 -3", "1 -2 3", "1 2 -3", "1 2 3"] 3
  assertBool "Expected UNSAT" (not result)

trivialUnsat :: Test
trivialUnsat = TestCase $ do
  result <- evaluateSeqDpllSolver ["1", "-1"] 1
  assertBool "Expected UNSAT" (not result)

emptyFormula :: Test
emptyFormula = TestCase $ do
  result <- evaluateSeqDpllSolver [] 0
  assertBool "Expected SAT" result
