module Game2048 where

import Data.List (transpose, maximumBy)
import Data.Ord (comparing)
import System.Random (randomRIO)
import Data.Maybe (catMaybes)
import Control.Parallel.Strategies
import System.Environment (getArgs)
import Control.Monad (when)

type Board = [[Int]]

-- Weight matrix for the board
weights :: [[Int]]
weights = [[6, 5, 4, 3], [5, 4, 3, 2], [4, 3, 2, 1], [3, 2, 1, 0]]

-- Paths for monotonicity evaluation
paths :: [[(Int, Int)]]
paths =
    [ [(0, 0), (0, 1), (0, 2), (0, 3), (1, 3), (1, 2), (1, 1), (1, 0), (2, 0), (2, 1), (2, 2), (2, 3), (3, 3), (3, 2), (3, 1), (3, 0)]
    , [(0, 0), (1, 0), (2, 0), (3, 0), (3, 1), (2, 1), (1, 1), (0, 1), (0, 2), (1, 2), (2, 2), (3, 2), (3, 3), (2, 3), (1, 3), (0, 3)]
    ]

-- Initialize an empty board
initialBoard :: IO Board
initialBoard = addRandomTile =<< addRandomTile (replicate 4 (replicate 4 0))

-- Add a random tile (2 or 4) to the board
addRandomTile :: Board -> IO Board
addRandomTile board = do
    let emptyCells = [(i, j) | i <- [0..3], j <- [0..3], (board !! i) !! j == 0]
    if null emptyCells
      then return board
      else do
        (i, j) <- (emptyCells !!) <$> randomRIO (0, length emptyCells - 1)
        randomValue <- randomRIO (0 :: Int, 9)
        let tile = if randomValue < 9 then 2 else 4
        let (rowBefore, rowAfter) = splitAt i board
        return $ case rowAfter of
          (row:rest) ->
            let (cellBefore, cellAfter) = splitAt j row
            in case cellAfter of
                (_:restCells) -> rowBefore ++ [cellBefore ++ [tile] ++ restCells] ++ rest
                []            -> rowBefore ++ [row] ++ rest -- Handle empty case
          [] -> board -- Handle empty board case

-- Utility to update a specific position on the board
updateBoard :: Board -> Int -> Int -> Int -> Board
updateBoard board i j tile =
    let (rowBefore, row:rowAfter) = splitAt i board
        (cellBefore, _:cellAfter) = splitAt j row
    in rowBefore ++ [cellBefore ++ [tile] ++ cellAfter] ++ rowAfter

-- Check if the board is full
isFull :: Board -> Bool
isFull = all (all (/= 0))

-- Check if the target tile is present
hasReachedTarget :: Board -> Bool
hasReachedTarget board = any (>= 2048) (concat board)

-- Heuristic Functions
getMonotonicity :: Board -> Int
getMonotonicity board =
    let monoWeight = 100000.0 :: Double
        monoRatio = 0.5 :: Double
        evalPath path =
            let pathValues = map (\(r, c) -> fromIntegral (board !! r !! c)) path -- Convert to Double for calculation
                weightedScores = zipWith (\val weight -> val * weight) pathValues (iterate (* monoRatio) monoWeight)
            in sum weightedScores
    in round $ maximum (map evalPath paths) -- Convert the result back to Int

getSmoothness :: Board -> Int
getSmoothness board =
    let smoothnessRow r c
            | r < 3 = abs (board !! r !! c - board !! (r + 1) !! c)
            | otherwise = 0
        smoothnessCol r c
            | c < 3 = abs (board !! r !! c - board !! r !! (c + 1))
            | otherwise = 0
    in sum [smoothnessRow r c + smoothnessCol r c | r <- [0..3], c <- [0..3]]

getWeightedSum :: Board -> Int
getWeightedSum board =
    sum [board !! r !! c * weights !! r !! c | r <- [0..3], c <- [0..3]]

getMaxCorner :: Board -> Int
getMaxCorner board =
    let maxTile = maximum (concat board)
    in if board !! 0 !! 0 == maxTile then 1000 else -1000

heuristic :: Board -> Int
heuristic board =
    let monotonicity = getMonotonicity board
        smoothness = getSmoothness board
        weightedSum = getWeightedSum board
        maxCorner = getMaxCorner board
    in monotonicity - smoothness + weightedSum + maxCorner

-- Expectimax Algorithm
expectimax :: Board -> Int -> Bool -> Int
expectimax board depth isMaximizing
    | depth == 0 || isFull board = heuristic board
    | isMaximizing =
        let boards = getMoves board
            scores = parMap rpar (\b -> expectimax b (depth - 1) False) boards
        in maximum scores
    | otherwise = calculateChance board depth

-- Calculate Chance Node
calculateChance :: Board -> Int -> Int
calculateChance board depth =
    let emptyCells = getEmptyCells board
        outcomes = concatMap (\(i, j) -> [updateBoard board i j 2, updateBoard board i j 4]) emptyCells
        probabilities :: [Double]
        probabilities = [0.9, 0.1]
        scores = parMap rpar (\b -> expectimax b (depth - 1) True) outcomes
    in round $ sum $ zipWith (\s p -> fromIntegral s * p) scores probabilities


-- Generate All Possible Moves
getMoves :: Board -> [Board]
getMoves board = catMaybes [ let b = move board d in if b == board then Nothing else Just b | d <- directions ]

-- Get Empty Cells
getEmptyCells :: Board -> [(Int, Int)]
getEmptyCells board = [(i, j) | i <- [0..3], j <- [0..3], board !! i !! j == 0]

-- Directions and Moves
data Direction = DirUp | DirDown | DirLeft | DirRight deriving (Enum, Bounded)

directions :: [Direction]
directions = [DirUp, DirDown, DirLeft, DirRight]

move :: Board -> Direction -> Board
move board DirUp = transpose $ moveLeft $ transpose board
move board DirDown = transpose $ moveRight $ transpose board
move board DirLeft = moveLeft board
move board DirRight = moveRight board

moveLeft, moveRight :: Board -> Board
moveLeft = map collapseRow
moveRight = map (reverse . collapseRow . reverse)

collapseRow :: [Int] -> [Int]
collapseRow row =
    let nonZero = filter (/= 0) row
        merged = mergeTiles nonZero
    in merged ++ replicate (length row - length merged) 0

mergeTiles :: [Int] -> [Int]
mergeTiles (x:y:zs) | x == y    = (x + y) : mergeTiles zs
                    | otherwise = x : mergeTiles (y:zs)
mergeTiles xs = xs

-- Function to print a board
printBoard :: Board -> IO ()
printBoard board = mapM_ print board >> putStrLn ""

-- Run a single game and return True if it reaches 2048, and the move count
runGame :: Bool -> Int -> IO Board
runGame printSteps depth = do
    board <- initialBoard
    (finalBoard, moveCount) <- playGame printSteps board depth 0
    when printSteps $ do
        putStrLn $ "Final board after " ++ show moveCount ++ " moves:"
        printBoard finalBoard
    return finalBoard

-- Play the game and return the final board and move count
playGame :: Bool -> Board -> Int -> Int -> IO (Board, Int)
playGame printSteps board depth moveCount
    | hasReachedTarget board = do
        when printSteps $ putStrLn $ "Reached 2048 in " ++ show moveCount ++ " moves!"
        when printSteps $ printBoard board
        return (board, moveCount)
    | isFull board = do
        when printSteps $ putStrLn $ "Game over! No more moves available after " ++ show moveCount ++ " moves."
        when printSteps $ printBoard board
        return (board, moveCount)
    | otherwise = do
        when printSteps $ putStrLn $ "Move " ++ show (moveCount + 1) ++ ":"
        when printSteps $ printBoard board
        let bestMove = maximumBy (comparing (\b -> expectimax b depth False)) (getMoves board)
        updatedBoard <- addRandomTile bestMove
        playGame printSteps updatedBoard depth (moveCount + 1)

-- Run 10 games and check results
running10Games :: Int -> IO ()
running10Games depth = do
    results <- mapM (\_ -> runGame False depth) [1..10] -- False disables printing
    let successes = length $ filter hasReachedTarget results
    putStrLn $ "Out of 10 games, the program reached 2048 " ++ show successes ++ " times."

