/* * Device driver for the VGA video generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_ball.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree vga_ball.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "vga_ball.h"

#define DRIVER_NAME "vga_ball"

/* Device registers */
#define BG_ARRAY_DATA(x) (x)
#define BG_ARRAY_ADDRESS(x) ((x) + 4)
#define BG_TILE_MAP(x) ((x) + 8)
#define MARIO_COORDINATES(x) ((x) + 12)
#define SPRITE_STATE(x) ((x)+16)
#define BARREL0_COORDINATES(x) ((x)+20)
#define BARREL1_COORDINATES(x) ((x)+24)
#define BARREL2_COORDINATES(x) ((x)+28)
#define BARREL3_COORDINATES(x) ((x)+32)
#define BARREL4_COORDINATES(x) ((x)+36)
#define HEART_HAMMER_STATE(x) ((x)+40)


/*
 * Information about our device
 */

struct vga_ball_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
        vga_ball_tile_array_data_t tile_data;
        vga_ball_tile_array_address_t tile_address;
        vga_ball_tile_map_t tile_map;
	vga_ball_xy_t mario_coordinates;
	vga_ball_sprite_state_t sprite_state;
	vga_ball_barrel0_xy_t barrel0;
	vga_ball_barrel1_xy_t barrel1;
	vga_ball_barrel2_xy_t barrel2;
	vga_ball_barrel3_xy_t barrel3;
	vga_ball_barrel4_xy_t barrel4;
	vga_ball_hh_state_t hh_state;
} dev;

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */

static void load_barrel0_coordinates(vga_ball_barrel0_xy_t *bxy){
	unsigned int bxy_coor =0;
	//fprintf(stderr, "vga.c barrel 0 (%d,%d)\n", bxy->bx0, bxy->by0);
	bxy_coor = (((unsigned int) bxy->bx0)<<16)+((unsigned int) bxy->by0);
	iowrite32(bxy_coor, BARREL0_COORDINATES(dev.virtbase));
	dev.barrel0 = *bxy;
}
static void load_barrel1_coordinates(vga_ball_barrel1_xy_t *bxy){
	unsigned int bxy_coor =0;
	//fprintf(stderr, "vga.c barrel 1 (%d,%d)\n", bxy->bx1, bxy->by1);

	bxy_coor = (((unsigned int) bxy->bx1)<<16)+((unsigned int) bxy->by1);
	iowrite32(bxy_coor, BARREL1_COORDINATES(dev.virtbase));
	dev.barrel1 = *bxy;
}
static void load_barrel2_coordinates(vga_ball_barrel2_xy_t *bxy){
	unsigned int bxy_coor =0;
	//fprintf(stderr,"vga.c barrel 2 (%d,%d)\n", bxy->bx2, bxy->by2);
	bxy_coor = (((unsigned int) bxy->bx2)<<16)+((unsigned int) bxy->by2);
	iowrite32(bxy_coor, BARREL2_COORDINATES(dev.virtbase));
	dev.barrel2 = *bxy;
}

static void load_barrel3_coordinates(vga_ball_barrel3_xy_t *bxy){
	unsigned int bxy_coor =0;
	//fprintf(stderr,"vga.c barrel 2 (%d,%d)\n", bxy->bx2, bxy->by2);
	bxy_coor = (((unsigned int) bxy->bx3)<<16)+((unsigned int) bxy->by3);
	iowrite32(bxy_coor, BARREL3_COORDINATES(dev.virtbase));
	dev.barrel3 = *bxy;
}

static void load_barrel4_coordinates(vga_ball_barrel4_xy_t *bxy){
	unsigned int bxy_coor =0;
	//fprintf(stderr,"vga.c barrel 2 (%d,%d)\n", bxy->bx2, bxy->by2);
	bxy_coor = (((unsigned int) bxy->bx4)<<16)+((unsigned int) bxy->by4);
	iowrite32(bxy_coor, BARREL4_COORDINATES(dev.virtbase));
	dev.barrel4 = *bxy;
}

static void load_sprite_state(vga_ball_sprite_state_t *s){
	iowrite32(s->state, SPRITE_STATE(dev.virtbase));
	dev.sprite_state = *s;
}

static void load_hh_state(vga_ball_hh_state_t *h){
	iowrite32(h->hhstate, HEART_HAMMER_STATE(dev.virtbase));
	dev.hh_state = *h;
}

static void load_mario_coordinates(vga_ball_xy_t *xy){
	unsigned int xy_coor =0;
	xy_coor = (((unsigned int) xy->x_coor)<<16)+((unsigned int) xy->y_coor);
	iowrite32(xy_coor, MARIO_COORDINATES(dev.virtbase));
	dev.mario_coordinates = *xy;
}

static void load_tile_array(vga_ball_tile_array_data_t *data, vga_ball_tile_array_address_t *address)
{
	unsigned int row = 0;
	row = (((unsigned int) data->tile_row0) << 24) + (((unsigned int) data->tile_row1) << 16) + (((unsigned int) data->tile_row2) << 8) + ((unsigned int) data->tile_row3);
	//fprintf(stderr, "ROW: %x, ADDRESS: %x\n", row, address->address);
	iowrite32(row, BG_ARRAY_DATA(dev.virtbase) );
	iowrite32(address->address, BG_ARRAY_ADDRESS(dev.virtbase) );
	dev.tile_data = *data;
	dev.tile_address = *address;
}

static void load_tile_map(vga_ball_tile_map_t *map)
{
	unsigned int map_int = 0;
	map_int = (((unsigned int) map->address) << 8) + ((unsigned int) map->data);
	//fprintf(stderr, "MAP: %x\n", map_int);
	iowrite32(map_int, BG_TILE_MAP(dev.virtbase) );
	dev.tile_map = *map;
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long vga_ball_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	vga_ball_arg_t vla;
        vga_ball_arg2_t vla2;
	vga_ball_arg3_t vla3;
	vga_ball_arg4_t vla4;
	vga_ball_arg5_t vla5;
	vga_ball_arg6_t vla6;
	vga_ball_arg7_t vla7;
	vga_ball_arg8_t vla8;
	vga_ball_arg9_t vla9;
	vga_ball_arg10_t vla10;

	switch (cmd) {
	case VGA_BALL_TILE_ARRAY:
		if (copy_from_user(&vla, (vga_ball_arg_t *) arg,
				   sizeof(vga_ball_arg_t)))
			return -EACCES;
		load_tile_array(&vla.data, &vla.address);
		break;

	case VGA_BALL_TILE_MAP:
		if (copy_from_user(&vla2, (vga_ball_arg2_t *) arg,
				   sizeof(vga_ball_arg2_t)))
			return -EACCES;
		load_tile_map(&vla2.tile_map);
		break;

	case VGA_BALL_MARIO_COORDINATES:
		if (copy_from_user(&vla3, (vga_ball_arg3_t *) arg,
				   sizeof(vga_ball_arg3_t *)))
			return -EACCES;
		load_mario_coordinates(&vla3.coor);
		break;

	case VGA_BALL_SPRITE_STATE:
		if (copy_from_user(&vla4, (vga_ball_arg4_t *) arg,
				   sizeof(vga_ball_arg4_t *)))
			return -EACCES;
		load_sprite_state(&vla4.m_state);
		break;

	case VGA_BALL_BARREL0_COORDINATES:
		if (copy_from_user(&vla5, (vga_ball_arg5_t *) arg,
				   sizeof(vga_ball_arg5_t *)))
			return -EACCES;
		load_barrel0_coordinates(&vla5.coor);
		break;

	case VGA_BALL_BARREL1_COORDINATES:
		if (copy_from_user(&vla6, (vga_ball_arg6_t *) arg,
				   sizeof(vga_ball_arg6_t *)))
			return -EACCES;
		load_barrel1_coordinates(&vla6.coor);
		break;

	case VGA_BALL_BARREL2_COORDINATES:
		if (copy_from_user(&vla7, (vga_ball_arg7_t *) arg,
				   sizeof(vga_ball_arg7_t *)))
			return -EACCES;
		load_barrel2_coordinates(&vla7.coor);
		break;
        case VGA_BALL_BARREL3_COORDINATES:
		if (copy_from_user(&vla8, (vga_ball_arg8_t *) arg,
				   sizeof(vga_ball_arg8_t *)))
			return -EACCES;
		load_barrel3_coordinates(&vla8.coor);
		break;

	case VGA_BALL_BARREL4_COORDINATES:
		if (copy_from_user(&vla9, (vga_ball_arg9_t *) arg,
				   sizeof(vga_ball_arg9_t *)))
			return -EACCES;
		load_barrel4_coordinates(&vla9.coor);
		break;
        case VGA_BALL_HH_STATE:
		if (copy_from_user(&vla10, (vga_ball_arg10_t *) arg,
				   sizeof(vga_ball_arg10_t *)))
			return -EACCES;
		load_hh_state(&vla10.m_state);
		break;
		

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations vga_ball_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = vga_ball_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice vga_ball_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &vga_ball_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init vga_ball_probe(struct platform_device *pdev)
{
        //vga_ball_color_t beige = { 0xf9, 0xe4, 0xb7 };
	int ret;

	/* Register ourselves as a misc device: creates /dev/vga_ball */
	ret = misc_register(&vga_ball_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	/* Set an initial color */
        //write_background(&beige);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&vga_ball_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int vga_ball_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&vga_ball_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id vga_ball_of_match[] = {
	{ .compatible = "csee4840,vga_ball-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, vga_ball_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver vga_ball_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(vga_ball_of_match),
	},
	.remove	= __exit_p(vga_ball_remove),
};

/* Called when the module is loaded: set things up */
static int __init vga_ball_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&vga_ball_driver, vga_ball_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit vga_ball_exit(void)
{
	platform_driver_unregister(&vga_ball_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(vga_ball_init);
module_exit(vga_ball_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("VGA ball driver");
