(* Ocamllex scanner for Cstar *)

{ open Parser }

let digit = ['0' - '9']
let digits = digit+
let ascii = ([' '-'!' '#'-'[' ']'-'~'])
let escape = '\\' ['\\' ''' '"' 'n' 'r' 't']
let string_lit = '"'((ascii|escape)* as lxm)'"'

rule token = parse
  [' ' '\t' '\r' '\n'] { token lexbuf } (* Whitespace *)
| "/*"     { comment lexbuf }           (* Comments *)
| '('      { LPAREN }
| ')'      { RPAREN }
| '{'      { LBRACE }
| '}'      { RBRACE }
| ';'      { SEMI }
| ','      { COMMA }
| '+'      { PLUS }
| '-'      { MINUS }
| '*'      { TIMES }
| '/'      { DIVIDE }
| '='      { ASSIGN }
| "=="     { EQ }
| "!="     { NEQ }
| '<'      { LT }
| "<="     { LEQ }
| ">"      { GT }
| ">="     { GEQ }
| "&"     { AND }
| "|"     { OR }
| "!"      { NOT }
| "if"     { IF }
| "else"   { ELSE }
| "for"    { FOR }
| "while"  { WHILE }
| "return" { RETURN }
| "int"    { INT }
| "bool"   { BOOL }
| "float"  { FLOAT }
| "void"   { VOID }
| "true"   { BLIT(true)  }
| "false"  { BLIT(false) }
| '.'      { DOT }
| '?'      { QMARK }
| "=>"     { ARROW }
| "||"     { OROR }
| "&&"     { ANDAND }
| '@'      { AT }
| '%'      { PERCENT }
| "<<"     { LSHIFT }
| ">>"     { RSHIFT }
| "let"    { LET }
| "continue" { CONTINUE }
| "break"  { BREAK } 
| "try"    { TRY }
| "match"  { MATCH }
| "defer"  { DEFER }
| "undefer" { UNDEFER }
| "in"     { IN }
| "mut"    { MUT }
| "use"    { USE }
| "fn"     { FN }
| "pub"    { PUB }
| ".."     { DOTDOT }
| "union"  { UNION }
| "enum"   { ENUM }
| "struct" { STRUCT }
| "impl"   { IMPL }
| "const"  { CONST }
| "string" { STRING }
| "trait"  { TRAIT }
| "//"     { scomment lexbuf } (* Single-line Comment *)

| digits as lxm { LITERAL(int_of_string lxm) }
| digits '.'  digit* ( ['e' 'E'] ['+' '-']? digits )? as lxm { FLIT(lxm) }
| ['a'-'z' 'A'-'Z']['a'-'z' 'A'-'Z' '0'-'9' '_']*     as lxm { ID(lxm) }
| string_lit { STRING_LITERAL(lxm) }
| eof { EOF }
| _ as char { raise (Failure("illegal character " ^ Char.escaped char)) }

and comment = parse
  "*/" { token lexbuf }
| _    { comment lexbuf }

and scomment = parse
  "\n" { token lexbuf }
| _    { scomment lexbuf }
