open Ast
open Printf
open Util
open Values
open Random

exception Output_error of string
let output_error msg = raise (Output_error msg)

let default_velocity = 90

(* Write the head of each smurf file, returns the number of tracks *)
(* write_head : out_channel -> value -> int *)
let write_head oc value = 
    let header = "***** Generated by SMURF *****" in 
    let number_of_track = (match value with
          VList(lst) -> (
              try 
                  match List.hd lst with 
                    VSystem(_) |  VList(VChord(_)::_) |
                    VList(VList(VNote(_,_,_)::_)::_)-> List.length lst (* list of system *)
                  | _ -> 1
              with 
                Failure _ -> 0 (* Empty list *)
              )
        | _ -> 1) in 
    let resolution = 4 in
    fprintf oc "%s\n" header; 
    fprintf oc "number of trace: %d\n" number_of_track;
    fprintf oc "Time Resolution (pulses per quarter note),%d,\n" resolution;
    Random.init 0;
    for i=1 to number_of_track 
        do fprintf oc "track %d,%d," i 48 
        done;
    fprintf oc "\n";
    for i=1 to number_of_track 
        do fprintf oc "Tick, Note (0-127), Velocity (0-127), " 
        done;
    fprintf oc "\n"; 
    number_of_track

(*
(* get the number of ticks of a beat *)
(* VBeat -> Int *)
let ticks_of_beat = function
      VBeat(VInt(i1), -1) -> i1
     | VBeat(VInt(i1),i2) -> 
      (int_of_float 
          ((2.0 *. (16.0/.(float_of_int i1))) -. ((16.0/.float_of_int i1) /.
              ((match i2 with 
                 0 -> 1.0
               | 1 -> 2.0
               | 2 -> 4.0
               | 3 -> 8.0
               | 4 -> 16.0
               | _ -> output_error ("Error in ticks_of_beat: Not valid numbers"))
                       ))))
    | _ -> output_error ("Error in ticks_of_beat: Not a beat")
*)

(* figure how many ticks are there in the output, so that an array with suitable size can be generated *)
(* value -> Int *)
let rec ticks_of_output value =
    match value with
      VNote(pc,reg,beat) -> 
      (match beat with
          | VBeat(-1) -> 0
          | VBeat(1) -> 2
          | VBeat(beat) -> beat
          | VInt(beat) -> beat
          | _ -> interp_error ("Invalid Beat value")
      )
    | VChord(nlst) -> List.fold_left (fun acc ch -> acc + ticks_of_output ch) 0 nlst
    | VSystem(slst) | VList((VList(VList(VNote(_,_,_) :: _) :: _) :: _) as slst) 
    | VList((VList(VChord(_) :: _) :: _) as slst)
    | VList((VSystem(_) :: _) as slst) | VList((VList(VNote(_,_,_) :: _) :: _)  as slst) -> List.fold_left (fun acc ch -> acc + ticks_of_output ch) 0 slst
    | VList((VNote(_,_,_) :: _) as nlst) | VList((VChord(_) ::_) as nlst) -> List.fold_left (fun acc ch -> acc + ticks_of_output ch) 0 nlst
    | _ ->  output_error ("Error in ticks_of_output")


(* Write a note into an array, return the next postion to be writen, and the next tick to begin with *)
(* Value -> Array -> Int -> Int -> Int -> (Int, Int, Int) *)
let rec write_to_array value arr ix iy tic = 
    (match value with
    | VNote(VInt(pc),VInt(reg),VBeat(beat)) ->if (pc = -1 && reg = -1 && beat = -1) then ix,iy,tic else
        let note = (match pc with
              -1 -> -1
            | _ -> pc+12*(reg+3)) in (
            arr.(ix).(iy) <- tic;                     (* tick *)
            arr.(ix).(iy+1) <- note;                  (* note *)
            arr.(ix).(iy+2) <- default_velocity;      (* velocity *)
            arr.(ix+1).(iy) <- tic+beat;
            arr.(ix+1).(iy+1) <- note;
            arr.(ix+1).(iy+2) <- 0;
            if beat = 1 then ix+beat+1,iy,tic+beat else ix+beat,iy,tic+beat)
    (* All notes in a chord should fills same set of ticks *)
    | VChord((VNote(_,_,VBeat(ticks))::xs) as nlst) | VList((VNote(_,_,VBeat(ticks))::xs) as nlst) -> 
        let actlst = if ticks = -1 then List.tl nlst else nlst in
        (let resx, resy, restic =
         (List.fold_left (fun (x,y,ntic) note ->
            let (nx,ny,ntic) = write_to_array note arr x y ntic
            in (nx,ny,tic)) (ix,iy,tic) actlst) in resx, resy, (if ticks = -1 then restic else restic+ticks))
    | VSystem(clst) | VList((VChord(_) :: _) as clst) | VList((VList(VNote(_,_,_) :: _) :: _) as clst)->
        (let resx, resy, resz = 
        List.fold_left (fun (x,y,ntic) chord -> 
            let (nx,ny,ntic) = write_to_array chord arr x y ntic
            in (nx,ny,ntic)) (ix,iy,tic) clst in (0,resy+3,0))
    | VList((x::xs) as slst) -> (match x with
          VSystem(_) | VChord(_) | VNote(_,_,_) | VList(VChord(_)::_) | VList(VList(VNote(_,_,_)::_)::_) ->
                    List.fold_left (fun (x,y,ntic) sys ->
                  let (nx,ny,ntic) = write_to_array sys arr x y ntic
                  in (nx,ny,ntic)) (ix,iy,tic) slst
        | _ ->  output_error ("Error in write_to_array: Expression bound to MAIN must "
                            ^ "be the empty list, a note, or a list of systems, chords, or notes"))
    | _ -> output_error ("Error in write_to_array: Input is not a valid value")
    )


(* Write a Chord or a System or a list of Systems to file with smurf specified format *)
(* write_to_file : string -> value -> unit *)
let write_to_file filename value = 
    let oc = open_out filename in 
    let number_of_track = write_head oc value in 
    match number_of_track with
          0 -> close_out oc; print_string ("===== main = [] Program Exits Normally =====\n"); exit 0
        | _ -> (
    let dimx = ticks_of_output value in 
    let dimy = number_of_track * 3 in
    let resArr = (Array.make_matrix (dimx+1) (dimy) (-1)) in 
    let _ = (write_to_array value resArr 0 0 0) in 
    for i=0 to dimx-1 do
        for j=0 to (number_of_track -1) do
            if resArr.(i).(3*j+1) <> (-1) then
                ignore(fprintf oc "%d,%d,%d," resArr.(i).(3*j) resArr.(i).(3*j+1) resArr.(i).(3*j+2))
            else
                ignore(fprintf oc ",,,")
        done;
        ignore(fprintf oc "\n")
    done;
    close_out oc
    )



