library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity ImageToVGA is
	port(
		signal clk : in std_logic;
		signal reset : in std_logic;
		signal FRAME_DONE : out std_logic;
		signal VALUE_AT_CUR : in std_logic_vector(1 downto 0);
		
		signal XPOS : out std_logic_vector(9 downto 0);
		signal YPOS : out std_logic_vector(9 downto 0);
		signal position : out std_logic_vector(18 downto 0);
		signal color_options : in std_logic_vector(2 downto 0);
		
		VGA_CLK,                                            
		VGA_HS,                                             
		VGA_VS,                                             
		VGA_BLANK,                                          
		VGA_SYNC : out std_logic;                           
		VGA_R,                                              
		VGA_G,                                              
		VGA_B : out unsigned(9 downto 0)                   
	);
end ImageToVGA;

architecture rtl of ImageToVGA is
  constant HTOTAL       : integer := 800;
  constant HSYNC        : integer := 96;
  constant HBACK_PORCH  : integer := 48;
  constant HACTIVE      : integer := 640;
  constant HFRONT_PORCH : integer := 16;
  
  constant HALFHLENGTH : integer := 10;
  constant HALFVLENGTH : integer := 10;
  
  constant VTOTAL       : integer := 525;
  constant VSYNC        : integer := 2;
  constant VBACK_PORCH  : integer := 33;
  constant VACTIVE      : integer := 480;
  constant VFRONT_PORCH : integer := 10;

  signal Hcount : unsigned(9 downto 0); 
  signal Vcount : unsigned(9 downto 0); 
  
  signal count : integer := 0;
  
  signal EndOfLine, EndOfField : std_logic;
  signal vga_hblank, vga_hsync,
    vga_vblank, vga_vsync : std_logic;
	 
  signal gradient : integer := 0;

begin
	HCounter : process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Hcount <= (others => '0');
      elsif EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
      end if;
		
		position <= std_logic_vector(to_unsigned(count, 19));
    end if;
  end process HCounter;
  
  
  EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';
  
  VCounter: process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Vcount <= (others => '0');
      elsif EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
			 gradient <= 0;
        else
          Vcount <= Vcount + 1;
			 gradient <= gradient + 1;
        end if;
      end if;
    end if;
  end process VCounter;
  
  
  EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';
  
  HSyncGen : process (clk)
  begin
    if rising_edge(clk) then     
      if reset = '1' or EndOfLine = '1' then
        vga_hsync <= '1';
      elsif Hcount = HSYNC - 1 then
        vga_hsync <= '0';
      end if;
    end if;
  end process HSyncGen;
  
  HBlankGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_hblank <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH then
        vga_hblank <= '0';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE then
        vga_hblank <= '1';
      end if;      
    end if;
  end process HBlankGen;

  VSyncGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_vsync <= '1';
      elsif EndOfLine ='1' then
        if EndOfField = '1' then
          vga_vsync <= '1';
        elsif Vcount = VSYNC - 1 then
          vga_vsync <= '0';
        end if;
      end if;      
    end if;
  end process VSyncGen;

  VBlankGen : process (clk)
  begin
    if rising_edge(clk) then    
      if reset = '1' then
        vga_vblank <= '1';
      elsif EndOfLine = '1' then
        if Vcount = VSYNC + VBACK_PORCH - 1 then
          vga_vblank <= '0';
        elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          vga_vblank <= '1';
        end if;
      end if;
    end if;
  end process VBlankGen;
  
  RasterGen : process(clk)
  begin
	if rising_edge(clk) then
		if((Hcount >= HSYNC + HBACK_PORCH) and (Hcount < HSYNC + HBACK_PORCH + HACTIVE)) then
			XPOS <= std_logic_vector(Hcount - HSYNC - HBACK_PORCH);
		end if;
		if((Vcount >= VSYNC + VBACK_PORCH - 1) and (Vcount < VSYNC + VBACK_PORCH + VACTIVE - 1) and (Hcount >= HSYNC + HBACK_PORCH) and (Hcount < HSYNC + HBACK_PORCH + HACTIVE)) then
			YPOS <= std_logic_vector(Vcount - VSYNC - VBACK_PORCH + 1);
			if((Vcount - VSYNC - VBACK_PORCH + 1) mod 2 = 0 and (Hcount - HSYNC - HBACK_PORCH) mod 2 = 0) then
				count <= count + 1;
			end if;
		end if;
		if EndOfField = '1' then
			count <= 0;
		end if;
		FRAME_DONE <= EndOfField;
	end if;
  end process RasterGen;
  
  
  
  VideoOut: process (clk, reset)
  begin
    if reset = '1' then
      VGA_R <= "0000000000";
      VGA_G <= "0000000000";
      VGA_B <= "0000000000";
    elsif clk'event and clk = '1' then
		if color_options = "000" then
			if VALUE_AT_CUR = "00" then
			  VGA_R <= to_unsigned(gradient, 10);
			  VGA_G <= "0010000000";
			  VGA_B <= "0000000000"; 
			elsif VALUE_AT_CUR = "10" then
			  VGA_R <= "1111111111";
			  VGA_G <= "0000000000";
			  VGA_B <= "1111111111";
			elsif VALUE_AT_CUR = "01" then
			  VGA_R <= "0000000000";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
			elsif VALUE_AT_CUR = "11" then
			  VGA_R <= "1111111111";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
			elsif vga_hblank = '0' and vga_vblank ='0' then
			  VGA_R <= "0000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			else
			  VGA_R <= "0000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			end if;
		elsif color_options = "001" then
			if VALUE_AT_CUR = "00" then
			  VGA_R <= to_unsigned(gradient, 10);
			  VGA_G <= "0010000000";
			  VGA_B <= "0000000000"; 
			elsif VALUE_AT_CUR = "10" then
			  VGA_R <= "0000000000";
			  VGA_G <= "1111111111";
			  VGA_B <= to_unsigned(gradient + 200, 10);
			elsif VALUE_AT_CUR = "01" then
			  VGA_R <= "0000000000";
			  VGA_G <= "1111111111";
			  VGA_B <= to_unsigned(gradient + 200, 10);
			elsif VALUE_AT_CUR = "11" then
			  VGA_R <= "1111111111";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
			elsif vga_hblank = '0' and vga_vblank ='0' then
			  VGA_R <= "0000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			else
			  VGA_R <= "0000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			end if;
		elsif color_options = "010" then
			if VALUE_AT_CUR = "00" then
			  VGA_R <= to_unsigned(gradient, 10);
			  VGA_G <= "0010000000";
			  VGA_B <= "0000000000"; 
			elsif VALUE_AT_CUR = "10" then
			  VGA_R <= "0000000000";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
			elsif VALUE_AT_CUR = "01" then
			  VGA_R <= "0000000000";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
			elsif VALUE_AT_CUR = "11" then
			  VGA_R <= "0000000000";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
			elsif vga_hblank = '0' and vga_vblank ='0' then
			  VGA_R <= "0000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			else
			  VGA_R <= "0000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			end if;
		elsif color_options = "100" then
			if VALUE_AT_CUR = "00" then
			  VGA_R <= "0000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000"; 
			elsif VALUE_AT_CUR = "10" then
			  VGA_R <= "1111111111";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
			elsif VALUE_AT_CUR = "01" then
			  VGA_R <= "1111111111";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
			elsif VALUE_AT_CUR = "11" then
			  VGA_R <= "1111111111";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
			elsif vga_hblank = '0' and vga_vblank ='0' then
			  VGA_R <= "0000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			else
			  VGA_R <= "0000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			end if;
		else
			if VALUE_AT_CUR = "00" then
			  VGA_R <= to_unsigned(gradient, 10);
			  VGA_G <= "0010000000";
			  VGA_B <= "0000000000"; 
			elsif VALUE_AT_CUR = "10" then
			  VGA_R <= "0000000000";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
			elsif VALUE_AT_CUR = "01" then
			  VGA_R <= "0000000000";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
			elsif VALUE_AT_CUR = "11" then
			  VGA_R <= "1111111111";
			  VGA_G <= "1111111111";
			  VGA_B <= "1111111111";
			elsif vga_hblank = '0' and vga_vblank ='0' then
			  VGA_R <= "0000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			else
			  VGA_R <= "0000000000";
			  VGA_G <= "0000000000";
			  VGA_B <= "0000000000";
			end if;
		end if;
    end if;
  end process VideoOut;

  VGA_CLK <= clk;
  VGA_HS <= not vga_hsync;
  VGA_VS <= not vga_vsync;
  VGA_SYNC <= '0';
  VGA_BLANK <= not (vga_hsync or vga_vsync);
  
end rtl;
