library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity FractalLandscapeGeneratorTop is
	port(
		signal CLOCK_50 : in std_logic;
		signal LEDR : out std_logic_vector(17 downto 0); 
		signal SW : in std_logic_vector(17 downto 0);
		signal KEY : in std_logic_vector(3 downto 0);
		
		SRAM_DQ : inout std_logic_vector(15 downto 0);
		SRAM_ADDR : out std_logic_vector(17 downto 0);
		SRAM_UB_N,            
		SRAM_LB_N,            
		SRAM_WE_N,            
		SRAM_CE_N,            
		SRAM_OE_N : out std_logic;

		
		VGA_CLK,                                            
		VGA_HS,                                             
		VGA_VS,                                             
		VGA_BLANK,                                          
		VGA_SYNC : out std_logic;                           
		VGA_R,                                              
		VGA_G,                                              
		VGA_B : out unsigned(9 downto 0)                   
	);
end FractalLandscapeGeneratorTop;

architecture rtl of FractalLandscapeGeneratorTop is

	signal clk25 : std_logic := '0';
  
	type states is (GEN_STATE, DRAWFACES_STATE, WAIT_STATE, CLEAR_STATE);
	signal state : states := DRAWFACES_STATE;
	
	signal drawfaces_start, drawfaces_done, frame_done : std_logic := '0';
	signal xpos, ypos : std_logic_vector(9 downto 0);
	signal position : std_logic_vector(18 downto 0);
	
	signal faces_rdaddress, faces_wraddress : STD_LOGIC_VECTOR (9 DOWNTO 0);
	signal faces_data : STD_LOGIC_VECTOR (35 DOWNTO 0);
	signal faces_wren	: STD_LOGIC := '0';
	signal faces_q	: STD_LOGIC_VECTOR (35 DOWNTO 0);
	signal backface_on : std_logic := '1';
	signal z_section : std_logic_vector(5 downto 0) := "111111";
	
	signal current_image_value, fbuffer1_data, fbuffer2_data, drawfaces_fdata	: STD_LOGIC_VECTOR (1 DOWNTO 0);
	signal fbuffer1_rdaddress, fbuffer2_rdaddress : STD_LOGIC_VECTOR (16 DOWNTO 0);
	signal fbuffer1_wraddress, fbuffer2_wraddress, drawfaces_faddress : STD_LOGIC_VECTOR (16 DOWNTO 0);
	signal fbuffer1_wren, fbuffer2_wren, drawfaces_fwren : STD_LOGIC  := '0';
	signal fbuffer1_q, fbuffer2_q : STD_LOGIC_VECTOR (1 DOWNTO 0);
	
	signal fbuffer_bool : std_logic := '0';
	
	signal counter : integer := 0;
	signal clear_counter : integer := 0;
	
	signal gen_start, gen_done, gen_wren : std_logic := '0';
	signal draw_rdaddress, gen_rdaddress : std_logic_vector(9 downto 0);
	signal in_key : std_logic_vector(3 downto 0) := "1111";
	signal in_sw : std_logic_vector(17 downto 0) := (others => '1');
	
	signal reset_n : std_logic := '1';
	signal next_hpos, next_vpos : std_logic_vector(15 downto 0);
	signal line_done, line_start, line_reset, line_plot : std_logic := '0';
	signal line_data : std_logic_vector(0 downto 0) := "1";
	signal x1,x2,x3,x4,y1,y2,y3,y4 : std_logic_vector(10 downto 0);
	signal rotcontrol_z : std_logic_vector(1 downto 0) := "00";
	signal debounce_counter : integer := 0;
	signal color_options : std_logic_vector(2 downto 0) := "000";
	
  
begin
	process(CLOCK_50)
	begin
		if rising_edge(CLOCK_50) then
			clk25 <= not clk25;
		end if;
		
		if rising_edge(clk25) then
			case state is
			when GEN_STATE =>
				LEDR <= (0 => '1', others => '0');
				reset_n <= '1';
				if gen_done = '1' then
					state <= DRAWFACES_STATE;
					gen_start <= '0';
				else
					state <= GEN_STATE;
					gen_start <= '1';
				end if;
			when DRAWFACES_STATE =>
				LEDR <= (1 => '1', others => '0');
				reset_n <= '1';
				if drawfaces_done = '1' then
					state <= WAIT_STATE;
					drawfaces_start <= '0';
				else
					state <= DRAWFACES_STATE;
					drawfaces_start <= '1';
				end if;
			when WAIT_STATE =>
				LEDR <= (2 => '1', others => '0');
				drawfaces_start <= '0';
				
				if counter = 1 then
					counter <= 0;
					clear_counter <= 0;
					state <= CLEAR_STATE;
					fbuffer_bool <= not fbuffer_bool;
				else
					if frame_done = '1' then
						counter <= counter + 1;
					end if;
					state <= WAIT_STATE;
				end if;
				
			when CLEAR_STATE =>
				LEDR <= (3 => '1', others => '0');
				
				if clear_counter < 76800 then
					clear_counter <= clear_counter + 1;
					state <= CLEAR_STATE;
				else
					if in_key(0) =  '0' then
						state <= GEN_STATE;
					else
						state <= DRAWFACES_STATE;
					end if;
				end if;
			end case;
			
			if in_key(0) /= KEY(0) then
				if debounce_counter >= 500 then
					in_key(0) <= KEY(0);
					debounce_counter <= 0;
				else
					debounce_counter <= debounce_counter + 1;
				end if;
			end if;
			if in_key(1) /= KEY(1) then
				if debounce_counter >= 500 then
					in_key(1) <= KEY(1);
					debounce_counter <= 0;
				else
					debounce_counter <= debounce_counter + 1;
				end if;
			end if;
			if in_key(2) /= KEY(2) then
				if debounce_counter >= 500 then
					in_key(2) <= KEY(2);
					debounce_counter <= 0;
				else
					debounce_counter <= debounce_counter + 1;
				end if;
			end if;
			if in_sw /= SW then
				if debounce_counter >= 500 then
					in_sw <= SW;
					debounce_counter <= 0;
				else
					debounce_counter <= debounce_counter + 1;
				end if;
			end if;
			color_options <= in_sw(7 downto 5);
			rotcontrol_z(1) <= in_key(2);
			rotcontrol_z(0) <= in_key(1);
		end if;
	end process;
	
	faces_wren <= gen_wren when state = GEN_STATE else '0';
	faces_rdaddress <= gen_rdaddress when state = GEN_STATE else draw_rdaddress;

	current_image_value <= fbuffer1_q when fbuffer_bool = '0' else fbuffer2_q;
	fbuffer1_rdaddress <= position(16 downto 0);
	fbuffer1_wraddress <= std_logic_vector(to_unsigned(clear_counter, 17)) when state = CLEAR_STATE else drawfaces_faddress;
	fbuffer1_data <= "00" when state = CLEAR_STATE else drawfaces_fdata;
	fbuffer1_wren <= '1' when state = CLEAR_STATE and fbuffer_bool = '1' else
						  drawfaces_fwren when fbuffer_bool = '1' else
						  '0';
	
	fbuffer2_rdaddress <= position(16 downto 0);
	fbuffer2_wraddress <= std_logic_vector(to_unsigned(clear_counter, 17)) when state = CLEAR_STATE else drawfaces_faddress;
	fbuffer2_data <= "00" when state = CLEAR_STATE else drawfaces_fdata;
	fbuffer2_wren <= '1' when state = CLEAR_STATE and fbuffer_bool = '0' else 
						  drawfaces_fwren when fbuffer_bool = '0' else
							'0';
	
	image_to_vga : entity work.ImageToVGA port map(
		reset => '0',
		clk => clk25,
		FRAME_DONE => frame_done,
		VALUE_AT_CUR => current_image_value,
		--VALUE_AT_CUR => '1',
		XPOS => xpos,
		YPOS => ypos,
		position => position,
		color_options => color_options,
		VGA_CLK => VGA_CLK,
		VGA_HS => VGA_HS,
		VGA_VS => VGA_VS,
		VGA_BLANK => VGA_BLANK,
		VGA_SYNC => VGA_SYNC,
		VGA_G => VGA_G,
		VGA_B => VGA_B,
		VGA_R => VGA_R
	);
	
	frame_buffer1 : entity work.framebuffer_ram port map(
		rdaddress	=> fbuffer1_rdaddress,
		wraddress	=> fbuffer1_wraddress,
		clock => clk25,
		data	=> fbuffer1_data,
		wren	=> fbuffer1_wren,
		q	=> fbuffer1_q
	
	);
	
	frame_buffer2 : entity work.framebuffer_ram port map(
		rdaddress	=> fbuffer2_rdaddress,
		wraddress	=> fbuffer2_wraddress,
		clock => clk25,
		data	=> fbuffer2_data,
		wren	=> fbuffer2_wren,
		q	=> fbuffer2_q
	
	);
	
	drawfaces : entity work.DrawFacesModule port map(
		clk => clk25,
		start => drawfaces_start,
		done => drawfaces_done,
		
		rdaddress => draw_rdaddress,
		q => faces_q,
		
		frame_address => drawfaces_faddress,
		frame_data => drawfaces_fdata,
		frame_wren => drawfaces_fwren,
		backface_on => backface_on,
		z_section => z_section,
		rotcontrol_z => rotcontrol_z
	);
	
	hmap_memory : entity work.HeightMapMemory port map(
		rdaddress => faces_rdaddress,
		wraddress => faces_wraddress,
		clock	=> clk25,
		data => faces_data,
		wren => faces_wren,
		q => faces_q
	);
	
	nios2 : entity work.nios_system port map(
	  clk_0 => clk25,
	  reset_n => reset_n,
	  
     data_from_the_de2_cpu_to_vga_0 => faces_data,
     gen_done_from_the_de2_cpu_to_vga_0 => gen_done,
	  gen_start_to_the_de2_cpu_to_vga_0 => gen_start,
	  hmap_address_from_the_de2_cpu_to_vga_0 => faces_wraddress,
	  wren_from_the_de2_cpu_to_vga_0 => gen_wren,



  -- the_sram
	  SRAM_ADDR_from_the_sram => SRAM_ADDR,
	  SRAM_CE_N_from_the_sram => SRAM_CE_N,
	  SRAM_DQ_to_and_from_the_sram => SRAM_DQ,
	  SRAM_LB_N_from_the_sram => SRAM_LB_N,
	  SRAM_OE_N_from_the_sram => SRAM_OE_N,
	  SRAM_UB_N_from_the_sram => SRAM_UB_N,
	  SRAM_WE_N_from_the_sram => SRAM_WE_N
);


end rtl;