open Sdl
open Audio
open Ast
open Bytecode

(* Stack layout just after "Ent":

              <-- SP
   Local n
   ...
   Local 0
   Saved FP   <-- FP
   Saved PC
   Arg 0
   ...
   Arg n *)

let execute_prog prog path =
  let stack = Array.make 131072 0
  and globals = Array.make prog.num_globals 0 
  and tr = 0.0174532925 (* to_radian conversion *)
  and rc = 0.5 (* rate coefficient *)
  and vc = 0.01 in (* volume coefficient *)
  let a, tune = load_wav path in
  let tlength = Bigarray.Array1.dim tune in
  globals.(1) <- 1; (* initialize rate to 1 *)
  globals.(4) <- 1; (* intialize volume to 1 *)
  globals.(7) <- 24; (* intialize Turtle to lower left coordinate (0,24) *)

  let tango dir dist =
    let position = Array.make 2 0.0 in (* local float position *)
    let rate = rc *. float_of_int globals.(1) in
    let radians = ((float_of_int globals.(0)) *. tr) in
    let dance = if globals.(5) > 0 then 1.0 else 0.0 in (* need pen down *)
    let steps = (* only dance to tune if playing  *)
      (if globals.(3) > 0 && globals.(5) > 0 then int_of_float 
	  (ceil (float_of_int dist /. rate)) else 1) in
    let stride = (if globals.(3) > 0 && globals.(5) > 0 then rate
    else (float_of_int dist)) in
    position.(0) <- float_of_int globals.(6) ;
    position.(1) <- float_of_int globals.(7) ;

    for i = 1 to steps do
      
      if (globals.(2) >= tlength) then globals.(2) <- 0 ;

      let x = 
      (-1.0 *. (sin radians)) *. dance *. (* dance modulation *)
      (float_of_int (tune.{globals.(2)} -
      (if globals.(2) > 0 then 
	tune.{globals.(2)-1} else 0))) *. (float_of_int globals.(4)) *. vc 
	+. (* + stride *)
	(dir *. stride *. (cos radians)) in

      let y = 
      (cos radians) *. dance *. (* dance modulation *)
      (float_of_int (tune.{globals.(2)} -
      (if globals.(2) > 0 then
	tune.{globals.(2)-1} else 0))) *. (float_of_int globals.(4)) *. vc  
	  +. (* + stride *)
	(dir *. stride *. (sin radians)) in

    print_endline (
      string_of_float x ^" "^
      string_of_float y ^
      if globals.(5) > 0 then " rlineto " else " rmoveto ") ; (* pd|pu *)
      globals.(2) <- globals.(2) + 1; (* update tune_index *)
      position.(0) <- position.(0) +. x; (* update current position *)
      position.(1) <- position.(1) +. y;

	done;
    globals.(6) <- int_of_float position.(0) ;
    globals.(7) <- int_of_float position.(1) ;
  in

  let rec exec fp sp pc = match prog.text.(pc) with
    Lit i  -> stack.(sp) <- i ; exec fp (sp+1) (pc+1)
  | Drp    -> exec fp (sp-1) (pc+1)
  | Bin op -> let op1 = stack.(sp-2) and op2 = stack.(sp-1) in     
      stack.(sp-2) <- (let boolean i = if i then 1 else 0 in
      match op with
	Add     -> op1 + op2
      | Sub     -> op1 - op2
      | Mult    -> op1 * op2
      | Div     -> op1 / op2
      | Equal   -> boolean (op1 =  op2)
      | Neq     -> boolean (op1 != op2)
      | Less    -> boolean (op1 <  op2)
      | Leq     -> boolean (op1 <= op2)
      | Greater -> boolean (op1 >  op2)
      | Geq     -> boolean (op1 >= op2)) ;
      exec fp (sp-1) (pc+1)
  | Lod i   -> stack.(sp)   <- globals.(i)  ; exec fp (sp+1) (pc+1)
  | Str i   -> globals.(i)  <- stack.(sp-1) ; exec fp sp     (pc+1)
  | Lfp i   -> stack.(sp)   <- stack.(fp+i) ; exec fp (sp+1) (pc+1)
  | Sfp i   -> stack.(fp+i) <- stack.(sp-1) ; exec fp sp     (pc+1)
  | Jsr(-1) -> print_endline (string_of_int stack.(sp-1)) ;
       exec fp sp (pc+1) (* print *)
  | Jsr(-2) -> print_endline (string_of_int stack.(sp-1)^" "^
			      string_of_int stack.(sp-2)^" moveto ") ;
      globals.(6) <- stack.(sp-1) ;
      globals.(7) <- stack.(sp-2) ;
      exec fp sp (pc+1) (* setPosition *)
  | Jsr(-3) -> print_endline 
	(string_of_float (float_of_int(stack.(sp-1))/.255.0)^" "^
	 string_of_float (float_of_int(stack.(sp-2))/.255.0)^" "^
	 string_of_float (float_of_int(stack.(sp-3))/.255.0)^" "^
	 "setrgbcolor ") ; (* setColor *)
      exec fp sp (pc+1)
  | Jsr(-4) -> print_endline 
	(string_of_float (float_of_int(stack.(sp-1))/.255.0)^" "^
	 string_of_float (float_of_int(stack.(sp-2))/.255.0)^" "^
	 string_of_float (float_of_int(stack.(sp-3))/.255.0)^" "^
	 "setrgbcolor clippath fill ") ; (* setDiscoColor *)
      exec fp sp (pc+1)
  | Jsr(-5) -> print_endline (string_of_int stack.(sp-1)^
			     " setlinewidth ") ; 
      exec fp sp (pc+1) (* setLineWidth *)
  | Jsr(-6) -> print_endline ("["^string_of_int stack.(sp-1)^" "^
			      string_of_int stack.(sp-2)^"] 0 setdash ") ; 
      exec fp sp (pc+1) (* setDash *)
  | Jsr(-7) -> globals.(0)  <- stack.(sp-1) ; (* setOrientation @ g0 *)
      exec fp sp     (pc+1)
  | Jsr(-8) -> stack.(sp) <- globals.(0); (* getOrientation @ g0 *)
      exec fp (sp+1)     (pc+1)
  | Jsr(-10) -> print_endline ("stroke ") ; 
      print_endline("newpath ") ; 
      print_endline(string_of_int globals.(6)^" "^string_of_int globals.(7)^
		   " moveto ") ;
      globals.(5) <- 1 ;
      exec fp (sp+1) (pc+1) (* pd *)
  | Jsr(-11) -> print_endline ("stroke newpath ") ; 
      print_endline(string_of_int globals.(6)^" "^string_of_int globals.(7)^
		   " moveto ") ;
      globals.(5) <- 0 ;
      exec fp (sp+1) (pc+1) (* pu *)
  | Jsr(-12) -> tango 1.0 stack.(sp-1) ; (* fd *)
      exec fp sp (pc+1)
  | Jsr(-13) -> tango (-1.0) stack.(sp-1) ; (* bk *)
       exec fp sp (pc+1)
  | Jsr(-14) -> globals.(0) <- (globals.(0) - stack.(sp-1)) ; (* rt *)
       exec fp sp (pc+1)
  | Jsr(-15) -> globals.(0) <- (globals.(0) + stack.(sp-1)) ; (* lt *)
       exec fp sp (pc+1)
  | Jsr(-20) -> globals.(1)  <- stack.(sp-1) ; (* setRate: rate @ g1 *)
      exec fp sp     (pc+1)
  | Jsr(-21) -> globals.(3)  <- 1 ; (* play: is_playing @ g3 *)
      exec fp (sp+1)     (pc+1)
  | Jsr(-22) -> globals.(3)  <- (-1) ; (* pause: is_playing @ g3 *)
      exec fp (sp+1)     (pc+1)
  | Jsr(-23) -> globals.(2)  <- stack.(sp-1) ; (* cue: tune_index @ g2 *)
      exec fp sp     (pc+1)
  | Jsr(-24) -> globals.(4) <- stack.(sp-1)  ; (* setVolume: volume @ g4 *)
      exec fp sp (pc+1) 
  | Jsr(-25) -> stack.(sp) <- tlength; (* getTuneLength *)
      exec fp (sp+1)     (pc+1)
  | Jsr(-26) -> stack.(sp) <- tune.{stack.(sp-1)}; (* getSample *)
      exec fp sp     (pc+1)
  | Jsr(-27) -> tune.{stack.(sp-1)} <- stack.(sp-2); (* setSample *)
      exec fp sp     (pc+1)
  | Jsr i   -> stack.(sp)   <- pc + 1       ; exec fp (sp+1) i
  | Ent i   -> stack.(sp)   <- fp           ; exec sp (sp+i+1) (pc+1)
  | Rts i   -> let new_fp = stack.(fp) and new_pc = stack.(fp-1) in
               stack.(fp-i-1) <- stack.(sp-1) ; exec new_fp (fp-i) new_pc
  | Beq i   -> exec fp (sp-1) (pc + if stack.(sp-1) =  0 then i else 1)
  | Bne i   -> exec fp (sp-1) (pc + if stack.(sp-1) != 0 then i else 1)
  | Bra i   -> exec fp sp (pc+i)
  | Hlt     -> print_endline("showpage"); free_wav tune; ()

  in exec 0 0 0
