// This is a part of the POSTAL language package.
// Copyright (C) Peter Nalyvayko (c) 2008
// All rights reserved.
//
// Programming Languages and Translators
// COMS WS4115
// Prof. Stephen Edwards
//
// This source code is only intended as a supplement to the
// POSTAL Language Reference and related
// electronic documentation provided with the language.
// See these sources for detailed information regarding the	
// POSTAL Language product.
//
package WS4115.Projects.Postal.builtin;

import WS4115.Projects.Postal.Types.*;
import com.aliasi.tokenizer.Tokenizer;
import com.aliasi.tokenizer.IndoEuropeanTokenizerFactory;
import WS4115.Projects.Postal.brown.*;

//import java.io.FileInputStream;
//import java.util.*;
//import java.io.FileInputStream;
//import java.io.IOException;
import java.io.*;

public final class BuiltinFunctionProvider {

	public BuiltinFunctionProvider() {
		
	}
	
	protected IPostalType removeLink(IPostalType link) {
		if (link instanceof PostalLink) 
			return ((PostalLink)link).resolve();
		return link;
	}
	/**
	 * Prints the expression to the standard output.
	 * @param expression
	 * @return nil
	 */
	public IPostalType println(IPostalType expression) {
		System.out.println(expression.toString());
		return PostalTypeFactory.createNil();
	}
	/**
	 * Prints the expression to the standard output. Carriage return and new line are not added.
	 * @param expression
	 * @return
	 */
	public IPostalType print(IPostalType expression) {
		System.out.print(expression.toString());
		return PostalTypeFactory.createNil();
	}
	/**
	 * Displays an assertion.
	 * @param expression
	 * @return
	 */
	public IPostalType doassert(IPostalType expression) throws Exception, AssertionException {
		boolean bDisplayAssertion = false;
		IPostalType value = removeLink(expression);
		value = removeLink(value);
		if (value instanceof PostalNilPtr)
			bDisplayAssertion = true;
		else {
			value = PostalOperations.instance().convert(value, Boolean.TYPE);
			bDisplayAssertion = !(value != null && value.isValid() && (value instanceof BooleanVariable) && ((BooleanVariable)value).value());
		}
		if (bDisplayAssertion) {
			if (value == null)
				value = PostalTypeFactory.createNil();
			//PostalEnv.instance().ReportError(value.toString() + " ==> assertion failed.");
		}
		//System.out.println(bDisplayAssertion);
		if (bDisplayAssertion) {
			throw new AssertionException("Assertion's occured.");
		}
		return PostalTypeFactory.createBoolean("assertion_result", !bDisplayAssertion);
	}
	
	/**
	 * Returns the rest of the list minus the first element. If the list contains
	 * just one element or the list is not actually a list, then the result of 
	 * the tail function is nil.
	 * @param expression
	 * @return a list object
	 */
	public IPostalType tail(IPostalType expression) {
		expression = removeLink(expression);
		if (expression instanceof PostalList) {
			PostalList newList = (PostalList)((PostalList)expression).tail();
			return newList;
		}
		return PostalTypeFactory.createNil();
	}
	
	/**
	 * Returns the head of the list.
	 * @param expression
	 * @return a list object
	 */
	public IPostalType head(IPostalType expression) {
		expression = removeLink(expression);
		if (expression instanceof PostalList) {
			PostalList newList = (PostalList)((PostalList)expression).head();
			return newList;
		}
		return PostalTypeFactory.createNil();
	}
	
	public IPostalType dup(IPostalType expression) {
		IPostalType result = null;
		expression = removeLink(expression);
		try
		{
			result = expression.clone();
		}
		catch(CloneNotSupportedException ex) {
			result = PostalTypeFactory.createError("Cannot duplicate the given expression");
		}
		return result;
	}
	
	/**
	 * Creates an empty list.
	 * @param expression
	 * @return
	 */
	public IPostalType list() {
		return PostalTypeFactory.createList("TRANSIENT_LIST");
	}
	/**
	 * Creates an empty associative collection.
	 * @return
	 */
	public IPostalType hashtable() {
		return PostalTypeFactory.createMap("TRANSIENT_HASHTABLE");
	}

	/**
	 * Returns a length of the list.
	 * @param list
	 * @return
	 */
	public IPostalType list_length(IPostalType list) {
		list = removeLink(list);
		if (list instanceof PostalList) {
			return PostalTypeFactory.createInteger("LIST_LENGTH", ((PostalList)list).size());
		}
		return PostalTypeFactory.createInteger("LIST_LENGTH", 0);
	}
	/**
	 * Returns whether the specified object is a list.
	 * @param list
	 * @return
	 */
	public IPostalType isList(IPostalType list) {
		list = removeLink(list);
		return list instanceof PostalList ? PostalTypeFactory.createBoolean("", true): PostalTypeFactory.createBoolean("", false);
	}
	/**
	 * Returns true if the given expression is a function or lambda expression.
	 * @param expression
	 * @return
	 */
	public IPostalType isLambda(IPostalType expression) {
		expression = removeLink(expression);
		return expression instanceof PostalFunction ? PostalTypeFactory.createBoolean("", true): PostalTypeFactory.createBoolean("", false);
	}
	/**
	 * Returns true if the given expression represents a numeric value.
	 * @param expression
	 * @return
	 */
	public IPostalType isNumeric(IPostalType expression) {
		expression = removeLink(expression);
		return (expression instanceof IntegerVariable) || (expression instanceof DoubleVariable) ? PostalTypeFactory.createBoolean("", true): PostalTypeFactory.createBoolean("", false);
	}
	
	/**
	 * Returns true if the given expression represents a literal (single character or a sequence of characters).
	 */
	public IPostalType isLiteral(IPostalType expression) {
		expression = removeLink(expression);
		return (expression instanceof IntegerVariable) || (expression instanceof DoubleVariable) ? PostalTypeFactory.createBoolean("", true): PostalTypeFactory.createBoolean("", false);
	}
	
	/**
	 * Returns n-th element of a list, an error if the position is invalid or 
	 * nil if the specified list object is not a list.
	 * @param pos
	 * @param list
	 * @return
	 */
	public IPostalType nth(IPostalType pos, IPostalType list) {
		pos = removeLink(pos);
		list =removeLink(list);
		if (list instanceof PostalList) {
			if (pos.getKind() == PostalTypeKind.eKindValue) {
				pos = ((PostalVariable)pos).convert(Integer.class);
				if (pos.isValid() && pos instanceof IntegerVariable) {
					return ((PostalList)list).get(((IntegerVariable)pos).value());
				}
			}
		}
		return PostalTypeFactory.createNil();
	}
	/**
	 * Inserts the element to the head of the list. If the specified list is not
	 * a list, then the function returns nil.
	 * @param elem
	 * @param list
	 * @return
	 */
	public IPostalType cons(IPostalType elem, IPostalType list) {
		list = removeLink(list);
		elem = removeLink(elem);
		
		if (list instanceof PostalNilPtr)
			list = (PostalList)PostalTypeFactory.createList("AUTO_LIST");
		if (list instanceof PostalList) {
			((PostalList)list).insert(0, elem);
			return list;
		}
		return PostalTypeFactory.createNil();
	}
	/**
	 * Removes and returns the first element of the list. The original list is affected.
	 * @param list
	 * @return the first element of the list or nil if the list is empty.
	 */
	public IPostalType pop(IPostalType list) {
		list = removeLink(list);
		if (list instanceof PostalList) {
			PostalList l = (PostalList)list;
			if (l.size() > 0) {
				IPostalType result = l.get(0);
				l.removeAt(0);
				return result;
			}
		}
		return PostalTypeFactory.createNil();
	}

	/**
	 * Tokenizes the input string and returns a list of tokens. The function uses the LingPipe API
	 * classes sunch as IndoEuropeanTokenizerFactory and Tokenizer to perform the actual
	 * tokenization.
	 * @param str
	 * @return
	 */
	public IPostalType words(IPostalType val) {
		//System.out.println("Calling 'words' with " + val.toString());
		val = removeLink(val);
		IPostalType str = PostalOperations.instance().convert(val, String.class);
		if (str.isValid() && str.getKind() == PostalTypeKind.eKindValue) {
			String s = str.toString();
			Tokenizer tokenizer = IndoEuropeanTokenizerFactory.FACTORY.tokenizer(s.toCharArray(), 0, s.length());
			String[] tokens = tokenizer.tokenize();

			PostalList list = (PostalList)PostalTypeFactory.createList("TOKENIZED_LIST");
			for(String token : tokens) {
				list.add(PostalTypeFactory.createString("TOKEN", token));
			}
			return list;
		}
		return PostalTypeFactory.createError("The input parameter was not a string.");
	}
	
	public IPostalType nbest(IPostalType val) {
		//System.out.println("Calling 'nbest' with " + val.toString());
		val = removeLink(val);
		if (val != null && val.getKind() == PostalTypeKind.eKindList) {
			PostalList list = (PostalList)val;
			HMMTagger tagger = HMMTagger.getTagger("WS4115/Projects/Postal/data/pos-en-general-brown.HiddenMarkovModel");
			if (tagger != null) {
				
				String[] tokens = new String[list.size()];
				for(int i = 0; i < list.size(); i++) {
					IPostalType t = list.get(i);
					tokens[i] = t.toString();
				}
				String[] tags = tagger.firstBest(tokens);

				if (tags.length != tokens.length)
					return PostalTypeFactory.createError("The tokens and the tags collections have different sizes.");
	
				PostalList result = (PostalList)PostalTypeFactory.createList("POS_LIST");

				for(int pos = 0; pos < tokens.length; pos++) {
					PostalList elem = (PostalList)PostalTypeFactory.createList("POST_NTAG_ELEM");
					elem.add(PostalTypeFactory.createString("TOKEN", tokens[pos]));
					elem.add(PostalTypeFactory.createString("TAG", tags[pos]));
					result.add(elem);
				}
				
				return result;
			}
		}
		return PostalTypeFactory.createError("Expected a list as an argument.");
	}
	
	public IPostalType openf(IPostalType fileName) {
		fileName =removeLink(fileName);
		if (fileName.isValid() && fileName.getKind() == PostalTypeKind.eKindValue) {
			fileName = ((PostalVariable)fileName).convert(String.class);
			String filePath = fileName.toString();

			String contents = TextFileHelper.getContents(new File(filePath));
			Tokenizer tokenizer = IndoEuropeanTokenizerFactory.FACTORY.tokenizer(contents.toCharArray(), 0, contents.length());
			String[] tokens = tokenizer.tokenize();

			PostalList list = (PostalList)PostalTypeFactory.createList("TOKENIZED_LIST");
			for(String token : tokens) {
				list.add(PostalTypeFactory.createString("TOKEN", token));
			}
			return list;
		}
		return PostalTypeFactory.createError("Cannot open specified file.");
	}
	/**
	 * Stores the contents of the given list into the file using the file name. The
	 * contents of the existing file are either overwritten or appended to depending
	 * on the value of the third argument.
	 * @param list
	 * @param fileName
	 * @param overwrite
	 * @return
	 */
	public IPostalType savef(IPostalType list, IPostalType fileName) throws FileNotFoundException, IOException {
		list = removeLink(list);
		fileName = removeLink(fileName);
		if (!(list instanceof PostalList)) {
			return PostalTypeFactory.createError("The first argument must be a list.");
		}
		if (!(fileName instanceof StringVariable)) {
			return PostalTypeFactory.createError("The second argument must a string.");
		}
		PostalList l = (PostalList)list;
		String sFileName = ((StringVariable)fileName).value();
		
		StringBuilder sb = new StringBuilder();
		for(int i = 0; i < l.size(); i++) {
			
		}
		
		File f = new File(sFileName);
		TextFileHelper.setContents(f, sb.toString());
		return PostalTypeFactory.createNil();
	}
	/**
	 * Returns a value associated with the given key. If the specified object
	 * is not a hashtable, the function will return nil.
	 * @param set
	 * @param key
	 * @return
	 */
	public IPostalType assoc(IPostalType set, IPostalType key) {
		set = removeLink(set);
		key = removeLink(key);
		if (set instanceof PostalSet) {
			PostalSet aSet = (PostalSet)set;
			String sKey = PostalOperations.instance().convert(key, String.class).toString();
			if (aSet.containsKey(sKey)) {
				return aSet.get(sKey);
			}
		}
		return PostalTypeFactory.createNil();
	}
	/**
	 * Sets a new value for the specified key.
	 * @param set
	 * @param key
	 * @param value
	 * @return
	 */
	public IPostalType sassoc(IPostalType set, IPostalType key, IPostalType value) {
		set = removeLink(set);
		key = removeLink(key);
		if (set instanceof PostalSet) {
			PostalSet aSet = (PostalSet)set;
			String sKey = PostalOperations.instance().convert(key, String.class).toString();
			aSet.put(sKey, value);
			return aSet;
		}
		return PostalTypeFactory.createNil();
	}
}
