// This is a part of the POSTAL language package.
// Copyright (C) Peter Nalyvayko (c) 2008
// All rights reserved.
//
// Programming Languages and Translators
// COMS WS4115
// Prof. Stephen Edwards
//
// This source code is only intended as a supplement to the
// POSTAL Language Reference and related
// electronic documentation provided with the language.
// See these sources for detailed information regarding the	
// POSTAL Language product.
//
/**
 * 
 */
package WS4115.Projects.Postal.Types;

import java.util.*;
/**
 * @author pnalyvayko
 *
 */
public final class PostalOperations {

	private static PostalOperations _instance;
	private Dictionary<UnaryOpKey, OpClass> _conversions;
	
	public class UnaryOpKey {
		protected Class _c1;
		protected PostalOpCodes _opCode;
		public UnaryOpKey(PostalOpCodes opCode, Class c1) {
			_c1 = c1;
			_opCode = opCode;
		}
		public PostalOpCodes OpCode() { return _opCode; }
		public Class c1() { return _c1; }
	}
	
	public final class BinaryOpKey extends UnaryOpKey {
	
		private Class _c2;
		public BinaryOpKey(PostalOpCodes opCode, Class c1, Class c2) {
			
			super(opCode, c1);
			_c2 = c2;
		}
		public Class c2() { return _c2; }
		public int hashCode() {
			String s = _c1.getCanonicalName() + ":" + _c2.getCanonicalName() + ":" + _opCode;
			return s.hashCode();
		}
		public boolean equals(Object rhs) {
			if (rhs instanceof BinaryOpKey) {
				BinaryOpKey rhsKey = (BinaryOpKey)rhs;
				boolean b = _c1 == rhsKey._c1 && _c2 == rhsKey._c2 && _opCode == rhsKey._opCode;
				return b;
			}
			return false;
		}
	}
	
	protected PostalOperations() {
		_conversions = new Hashtable<UnaryOpKey, OpClass>();
		// Assignment operations
		// Integer
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, Integer.TYPE, Integer.TYPE), AssignOperations.instance().new DoInt2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, Integer.TYPE, Double.TYPE), AssignOperations.instance().new DoDouble2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, Integer.TYPE, Boolean.TYPE), AssignOperations.instance().new DoBoolean2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, Integer.TYPE, String.class), AssignOperations.instance().new DoString2Int());
		// Double
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, Double.TYPE, Double.TYPE), AssignOperations.instance().new DoDouble2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, Double.TYPE, Integer.TYPE), AssignOperations.instance().new DoInt2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, Double.TYPE, Boolean.TYPE), AssignOperations.instance().new DoBoolean2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, Double.TYPE, String.class), AssignOperations.instance().new DoString2Double());
		// Boolean
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, Boolean.TYPE, Boolean.TYPE), AssignOperations.instance().new DoBoolean2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, Boolean.TYPE, Integer.TYPE), AssignOperations.instance().new DoInteger2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, Boolean.TYPE, Double.TYPE), AssignOperations.instance().new DoDouble2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, Boolean.TYPE, String.class), AssignOperations.instance().new DoString2Boolean());
		// String
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, String.class, String.class), AssignOperations.instance().new DoString2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, String.class, Integer.TYPE), AssignOperations.instance().new DoInteger2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, String.class, Double.TYPE), AssignOperations.instance().new DoDouble2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAssign, String.class, Boolean.TYPE), AssignOperations.instance().new DoBoolean2String());
		//
		// Equality operations
		//
		// Integer
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, Integer.TYPE, Integer.TYPE), EQOperations.instance().new DoInt2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, Integer.TYPE, Double.TYPE), EQOperations.instance().new DoDouble2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, Integer.TYPE, Boolean.TYPE), EQOperations.instance().new DoBoolean2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, Integer.TYPE, String.class), EQOperations.instance().new DoString2Int());
		// Double
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, Double.TYPE, Double.TYPE), EQOperations.instance().new DoDouble2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, Double.TYPE, Integer.TYPE), EQOperations.instance().new DoInt2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, Double.TYPE, Boolean.TYPE), EQOperations.instance().new DoBoolean2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, Double.TYPE, String.class), EQOperations.instance().new DoString2Double());
		// Boolean
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, Boolean.TYPE, Boolean.TYPE), EQOperations.instance().new DoBoolean2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, Boolean.TYPE, Integer.TYPE), EQOperations.instance().new DoInteger2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, Boolean.TYPE, Double.TYPE), EQOperations.instance().new DoDouble2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, Boolean.TYPE, String.class), EQOperations.instance().new DoString2Boolean());
		// String
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, String.class, String.class), EQOperations.instance().new DoString2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, String.class, Integer.TYPE), EQOperations.instance().new DoInteger2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, String.class, Double.TYPE), EQOperations.instance().new DoDouble2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eEQ, String.class, Boolean.TYPE), EQOperations.instance().new DoBoolean2String());
		//
		// Inequality operations
		//
		// Integer
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, Integer.TYPE, Integer.TYPE), NEQOperations.instance().new DoInt2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, Integer.TYPE, Double.TYPE), NEQOperations.instance().new DoDouble2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, Integer.TYPE, Boolean.TYPE), NEQOperations.instance().new DoBoolean2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, Integer.TYPE, String.class), NEQOperations.instance().new DoString2Int());
		// Double
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, Double.TYPE, Double.TYPE), NEQOperations.instance().new DoDouble2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, Double.TYPE, Integer.TYPE), NEQOperations.instance().new DoInt2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, Double.TYPE, Boolean.TYPE), NEQOperations.instance().new DoBoolean2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, Double.TYPE, String.class), NEQOperations.instance().new DoString2Double());
		// Boolean
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, Boolean.TYPE, Boolean.TYPE), NEQOperations.instance().new DoBoolean2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, Boolean.TYPE, Integer.TYPE), NEQOperations.instance().new DoInteger2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, Boolean.TYPE, Double.TYPE), NEQOperations.instance().new DoDouble2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, Boolean.TYPE, String.class), NEQOperations.instance().new DoString2Boolean());
		// String
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, String.class, String.class), NEQOperations.instance().new DoString2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, String.class, Integer.TYPE), NEQOperations.instance().new DoInteger2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, String.class, Double.TYPE), NEQOperations.instance().new DoDouble2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eNEQ, String.class, Boolean.TYPE), NEQOperations.instance().new DoBoolean2String());
		//
		// Greater-than operations
		//
		// Integer
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, Integer.TYPE, Integer.TYPE), GTOperations.instance().new DoInt2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, Integer.TYPE, Double.TYPE), GTOperations.instance().new DoDouble2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, Integer.TYPE, Boolean.TYPE), GTOperations.instance().new DoBoolean2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, Integer.TYPE, String.class), GTOperations.instance().new DoString2Int());
		// Double
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, Double.TYPE, Double.TYPE), GTOperations.instance().new DoDouble2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, Double.TYPE, Integer.TYPE), GTOperations.instance().new DoInt2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, Double.TYPE, Boolean.TYPE), GTOperations.instance().new DoBoolean2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, Double.TYPE, String.class), GTOperations.instance().new DoString2Double());
		// Boolean
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, Boolean.TYPE, Boolean.TYPE), GTOperations.instance().new DoBoolean2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, Boolean.TYPE, Integer.TYPE), GTOperations.instance().new DoInteger2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, Boolean.TYPE, Double.TYPE), GTOperations.instance().new DoDouble2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, Boolean.TYPE, String.class), GTOperations.instance().new DoString2Boolean());
		// String
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, String.class, String.class), GTOperations.instance().new DoString2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, String.class, Integer.TYPE), GTOperations.instance().new DoInteger2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, String.class, Double.TYPE), GTOperations.instance().new DoDouble2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGT, String.class, Boolean.TYPE), GTOperations.instance().new DoBoolean2String());
		//
		// Greater-or-equal-than operations
		//
		// Integer
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, Integer.TYPE, Integer.TYPE), GTEOperations.instance().new DoInt2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, Integer.TYPE, Double.TYPE), GTEOperations.instance().new DoDouble2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, Integer.TYPE, Boolean.TYPE), GTEOperations.instance().new DoBoolean2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, Integer.TYPE, String.class), GTEOperations.instance().new DoString2Int());
		// Double
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, Double.TYPE, Double.TYPE), GTEOperations.instance().new DoDouble2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, Double.TYPE, Integer.TYPE), GTEOperations.instance().new DoInt2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, Double.TYPE, Boolean.TYPE), GTEOperations.instance().new DoBoolean2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, Double.TYPE, String.class), GTEOperations.instance().new DoString2Double());
		// Boolean
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, Boolean.TYPE, Boolean.TYPE), GTEOperations.instance().new DoBoolean2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, Boolean.TYPE, Integer.TYPE), GTEOperations.instance().new DoInteger2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, Boolean.TYPE, Double.TYPE), GTEOperations.instance().new DoDouble2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, Boolean.TYPE, String.class), GTEOperations.instance().new DoString2Boolean());
		// String
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, String.class, String.class), GTEOperations.instance().new DoString2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, String.class, Integer.TYPE), GTEOperations.instance().new DoInteger2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, String.class, Double.TYPE), GTEOperations.instance().new DoDouble2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eGTE, String.class, Boolean.TYPE), GTEOperations.instance().new DoBoolean2String());
		//
		// Less-than operations
		//
		// Integer
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, Integer.TYPE, Integer.TYPE), LTOperations.instance().new DoInt2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, Integer.TYPE, Double.TYPE), LTOperations.instance().new DoDouble2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, Integer.TYPE, Boolean.TYPE), LTOperations.instance().new DoBoolean2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, Integer.TYPE, String.class), LTOperations.instance().new DoString2Int());
		// Double
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, Double.TYPE, Double.TYPE), LTOperations.instance().new DoDouble2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, Double.TYPE, Integer.TYPE), LTOperations.instance().new DoInt2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, Double.TYPE, Boolean.TYPE), LTOperations.instance().new DoBoolean2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, Double.TYPE, String.class), LTOperations.instance().new DoString2Double());
		// Boolean
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, Boolean.TYPE, Boolean.TYPE), LTOperations.instance().new DoBoolean2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, Boolean.TYPE, Integer.TYPE), LTOperations.instance().new DoInteger2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, Boolean.TYPE, Double.TYPE), LTOperations.instance().new DoDouble2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, Boolean.TYPE, String.class), LTOperations.instance().new DoString2Boolean());
		// String
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, String.class, String.class), LTOperations.instance().new DoString2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, String.class, Integer.TYPE), LTOperations.instance().new DoInteger2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, String.class, Double.TYPE), LTOperations.instance().new DoDouble2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLT, String.class, Boolean.TYPE), LTOperations.instance().new DoBoolean2String());
		//
		// Less-or-equal-than operations
		//
		// Integer
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, Integer.TYPE, Integer.TYPE), LTEOperations.instance().new DoInt2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, Integer.TYPE, Double.TYPE), LTEOperations.instance().new DoDouble2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, Integer.TYPE, Boolean.TYPE), LTEOperations.instance().new DoBoolean2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, Integer.TYPE, String.class), LTEOperations.instance().new DoString2Int());
		// Double
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, Double.TYPE, Double.TYPE), LTEOperations.instance().new DoDouble2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, Double.TYPE, Integer.TYPE), LTEOperations.instance().new DoInt2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, Double.TYPE, Boolean.TYPE), LTEOperations.instance().new DoBoolean2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, Double.TYPE, String.class), LTEOperations.instance().new DoString2Double());
		// Boolean
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, Boolean.TYPE, Boolean.TYPE), LTEOperations.instance().new DoBoolean2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, Boolean.TYPE, Integer.TYPE), LTEOperations.instance().new DoInteger2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, Boolean.TYPE, Double.TYPE), LTEOperations.instance().new DoDouble2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, Boolean.TYPE, String.class), LTEOperations.instance().new DoString2Boolean());
		// String
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, String.class, String.class), LTEOperations.instance().new DoString2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, String.class, Integer.TYPE), LTEOperations.instance().new DoInteger2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, String.class, Double.TYPE), LTEOperations.instance().new DoDouble2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eLTE, String.class, Boolean.TYPE), LTEOperations.instance().new DoBoolean2String());
		//
		// Subtraction operations
		//
		// Integer
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMinus, Integer.TYPE, Integer.TYPE), SubtractionOperations.instance().new DoInt2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMinus, Integer.TYPE, Double.TYPE), SubtractionOperations.instance().new DoDouble2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMinus, Integer.TYPE, Boolean.TYPE), SubtractionOperations.instance().new DoBoolean2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMinus, Integer.TYPE, String.class), SubtractionOperations.instance().new DoString2Int());
		// Double
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMinus, Double.TYPE, Double.TYPE), SubtractionOperations.instance().new DoDouble2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMinus, Double.TYPE, Integer.TYPE), SubtractionOperations.instance().new DoInt2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMinus, Double.TYPE, Boolean.TYPE), SubtractionOperations.instance().new DoBoolean2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMinus, Double.TYPE, String.class), SubtractionOperations.instance().new DoString2Double());
		//
		// Addition operations
		//
		// Integer
		_conversions.put(new BinaryOpKey(PostalOpCodes.ePlus, Integer.TYPE, Integer.TYPE), AdditionOperations.instance().new DoInt2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.ePlus, Integer.TYPE, Double.TYPE), AdditionOperations.instance().new DoDouble2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.ePlus, Integer.TYPE, Boolean.TYPE), AdditionOperations.instance().new DoBoolean2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.ePlus, Integer.TYPE, String.class), AdditionOperations.instance().new DoString2Int());
		// Double
		_conversions.put(new BinaryOpKey(PostalOpCodes.ePlus, Double.TYPE, Double.TYPE), AdditionOperations.instance().new DoDouble2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.ePlus, Double.TYPE, Integer.TYPE), AdditionOperations.instance().new DoInt2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.ePlus, Double.TYPE, Boolean.TYPE), AdditionOperations.instance().new DoBoolean2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.ePlus, Double.TYPE, String.class), AdditionOperations.instance().new DoString2Double());
		// String
		_conversions.put(new BinaryOpKey(PostalOpCodes.ePlus, String.class, String.class), AdditionOperations.instance().new DoString2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.ePlus, String.class, Integer.TYPE), AdditionOperations.instance().new DoInteger2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.ePlus, String.class, Double.TYPE), AdditionOperations.instance().new DoDouble2String());
		_conversions.put(new BinaryOpKey(PostalOpCodes.ePlus, String.class, Boolean.TYPE), AdditionOperations.instance().new DoBoolean2String());
		//
		// Multiplication operations
		//
		// Integer
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMultiply, Integer.TYPE, Integer.TYPE), MulOperations.instance().new DoInt2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMultiply, Integer.TYPE, Double.TYPE), MulOperations.instance().new DoDouble2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMultiply, Integer.TYPE, Boolean.TYPE), MulOperations.instance().new DoBoolean2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMultiply, Integer.TYPE, String.class), MulOperations.instance().new DoString2Int());
		// Double
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMultiply, Double.TYPE, Double.TYPE), MulOperations.instance().new DoDouble2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMultiply, Double.TYPE, Integer.TYPE), MulOperations.instance().new DoInt2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMultiply, Double.TYPE, Boolean.TYPE), MulOperations.instance().new DoBoolean2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eMultiply, Double.TYPE, String.class), MulOperations.instance().new DoString2Double());
		//
		// Division operations
		//
		// Integer
		_conversions.put(new BinaryOpKey(PostalOpCodes.eDivide, Integer.TYPE, Integer.TYPE), DivideOperations.instance().new DoInt2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eDivide, Integer.TYPE, Double.TYPE), DivideOperations.instance().new DoDouble2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eDivide, Integer.TYPE, Boolean.TYPE), DivideOperations.instance().new DoBoolean2Int());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eDivide, Integer.TYPE, String.class), DivideOperations.instance().new DoString2Int());
		// Double
		_conversions.put(new BinaryOpKey(PostalOpCodes.eDivide, Double.TYPE, Double.TYPE), DivideOperations.instance().new DoDouble2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eDivide, Double.TYPE, Integer.TYPE), DivideOperations.instance().new DoInt2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eDivide, Double.TYPE, Boolean.TYPE), DivideOperations.instance().new DoBoolean2Double());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eDivide, Double.TYPE, String.class), DivideOperations.instance().new DoString2Double());
		
		//
		// Postfix increment, decrement and unary minus need to be implemented here.
		//
		
		// Conjunction and disjunction operations
		_conversions.put(new BinaryOpKey(PostalOpCodes.eAND, Boolean.TYPE, Boolean.TYPE), ANDOperations.instance().new DoBoolean2Boolean());
		_conversions.put(new BinaryOpKey(PostalOpCodes.eOR, Boolean.TYPE, Boolean.TYPE), OROperations.instance().new DoBoolean2Boolean());
	}
	
	public static PostalOperations instance() {
		if (_instance == null) {
			_instance = new PostalOperations();
		}
		return _instance;
	}
	
	public IPostalType removeLink(IPostalType link) {
		if (link instanceof PostalLink)
			return ((PostalLink)link).resolve();
		return link;
	}
	
	private boolean binaryOpExists(PostalOpCodes opCode, IPostalType lhs, IPostalType rhs) {
		lhs = removeLink(lhs);
		rhs = removeLink(rhs);
		if (lhs != null && rhs != null && lhs.isValid() && rhs.isValid() &&
				lhs.getKind() == PostalTypeKind.eKindValue && rhs.getKind() == PostalTypeKind.eKindValue)
		{
			PostalVariable lhsVar = (PostalVariable)lhs;
			PostalVariable rhsVar = (PostalVariable)rhs;
			OpClass op = _conversions.get(new BinaryOpKey(opCode, lhsVar.getVarClass(), rhsVar.getVarClass()));
			return (op != null); 
		}
		else
		{
			return false;
		}
	}
	
	private IPostalType binary_op(PostalOpCodes opCode, IPostalType lhs, IPostalType rhs) {
		// Resolve links
		lhs = removeLink(lhs);
		rhs = removeLink(rhs);
		if (lhs == null || rhs == null)
			return PostalTypeFactory.createError("Either left or right operand is null.");
		
		PostalVariable lhsVar = null;
		PostalVariable rhsVar = null;
		if (lhs != null && rhs != null && lhs.isValid() && rhs.isValid() &&
				lhs.getKind() == PostalTypeKind.eKindValue && rhs.getKind() == PostalTypeKind.eKindValue)
		{
			lhsVar = (PostalVariable)lhs;
			rhsVar = (PostalVariable)rhs;
		}
		else if (rhs != null && lhs != null && lhs.isValid() && lhs.getKind() == PostalTypeKind.eKindValue)
		{
			lhsVar = (PostalVariable)lhs;
			rhsVar = (PostalVariable)PostalTypeFactory.createString("", rhs.toString());
		}
		if (lhsVar != null && rhsVar != null)
		{
			OpClass op = _conversions.get(new BinaryOpKey(opCode, lhsVar.getVarClass(), rhsVar.getVarClass()));
			if (op != null && (op instanceof IBinaryOp)) {
				IBinaryOp binaryOp = (IBinaryOp)op;
				return binaryOp.invoke(lhsVar, rhsVar);
			}
			else if (op == null)
			{
				lhs = PostalTypeFactory.createError("Operation \"" + opCode + "\" is not supported.");
				System.out.println(((PostalError)lhs).getMessage());
			}
			else
			{
				lhs = PostalTypeFactory.createError("Specified operation \"" + opCode + "\"is not a binary op.");
				System.out.println(((PostalError)lhs).getMessage());
			}
			return lhs;
		}
		else
		{
			return PostalTypeFactory.createError(opCode.toString() + " operation is not allowed.");
		}
	}
	
	private IPostalType unary_op(PostalOpCodes opCode, IPostalType lhs) {
		lhs = removeLink(lhs);
		// Resolve links
		if (lhs == null)
			return PostalTypeFactory.createError("The input operand is null.");
		PostalVariable lhsVar = null;
		if (lhs != null && lhs.isValid() &&	lhs.getKind() == PostalTypeKind.eKindValue)
		{
			lhsVar = (PostalVariable)lhs;
		}
		else if (lhs != null)
		{
			lhsVar = (PostalVariable)PostalTypeFactory.createString("", lhs.toString());
		}
		if (lhsVar != null)
		{
			OpClass op = _conversions.get(new UnaryOpKey(opCode, lhsVar.getVarClass()));
			if (op instanceof IUnaryOp) {
				IUnaryOp unaryOp = (IUnaryOp)op;
				return unaryOp.invoke(lhsVar);
			}
			return lhs;
		}		
		else
		{
			return PostalTypeFactory.createError(opCode.toString() + " operation is not allowed.");
		}
	}
	
	public IPostalType convert(IPostalType lhs, Class c) {
		// Resolve links
		if (lhs == null)
			return PostalTypeFactory.createError("The input operand is null.");
		if (lhs.getKind() == PostalTypeKind.eKindLink) {
			lhs = ((PostalLink)lhs).resolve();
		}
		PostalVariable lhsVar = null;
		if (lhs != null && lhs.isValid() &&	lhs.getKind() == PostalTypeKind.eKindValue)
		{
			lhsVar = (PostalVariable)lhs;
		}
		else  if (lhs != null)
		{
			lhsVar = (PostalVariable)PostalTypeFactory.createString("", lhs.toString());
		}
		if (lhsVar != null)
		{
			OpClass op = _conversions.get(new BinaryOpKey(PostalOpCodes.eAssign, c, lhsVar.getVarClass()));
			if (op instanceof IBinaryOp) 
			{
				IBinaryOp binaryOp = (IBinaryOp)op;
				
				PostalVariable result = null;
				if (c == Integer.TYPE)
					result = (PostalVariable)PostalTypeFactory.createInteger("", 0);
				else if (c == Double.TYPE)
					result = (PostalVariable)PostalTypeFactory.createDouble("", 0.0);
				else if (c == String.class)
					result = (PostalVariable)PostalTypeFactory.createString("", "");
				else if (c == Boolean.TYPE)
					result = (PostalVariable)PostalTypeFactory.createBoolean("", true);
				else {
					return PostalTypeFactory.createError("Unsupported type.");
				}
				if (result != null) {
					return binaryOp.invoke(result, lhsVar);
				}
			}
		}
		else
		{
			return PostalTypeFactory.createError("No conversion exists.");
		}
		return lhs;
	}
	
	public IPostalType assign(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.eAssign, lhs, rhs);
	}
	public boolean canAssign(IPostalType lhs, IPostalType rhs) {
		return binaryOpExists(PostalOpCodes.eAssign, lhs, rhs);
	}
	public IPostalType greaterThan(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.eGT, lhs, rhs);
	}
	public IPostalType greaterOrEqualThan(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.eGTE, lhs, rhs);
	}
	public IPostalType lessThan(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.eLT, lhs, rhs);
	}
	public IPostalType lessOrEqualThan(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.eLTE, lhs, rhs);
	}
	public IPostalType equalTo(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.eEQ, lhs, rhs);
	}
	public IPostalType nonEqualTo(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.eNEQ, lhs, rhs);
	}
	public IPostalType subtract(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.eMinus, lhs, rhs);
	}
	public IPostalType add(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.ePlus, lhs, rhs);
	}
	public IPostalType multiply(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.eMultiply, lhs, rhs);
	}
	public IPostalType divide(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.eDivide, lhs, rhs);
	}
	
	public IPostalType increment(IPostalType lhs) {
		return unary_op(PostalOpCodes.ePostfixIncrement, lhs);
	}
	
	public IPostalType decrement(IPostalType lhs) {
		return unary_op(PostalOpCodes.ePostfixDecrement, lhs);
	}
	
	public IPostalType minus(IPostalType lhs) {
		return unary_op(PostalOpCodes.eUnaryMinus, lhs);
	}
	
	public IPostalType and(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.eAND, lhs, rhs);
	}
	
	public IPostalType or(IPostalType lhs, IPostalType rhs) {
		return binary_op(PostalOpCodes.eOR, lhs, rhs);
	}
}
