// This is a part of the POSTAL language package.
// Copyright (C) Peter Nalyvayko (c) 2008
// All rights reserved.
//
// Programming Languages and Translators
// COMS WS4115
// Prof. Stephen Edwards
//
// This source code is only intended as a supplement to the
// POSTAL Language Reference and related
// electronic documentation provided with the language.
// See these sources for detailed information regarding the	
// POSTAL Language product.
//
package WS4115.Projects.Postal.Types;

import java.util.LinkedList;
import java.lang.StringBuilder;

public class PostalLinkedListStrategy implements IPostalListStrategy {

	private LinkedList<IPostalType> _list;
	private int _offset;
	
	public PostalLinkedListStrategy() {
		_offset = 0;
		_list = new LinkedList<IPostalType>();
	}
	
	public PostalLinkedListStrategy(LinkedList<IPostalType> masterList, int offset) {
		_list = masterList;
		_offset = offset;
	}
	
	public int add(PostalList lhs, IPostalType item) {
		_list.add(item);
		return _list.size() - 1;
	}

	public IPostalType assign(PostalList lhs, IPostalType rhs) {
		IPostalType result = lhs;
		if (canAssign(lhs, rhs)) {
			
			try
			{
				PostalList rhsList = (PostalList)rhs;
				((PostalList)lhs).reconstruct(rhsList);
			}
			catch(CloneNotSupportedException ex) {
				result = PostalTypeFactory.createError("Cannot reconstruct the list.");
			}
			return result;
			
		}
		return PostalTypeFactory.createError("Cannot assign incompatible types.");
	}
	
	public boolean canAssign(PostalList lhs, IPostalType rhs) {
		if (rhs != null && rhs instanceof PostalList) {
			return true;
		}
		return false;
	}

	public void clear(PostalList lhs) {
		while (_offset < _list.size())
			_list.removeLast();

	}

	public IPostalListStrategy clone(PostalList lhs) throws CloneNotSupportedException {
		PostalLinkedListStrategy cloned = (PostalLinkedListStrategy)super.clone();

		cloned._list =new LinkedList<IPostalType>();
		
		for(IPostalType t : _list) {
			cloned._list.add(t.clone());
		}
		
		return cloned;
	}

	public IPostalType get(PostalList lhs, int pos) {
		if (pos + _offset >= _list.size())
			return PostalTypeFactory.createNil();
		return _list.get(pos + _offset);
	}

	public void insert(PostalList lhs, int pos, IPostalType item) {
		_list.add(pos + _offset, item);
	}

	public void removeAt(PostalList lhs, int pos) {
		_list.remove(pos + _offset);
	}

	public int size(PostalList lhs) {
		return _list.size() - _offset;
	}

	public Object[] toArray(PostalList lhs) {
		Object[] copy = new Object[_list.size() - _offset];
		for (int i = _offset; i < _list.size(); i++) {
			copy[i - _offset] = _list.get(i);
		}
		return copy;
	}

	public IPostalListStrategy tail(PostalList lhs) {
		return new PostalLinkedListStrategy(_list, _offset + 1);
	}
	
	public IPostalListStrategy head(PostalList lhs) {
		return new PostalLinkedListStrategy(_list, 0);
	}
	
	public String toString(PostalList lhs) {
		StringBuilder sb = new StringBuilder();
		for(int i = _offset; i < _list.size(); i++) {
			IPostalType t = _list.get(i);
			if (sb.length() != 0)
				sb.append(' ');
			sb.append(t.toString());
		}
		sb.insert(0, '(');
		sb.append(")");
		return sb.toString();
	}
	
}
