// This is a part of the POSTAL language package.
// Copyright (C) Peter Nalyvayko (c) 2008
// All rights reserved.
//
// Programming Languages and Translators
// COMS WS4115
// Prof. Stephen Edwards
//
// This source code is only intended as a supplement to the
// POSTAL Language Reference and related
// electronic documentation provided with the language.
// See these sources for detailed information regarding the	
// POSTAL Language product.
//
/**
 * 
 */
package WS4115.Projects.Postal.Types;
import WS4115.Projects.Postal.builtin.*;
import java.util.*;
import antlr.*;
import java.lang.reflect.Method;
//import java.lang.reflect.Type;
/**
 * @author pnalyvayko
 *
 */
public final class PostalEnv {
	
	private static PostalEnv _instance;
	private Stack<ScopeLinkedItem> _scopes;
	private Object _interpreter;
	private boolean _returnRequested;
	
	public final class ScopeLinkedItem {
		private PostalScope _scope;
		private ScopeLinkedItem _parent;
		
		public ScopeLinkedItem(PostalScope scope) {
			_scope = scope;
			_parent = null;		}
		public ScopeLinkedItem(PostalScope scope, ScopeLinkedItem parent) {
			_scope = scope;
			_parent = parent;
		}
		public PostalScope getScope() { return _scope; }
		public ScopeLinkedItem getParent() { return _parent; }
	}
	
	protected PostalEnv() {
		_scopes = new Stack<ScopeLinkedItem>();
		PostalScope scope;
		_scopes.push(new ScopeLinkedItem(scope = (PostalScope)PostalTypeFactory.createScope("GLOBALSCOPE")));
		scope.putFlags(PostalScopeFlags.eNone);
		
		_returnRequested = false;
		
		// populate the global scope with buil-in functions
		BuiltinFunctionProvider builtinProvider = new BuiltinFunctionProvider();
		try
		{
			Method mi;
			ReflectedFunction f;
			
			// println function
			mi = builtinProvider.getClass().getMethod("println", IPostalType.class);
			f = new ReflectedFunction("println", builtinProvider, mi);
			scope.put(f);
			
			// print function
			mi = builtinProvider.getClass().getMethod("print", IPostalType.class);
			f = new ReflectedFunction("print", builtinProvider, mi);
			scope.put(f);
			
			// tail function
			mi = builtinProvider.getClass().getMethod("tail", IPostalType.class);
			f = new ReflectedFunction("tail", builtinProvider, mi);
			scope.put(f);

			// head function
			mi = builtinProvider.getClass().getMethod("head", IPostalType.class);
			f = new ReflectedFunction("head", builtinProvider, mi);
			scope.put(f);
			
			// list function
			mi = builtinProvider.getClass().getMethod("list");
			f = new ReflectedFunction("list", builtinProvider, mi);
			scope.put(f);

			// hashtable function
			mi = builtinProvider.getClass().getMethod("hashtable");
			f = new ReflectedFunction("hashtable", builtinProvider, mi);
			scope.put(f);

			// words function
			mi = builtinProvider.getClass().getMethod("words", IPostalType.class);
			f = new ReflectedFunction("words", builtinProvider, mi);
			scope.put(f);
			
			// nbest function
			mi = builtinProvider.getClass().getMethod("nbest", IPostalType.class);
			f = new ReflectedFunction("nbest", builtinProvider, mi);
			scope.put(f);
			
			// openf function
			mi = builtinProvider.getClass().getMethod("openf", IPostalType.class);
			f = new ReflectedFunction("openf", builtinProvider, mi);
			scope.put(f);
			
			// savef function
			mi = builtinProvider.getClass().getMethod("savef", IPostalType.class, IPostalType.class);
			f = new ReflectedFunction("savef", builtinProvider, mi);
			scope.put(f);

			// cons function
			mi = builtinProvider.getClass().getMethod("cons", IPostalType.class, IPostalType.class);
			f = new ReflectedFunction("cons", builtinProvider, mi);
			scope.put(f);
			
			// pop function
			mi = builtinProvider.getClass().getMethod("pop", IPostalType.class);
			f = new ReflectedFunction("pop", builtinProvider, mi);
			scope.put(f);

			// length function
			mi = builtinProvider.getClass().getMethod("list_length", IPostalType.class);
			f = new ReflectedFunction("length", builtinProvider, mi);
			scope.put(f);

			// isList function
			mi = builtinProvider.getClass().getMethod("isList", IPostalType.class);
			f = new ReflectedFunction("islist", builtinProvider, mi);
			scope.put(f);
			
			// nth function
			mi = builtinProvider.getClass().getMethod("nth", IPostalType.class, IPostalType.class);
			f = new ReflectedFunction("nth", builtinProvider, mi);
			scope.put(f);

			// DEBUG: assert function
			mi = builtinProvider.getClass().getMethod("doassert", IPostalType.class);
			f = new ReflectedFunction("assert", builtinProvider, mi);
			scope.put(f);
			
			// Symbol tables
			mi = this.getClass().getMethod("printSymbolTables");
			f = new ReflectedFunction("symboltable", this, mi);
			scope.put(f);

			// Dup
			mi =  builtinProvider.getClass().getMethod("dup", IPostalType.class);
			f = new ReflectedFunction("dup", builtinProvider, mi);
			scope.put(f);

			// isLambda
			mi =  builtinProvider.getClass().getMethod("isLambda", IPostalType.class);
			f = new ReflectedFunction("islambda", builtinProvider, mi);
			scope.put(f);

			// isNumeric
			mi =  builtinProvider.getClass().getMethod("isNumeric", IPostalType.class);
			f = new ReflectedFunction("isnumeric", builtinProvider, mi);
			scope.put(f);
			
			// isLiteral
			mi =  builtinProvider.getClass().getMethod("isLiteral", IPostalType.class);
			f = new ReflectedFunction("isliteral", builtinProvider, mi);
			scope.put(f);
			
			// assoc
			mi =  builtinProvider.getClass().getMethod("assoc", IPostalType.class, IPostalType.class);
			f = new ReflectedFunction("assoc", builtinProvider, mi);
			scope.put(f);
			
			// sassoc
			mi =  builtinProvider.getClass().getMethod("sassoc", IPostalType.class, IPostalType.class, IPostalType.class);
			f = new ReflectedFunction("sassoc", builtinProvider, mi);
			scope.put(f);
		}
		catch(NoSuchMethodException ex) {
			// eat the exception
			System.out.println("EXCEPTION:" + ex.getLocalizedMessage());
		}
	}
	
	public static PostalEnv instance() {
		if (_instance == null) {
			_instance = new PostalEnv();
		}
		return _instance;
	}
	
	public static void clear() {
		_instance = null;
	}
	
	public void ReportError(String errMsg) throws RecognitionException {
		throw new RecognitionException(errMsg);
	}
	
	public IPostalType findSymbol(String name, boolean parentLookup) {
		
		ScopeLinkedItem last = _scopes.lastElement();
		while (last != null) {
			PostalScope scope = last.getScope();
			if (scope.contains(name)) {
				return scope.get(name);
			}
			if (parentLookup)
				last = last.getParent();
			else
				last = null;
		}
		String msg = "Symbol \"" + name + "\" not found.";
		return PostalTypeFactory.createError(msg);
	}
	
	public void setInterpreter(Object interpreter) {
		_interpreter = interpreter;
	}
	
	public Object getInterpreter() {
		return _interpreter;
	}
	/**
	 * The method is used to create a new scope parented to the last scope. Use this function
	 * when calling lambda expressions (anonymous functions), block statements, etc.
	 * @param scope
	 */
	public void enterScope(PostalScope scope) {
		_scopes.push(new ScopeLinkedItem(scope, _scopes.lastElement()));
	}
	/**
	 * Function scope is always linked to the global scope as its parent scope.
	 * @param scope
	 */
	public void enterFunctionScope(PostalScope scope) {
		_scopes.push(new ScopeLinkedItem(scope, _scopes.firstElement()));
	}

	public PostalScope exitScope() throws IllegalStateException {
	
		// There always must be at least one scope 
		if (!_scopes.empty() && _scopes.size() > 1) {
			ScopeLinkedItem last = _scopes.pop();
			return last.getScope();
		}
		else {
			throw new IllegalStateException(); 
		}
	}
	/**
	 * Returns the scope that was last added to the context
	 * @return
	 */
	public PostalScope getCurrentScope() {
		ScopeLinkedItem last = _scopes.peek();
		return last.getScope();
	}
	/**
	 * Returns the global scope object
	 * @return
	 */
	public PostalScope getRootScope() {
		return _scopes.firstElement().getScope();
	}
	
	public boolean isReturnAllowed() {
		ScopeLinkedItem last = _scopes.lastElement();
		while (last != null) {
			PostalScope scope = last.getScope();
			last = last.getParent();
			PostalScopeFlags flags = scope.getFlags();
			if (flags == PostalScopeFlags.eTransparent)
				continue;
			if (flags == PostalScopeFlags.eReturnAllowed)
				return true;
			break;
		}
		return false;
	}
	
	public void requestReturn() {
		_returnRequested = true;
	}
	
	public void resetReturn() {
		_returnRequested = false;
	}
	
	public boolean isReturnRequested() {
		return _returnRequested;
	}
	
	public IPostalType printSymbolTables() {
		PostalList list = (PostalList)PostalTypeFactory.createList("SYMBOL_LIST");
		
		ScopeLinkedItem last = _scopes.lastElement();
		while (last != null) {
			PostalScope scope = last.getScope();
			last = last.getParent();
			
			Enumeration<String> symbols = scope.keys();
			while(symbols.hasMoreElements()) {
				IPostalType elem = scope.get(symbols.nextElement());
				PostalList elelem = (PostalList)PostalTypeFactory.createList("SYMBOL_ELEM");
				elelem.add(PostalTypeFactory.createString("SCOPE", scope.getName()));
				elelem.add(PostalTypeFactory.createString("SYMBOL", "'" + elem.getName() + "'" ));
				elelem.add(PostalTypeFactory.createString("VALUE", "'" + elem.toString() + "'"));
				list.add(elelem);
			}
		}		
		return list;
	}
	
}
