header {
  package WS4115.Projects.Postal.Grammar;
  import java.*;
  import java.io.*;
  import java.util.*;
  import WS4115.Projects.Postal.Types.*;
  import WS4115.Projects.Postal.AST.*;
  import java.lang.reflect.InvocationTargetException;
}

class POSTALWalker extends TreeParser;
options
{
	importVocab=POSTALParser;
} 

{
	protected IPostalType evaluateFunction(String methodName, AST argList, AST targetNode) throws RecognitionException
	{
		IPostalType result = null;
		 IPostalType func = PostalEnv.instance().findSymbol(methodName, true);
		 if (func instanceof PostalLink)
		 	func = ((PostalLink)func).resolve();
		 	
		 if (!func.isValid() || (func.getKind() != PostalTypeKind.eKindFunction)) {
		 	PostalEnv.instance().ReportError("Function \"" + methodName + "\" is undefined.");
		 }
		 PostalList exprList = null;
		 if (argList != null)
		 	exprList = expressionList(argList);
	 	else
			exprList = (PostalList)PostalTypeFactory.createList("EXPR_LIST");
		 	
		if (func instanceof PostalFunctionInterpreter) {
			PostalFunctionInterpreter intrf = (PostalFunctionInterpreter)func;
			if (exprList.size() != intrf.arguments().size())
			{
				result = PostalTypeFactory.createError("The function is declared to take \"" + intrf.arguments().size() + "\", but \"" + exprList.size() + "\" was specified.");
			}
			else {
				// Create a function scope and populate it with arguments
				PostalScope scope = (PostalScope)PostalTypeFactory.createScope("METHOD_CALL_SCOPE");
				scope.putFlags(PostalScopeFlags.eReturnAllowed);
				
				for (int i = 0; i < intrf.arguments().size(); i++) {
					String varName = intrf.arguments().get(i).getName();
					scope.put(PostalTypeFactory.createVariable(varName, exprList.get(i)));
				}
				
				PostalEnv.instance().enterScope(scope);
				// Interpret and execute the function implementation
				result = blockStatement((AST)intrf.getImplementation());
				PostalEnv.instance().resetReturn();
				PostalEnv.instance().exitScope();
			}
		}
		else if (func instanceof ReflectedFunction) 
		{
			//System.out.println("Invoking the built-in function \"" + func.getName() + "\"...");
			ReflectedFunction reflectedFunc = (ReflectedFunction)func;
			try
			{
				result = reflectedFunc.invoke(exprList);
			}
			catch(InvocationTargetException ex)
			{
				result = PostalTypeFactory.createNil();
				long line = ((AdvancedCommonAST)targetNode).getline();
				long column = ((AdvancedCommonAST)targetNode).getcolumn();
				PostalEnv.instance().ReportError("Assertion at line:" + line + ", column:" + column + "\n" + ex);
			}
			catch(Exception ex)
			{
				result = PostalTypeFactory.createNil();
				long line = ((AdvancedCommonAST)targetNode).getline();
				long column = ((AdvancedCommonAST)targetNode).getcolumn();
				PostalEnv.instance().ReportError("Exception at line:" + line + ", column:" + column + "\n" + ex);
			}
		}
		return result;
	}
}

program	
{
	IPostalType result = null;
	PostalEnv.instance().setInterpreter(this);
}
	: (result=functionDeclaration | result=variableDeclaration)*
	;

functionDeclaration returns [IPostalType result]
{
	PostalList parameters = null;
	result = null;
}
	: #(DECL_METHOD fid:ID parameters=argList implementation:.) 
		{
			IPostalType found = PostalEnv.instance().findSymbol(#fid.getText(), true);
			if (found.isValid() && found.getKind() == PostalTypeKind.eKindFunction)
			{
				PostalEnv.instance().ReportError("Function \"" + #fid.getText() + "\" has already been defined.");
			}
			 
			 PostalFunction f = (PostalFunction)PostalTypeFactory.createFunctionInterpreter(#fid.toString(), parameters, #implementation);
			 PostalEnv.instance().getRootScope().put(f);
			 result = f;
		}
	;
	
private	
argList returns [PostalList sv]
{
	sv = null;
	PostalList tempList = (PostalList)PostalTypeFactory.createList("ARG_LIST");
}
	: #(ELIST (s:ID
				{ 
					tempList.add(PostalTypeFactory.createString(s.getText(), s.getText())); 
				}
			)*
		) 
		{
			sv = tempList;
		}
	;

private
expressionList returns [PostalList result]	
{
	result = (PostalList)PostalTypeFactory.createList("EXPLIST");
}
	: #(EXPLIST {	IPostalType param = null; }	(param=expression { result.add(param); } )*	)
	;
	
variableDeclaration returns [IPostalType result]
{
	result = null;
}
	: #(DECL_VAR vid:ID 
		{ 
			String varName = #vid.getText();
			IPostalType found = PostalEnv.instance().findSymbol(varName, false); // do not look up parent scopes
			if (found.isValid() && found.getKind() == PostalTypeKind.eKindLink)
			{
				PostalEnv.instance().ReportError("Variable \"" + varName + "\" has already been defined.");
			}
			result = PostalTypeFactory.createVariable(varName);
	 		PostalEnv.instance().getCurrentScope().put(result);
		}
				#(VAR_INITIALIZER
					(initexpr:.
						{ 
							// Note: forward declarations are not supported, i.e. any functions returning
							// values used to initialize the variable must be declared before the 
							// variable declaration.
							IPostalType initialValue = expression(#initexpr);	
							result.assign(initialValue);
						}
					)?
				)
		)
	;
			
blockStatement returns [IPostalType result]
{
	result = null;
}
	: #(DECL_SCOPE  
			{ 
				PostalScope scope = (PostalScope)PostalTypeFactory.createScope("DECL_SCOPE");
				scope.putFlags(PostalScopeFlags.eTransparent);
				PostalEnv.instance().enterScope(scope);
			}
			(stat:. 
				{
					result = statement(#stat);
					// If the program requested an early return, then
					// short-circuit the current control flow and exit the loop.
					if (PostalEnv.instance().isReturnRequested())
						break;
				}
			)*
			{ 
				scope = PostalEnv.instance().exitScope();
			}
		)
	;

statement returns [IPostalType result]
{
	result = null;
}
	: result = variableDeclaration
	| #(LITERAL_return (result = expression)?
			{ 
				if (result == null)
					result = PostalTypeFactory.createBoolean("RET_VAL", true);
				if (!PostalEnv.instance().isReturnAllowed())
				{
					PostalEnv.instance().ReportError("Current program flow does not allow early return.");
				}
				PostalEnv.instance().requestReturn();
			}
		)
	| #(LITERAL_if cond1:. expr1:. (elseexpr:.)?
			{
				boolean bSuccess = false;
				IPostalType oBool = expression(#cond1);
				oBool = PostalOperations.instance().convert(oBool, Boolean.TYPE);
				if (oBool.isValid() && (oBool instanceof BooleanVariable))
					bSuccess = ((BooleanVariable)oBool).value();
				if (bSuccess)
					result = expression(#expr1);
				else 
				{
					if (elseexpr != null)
						result = expression(#elseexpr);
				}
			}
		)
	| #(LITERAL_dolist dolistiter:. dolistexpr:.
		{
			IPostalType o = expression(#dolistiter);
			if (o.getKind() == PostalTypeKind.eKindLink)
				o = ((PostalLink)o).resolve();
			if (!(o instanceof PostalList)) {
				System.out.println("Specified type is: " + o.getKind());
				PostalEnv.instance().ReportError("dolist operation requires a list.");
			}
			else 
			{
				PostalList list = (PostalList)o;
				assert list != null;
				
				String methodName=lambdaOrIdExpression(#dolistexpr);
				IPostalType func = PostalEnv.instance().findSymbol(methodName, true);
				if (!func.isValid() || (func.getKind() != PostalTypeKind.eKindFunction)) {
					PostalEnv.instance().ReportError("Function \"" + methodName + "\" is undefined.");
				}
				if (((PostalFunction)func).arguments().size() != 1)
					PostalEnv.instance().ReportError("dolist expects a function or a lambda expression with a single argument.");
					
				PostalList exprList = (PostalList)PostalTypeFactory.createList("EXPR_LIST");
				while (list.size() != 0)
				{
					exprList.clear();
					exprList.add(list.get(0));
					
					if (func instanceof PostalFunctionInterpreter) {
						PostalFunctionInterpreter intrf = (PostalFunctionInterpreter)func;
						// Create a function scope and populate it with arguments
						PostalScope scope = (PostalScope)PostalTypeFactory.createScope("METHOD_CALL_SCOPE");
						scope.putFlags(PostalScopeFlags.eReturnAllowed);
						
						for (int i = 0; i < intrf.arguments().size(); i++) {
							String varName = intrf.arguments().get(i).getName();
							scope.put(PostalTypeFactory.createVariable(varName, exprList.get(i)));
						}
						
						PostalEnv.instance().enterScope(scope);
						// Interpret and execute the function implementation
						result = blockStatement((AST)intrf.getImplementation());
						PostalEnv.instance().resetReturn();
						PostalEnv.instance().exitScope();
					}
					else if (func instanceof ReflectedFunction) 
					{
						ReflectedFunction reflectedFunc = (ReflectedFunction)func;
						try
						{
							result = reflectedFunc.invoke(exprList);
						}
						catch(Exception ex)
						{
							result = PostalTypeFactory.createNil();
							PostalEnv.instance().ReportError(ex.getMessage());
						}
					}
					list = list.tail();
				}
				list = list.head();
			}
		}
		)
	| #(LITERAL_while cond:. while_body:.
			{
				boolean bContinue =  false;
				IPostalType oBool = result = expression(#cond);
				oBool = PostalOperations.instance().convert(oBool, Boolean.TYPE);
				if (oBool.isValid() && (oBool instanceof BooleanVariable))
					bContinue = ((BooleanVariable)oBool).value();
				while (bContinue)
				{
					result = expression(#while_body);
					oBool = expression(#cond);
					//System.out.println(oBool.toString());
					if (oBool.isValid() && (oBool instanceof BooleanVariable))
						bContinue = ((BooleanVariable)oBool).value();
					else
						bContinue = false;
				}
			}
		)
	| result = expression 
	;
	
expression returns [IPostalType result]
{
	result = null;
}
	: #(ASSIGN lhs:. rhs:.)
		{ 
			result = expression(#lhs);						
			IPostalType rhsVal = expression(#rhs);
			result = result.assign(rhsVal);
		}
	| result = expressionNoAssignment
	;
	
private
expressionNoAssignment returns [IPostalType result]
{
	result = null;
	IPostalType rhs = null;
	IPostalType expr = null;
	String methodName = null;
	IPostalType lhs;
}
	: #(LT lhs=expression rhs=expression
		{ 
			result = PostalOperations.instance().lessThan(lhs, rhs);
		}
		)
	| #(GT lhs=expression rhs=expression
		{ 
			result = PostalOperations.instance().greaterThan(lhs, rhs);
		}
		)
	| #(EQ lhs=expression rhs=expression
		{ 
			result = PostalOperations.instance().equalTo(lhs, rhs);
		}
		)
	| #(NEQ lhs=expression rhs=expression
		{ 
			result = PostalOperations.instance().nonEqualTo(lhs, rhs);
		}
		)
	| #(GTE lhs=expression rhs=expression
		{ 
			result = PostalOperations.instance().greaterOrEqualThan(lhs, rhs);
		}
		)
	| #(LTE lhs=expression rhs=expression
		{ 
			result = PostalOperations.instance().lessOrEqualThan(lhs, rhs);
		}
		)
	| #(AND lhs=expression rhs=expression
		{ 
			result = PostalOperations.instance().and(lhs, rhs);
		}
		)
	| #(OR lhs=expression rhs=expression
		{ 
			result = PostalOperations.instance().or(lhs, rhs);
		}
		)
	| #(PLUS lhs=expression rhs=expression
		{ 
			result = PostalOperations.instance().add(lhs, rhs);
		}
		)
	| #(MINUS lhsexp:. (rhsexp:.)?
		{ 
			lhs = expression(#lhsexp);
			if (rhsexp != null) {
				rhs = expression(#rhsexp);
				result = PostalOperations.instance().subtract(lhs, rhs);
			}
			else {
				lhs = expression(#lhsexp);
				if (lhs instanceof PostalLink)
					lhs = ((PostalLink)lhs).resolve();
				if (lhs instanceof IntegerVariable)
					result = PostalOperations.instance().subtract(PostalTypeFactory.createInteger("", 0), lhs);
				else
					result = PostalOperations.instance().subtract(PostalTypeFactory.createDouble("", 0.0), lhs);
			}
		}
		)
	| #(STAR lhs=expression rhs=expression
		{ 
			result = PostalOperations.instance().multiply(lhs, rhs);
		}
		)
	| #(DIV lhs=expression rhs=expression
		{ 
			result = PostalOperations.instance().divide(lhs, rhs);
		}
		)
	| #(NUM_CONST 
		{ 
			// NUM_CONST can be a integer or floating-point number.
			try 
			{ 
				// Trying to convert to integer value
				int iVal = Integer.parseInt(#NUM_CONST.getText());
				result = PostalTypeFactory.createInteger("INT_CONST", iVal);
			} 
			catch(NumberFormatException ex) 
			{
				// conversion to integer has failed, next step is to 
				// try converting to a floating-point number.
				try
				{
					Double dblVal = Double.parseDouble(#NUM_CONST.getText());
					result = PostalTypeFactory.createDouble("DBL_CONST", dblVal);
				}
				catch(NumberFormatException ex2) 
				{
					// There is something wrong, bail out
					throw ex2;
				}
			}
		 })
	| #(CHAR_LITERAL 
		{ result = PostalTypeFactory.createString("STR_CONST", #CHAR_LITERAL.getText()); })
	| #(STRING_LITERAL 
		{ 
			String s = #STRING_LITERAL.getText();
			result = PostalTypeFactory.createString("STR_CONST", s.substring(1, s.length() - 1)); 
		})
	| #(LITERAL_true 
		{ result = PostalTypeFactory.createBoolean("BOOL_CONST", true); })
	| #(LITERAL_nil 
		{ result = PostalTypeFactory.createNil(); })
	| #(LITERAL_numeric rhs=expression 
		{ result = PostalOperations.instance().convert(rhs, Integer.TYPE); } )
	| result = lambdaExpression
	| #(LITERAL_apply methodName=lambdaOrIdExpression (explist:.)?
			{
				result = evaluateFunction(methodName, #explist, #LITERAL_apply);
			}
		)
	| #(ID 
			{ 
				result = PostalEnv.instance().findSymbol(#ID.getText(), true);
				if (result.getKind() != PostalTypeKind.eKindLink) {
					PostalEnv.instance().ReportError("Symbol \"" + #ID.getText() + "\" is undefined.");
				}
			}
		)
	| #(METHOD_CALL methodName=methodNameIdentifier (method_explist:.)?
			{
				result = evaluateFunction(methodName.toString(), #method_explist, #METHOD_CALL);
			}
		)	
	;

private
methodNameIdentifier returns [String result]
{
	result = null;
}
	: #(ID 
		{ result = #ID.getText(); } 
		)
	;
	
private
lambdaExpression returns [IPostalType fnLambda]
{
	PostalList parameters = null; 
	fnLambda = null;
}
	 : #(LITERAL_lambda parameters=argList implementation:.)
	 	{ 
	 		// Create a transient name for the lambda expression
	 		String name = "anonymous_lambda_expression-" + Long.toString(System.currentTimeMillis());
	 		fnLambda = PostalTypeFactory.createFunctionInterpreter(name, parameters, #implementation); 
			PostalEnv.instance().getCurrentScope().put(fnLambda);
 		}
	 ;
	 
	
private	 
lambdaOrIdExpression returns [String anonymousName]
{
	anonymousName = null;
	IPostalType lambdaDecl = null;
}	
	: #(ID { anonymousName = #ID.getText(); } )
	| lambdaDecl=lambdaExpression
		{ anonymousName = lambdaDecl.getName(); }
	;