// This is a part of the POSTAL language package.
// Copyright (C) Peter Nalyvayko (c) 2008
// All rights reserved.
//
// Programming Languages and Translators
// COMS WS4115
// Prof. Stephen Edwards
//
// This source code is only intended as a supplement to the
// POSTAL Language Reference and related
// electronic documentation provided with the language.
// See these sources for detailed information regarding the	
// POSTAL Language product.
//

import java.io.*;
import WS4115.Projects.Postal.Grammar.*;
import antlr.collections.*;
import WS4115.Projects.Postal.Types.*;

public class Main {

	static boolean showTree = false;
	/**
	 * @param args
	 */
	public static void main(String[] args) {
		// Use a try/catch block for parser exceptions
		try
		{
			// if we have at least one command-line argument
			if (args.length > 0)
			{
				//System.out.println("Parsing...");				
				// for each directory/file specified on the command line
				 for (int i = 0; i < args.length; i++)
				{
					if (args[i].equals("-showtree"))
					{
						showTree = true;
					}
					else
					{
						doFile(new File(args[i])); // parse it
					}
				}
			}
			else
				System.err.println("Usage: java Main [-showtree] " + "[<Directory> | <file name>]");
		}
		catch (Exception e)
		{
			System.err.println("Exception: "+e);
			e.printStackTrace();   // so we can get stack trace		
		}
	}
	
	private static void doFile(File f) throws Exception
	{
		// If this is a directory, walk each file/dir in that directory
		if (f.isDirectory()) {
			String files[] = f.list();
			for(int i=0; i < files.length; i++)
				doFile(new File(f, files[i]));
		}

		// otherwise, if this is a java file, parse it!
		else  {
			//System.err.println("   "+f.getAbsolutePath());
			parseFile(f.getName(), new BufferedReader(new FileReader(f)));
		}
	}

	// Here's where we do the real work...
	public static void parseFile(String f, Reader r)
								 throws Exception {
		try {
			// Create a scanner that reads from the input stream passed to us
			POSTALLexer lexer = new POSTALLexer(r);
			// Create a parser that reads from the scanner
			POSTALParser parser = new POSTALParser(lexer);
			parser.setASTNodeClass("WS4115.Projects.Postal.AST.AdvancedCommonAST");
			//
			// Parse the input file.
			//
			parser.program();
			if (!parser.hasErrorOccured()) {
				//
				// Perform the semantic analysis
				//
				POSTALWalker walker = new POSTALWalker();
				walker.setASTNodeClass("WS4115.Projects.Postal.AST.AdvancedCommonAST");
				AST rootNode = parser.getAST();
				walker.program(rootNode);
				
				// Execute the program's entry point. Entry point is a designated function,
				// which is executed by the compiler once the syntactical and semantical
				// phases are completed.
				
				IPostalType entryPoint = PostalEnv.instance().findSymbol("main", true);
				if (entryPoint instanceof PostalLink) 
					entryPoint = ((PostalLink)entryPoint).resolve();
				if (entryPoint.isValid() && 
						entryPoint.getKind() == PostalTypeKind.eKindFunction &&
						entryPoint instanceof PostalFunctionInterpreter) {
					PostalFunctionInterpreter intrf = (PostalFunctionInterpreter)entryPoint;
					if (intrf.arguments().size() != 0)
					{
						PostalEnv.instance().ReportError("The main expects no arguments.");
					}
					else {
						// Create a function scope and populate it with arguments
						PostalScope scope = (PostalScope)PostalTypeFactory.createScope("METHOD_CALL_SCOPE");
						scope.putFlags(PostalScopeFlags.eReturnAllowed);
						PostalEnv.instance().enterScope(scope);
						// Interpret and execute the function implementation
						walker.blockStatement((AST)intrf.getImplementation());
						PostalEnv.instance().exitScope();
					}
				}
				PostalEnv.clear();
				
				if (showTree)
				{
					antlr.DumpASTVisitor visitor = new antlr.DumpASTVisitor();
					visitor.visit(rootNode);
				}
			}
		}
		catch (Exception e) {
			System.err.println("error.\nparser exception: "+e);
			e.printStackTrace();   // so we can get stack trace
		}
	}
}
