------------------------------
-- CSEE 4840 Embedded System Design - Audio Sampling OPB Peripheral
--
-- SOBA Server
--
--  Team Warriors: Avraham Shinnar  as1619@columbia.edu
--                 Benjamin Dweck  bjd2102@columbia.edu
--                 Oliver Irwin    omi3@columbia.edu
--                 Sean White      sw2061@columbia.edu
--
------------------------------


---------------------------------------------------------------------------
--
-- 32-bit Positive Edge Triggered Left Shift Register
-- with Serial In and Parallel Out
-- 
-- Used to receive samples serially from the audio codec
--
---------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;

entity shift_32 is
  port(
    SI : in  std_logic;
    C  : in  std_logic;
    D  : out std_logic_vector(31 downto 0)
  );
end shift_32;

architecture arch of shift_32 is

  signal tmp : std_logic_vector(31 downto 0);

begin

  process (C)
  begin
    if C'event and C='1' then
      tmp <= tmp(30 downto 0) & SI;
    end if;
  end process;

  D <= tmp;

end arch;



---------------------------------------------------------------------------
--
-- 32-bit Positive Edge Triggered Latch
-- with Asynchronous Reset
-- 
-- Used to receive samples serially from the audio codec
--
---------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;

entity latch_32 is
  port(
    R : in  std_logic;
    C : in  std_logic;
    D : in  std_logic_vector(31 downto 0);
    Q : out std_logic_vector(31 downto 0)
  );
end latch_32;

architecture arch of latch_32 is

  signal tmp : std_logic_vector(31 downto 0);

begin

  process (C, R)
  begin
    if R='1' then
      tmp <= X"00000000";
    elsif C'event and C='1' then
      tmp <= D;
    end if;
  end process;

  Q <= tmp;

end arch;



---------------------------------------------------------------------------
--
-- Audio Sampler Peripheral
-- 
---------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;

entity opb_audio_sampler is
  
  generic (
    C_OPB_AWIDTH : integer                   := 32;
    C_OPB_DWIDTH : integer                   := 32;
    C_BASEADDR   : std_logic_vector(0 to 31) := X"00000000";
    C_HIGHADDR   : std_logic_vector(0 to 31) := X"FFFFFFFF");

  port (
    -- OPB Input Signals
    OPB_Clk     : in  std_logic;
    OPB_Rst     : in  std_logic;
    OPB_ABus    : in  std_logic_vector(0 to C_OPB_AWIDTH-1);
    OPB_BE      : in  std_logic_vector(0 to C_OPB_DWIDTH/8-1);
    OPB_DBus    : in  std_logic_vector(0 to C_OPB_DWIDTH-1);
    OPB_RNW     : in  std_logic;
    OPB_select  : in  std_logic;
    OPB_seqAddr : in  std_logic;        -- Sequential Address

    -- OPB Output Signals
    AUS_DBus    : out std_logic_vector(0 to C_OPB_DWIDTH-1);
    AUS_errAck  : out std_logic;        -- (unused)
    AUS_retry   : out std_logic;        -- (unused)
    AUS_toutSup : out std_logic;        -- Timeout suppress
    AUS_xferAck : out std_logic;        -- Transfer acknowledge

    -- Interrupt
    Interrupt   : out std_logic;        -- Interrupt Signal

    -- Audio IO Signals
    AU_MCLK     : out std_logic;        -- Audio Chip Master Clock
    AU_LRCK     : out std_logic;        -- Audio Left/Right Channel Clock
    AU_BCLK     : out std_logic;        -- Audio Bit Clock
    AU_SDTI     : out std_logic;        -- Audio Data In (TO Codec)
    AU_SDTO0    : in  std_logic);       -- Audio Data Out 0 (FROM Codec)

end opb_audio_sampler;

architecture behavioral of opb_audio_sampler is

    -- Clock counter used to generate audio transmission clocks
    signal opb_clk_count : std_logic_vector(10 downto 0);

    -- Internal Utility Signals
    signal int_selected    : std_logic;    -- Decoded chip select signal
    signal int_aus_dbus_en : std_logic;    -- Enables output to AUS_DBus
    signal int_shift_out   : std_logic_vector(31 downto 0);    -- Output of shift-in register
    signal int_sample      : std_logic_vector(31 downto 0);    -- Last received sample
    
    -- Internal Buffer Signals
    signal int_interrupt : std_logic;
    signal int_au_mclk   : std_logic;
    signal int_au_bclk   : std_logic;
    signal int_au_lrck   : std_logic;
    signal int_au_sdto0  : std_logic;

    -- State constants
    constant Idle : std_logic := '0';
    constant Xfer : std_logic := '1';
  
    signal present_state, next_state : std_logic;

    -- 32-bit Shift Left Register with Serial In and Parallel Out
    component shift_32 is
        port (
            SI : in  std_logic;
            C  : in  std_logic;
            D  : out std_logic_vector(31 downto 0));
    end component;
  
    -- 32-bit Latch with Asynchronous Reset
    component latch_32 is
        port (
            R : in  std_logic;
            C : in  std_logic;
            D : in  std_logic_vector(31 downto 0);
            Q : out std_logic_vector(31 downto 0));
    end component;

    -- Output buffer for FPGA outputs to audio codec
    component OBUF_F_8 is
      port (
        O : out std_ulogic;     -- Out to Audio codec
        I : in  std_ulogic);    -- In from Audio codec
    end component;

    component IBUF is
      port (
        I : in  std_logic;
        O : out std_logic);
    end component;

begin

    -- Receiving Shift Register
    shift_rcv : shift_32
        port map (SI => int_au_sdto0,
                  C  => int_au_bclk,
                  D  => int_shift_out);

    -- Sample Latch
    latch_sample : latch_32
        port map (R => OPB_Rst,
                  C => int_au_lrck,
                  D => int_shift_out,
                  Q => int_sample);

    -- AU_MCLK Buffer
    au_mclk_buff : OBUF_F_8
        port map (
          O => AU_MCLK,
          I => int_au_mclk);

    -- AU_BCLK Buffer
    au_bclk_buff : OBUF_F_8
        port map (
          O => AU_BCLK,
          I => int_au_bclk);

    -- AU_LRCK Buffer
    au_lrck_buff : OBUF_F_8
        port map (
          O => AU_LRCK,
          I => int_au_lrck);

    -- AU_SDTI Buffer
    au_sdti_buff : OBUF_F_8
        port map (
          O => AU_SDTI,
          I => int_au_sdto0);

    -- AU_MCLK Buffer
    au_sdto0_buff : IBUF
        port map (
          O => int_au_sdto0,
          I => AU_SDTO0);

    -- Produce chip select signal by decoding OPB address and checking OPB_select
    int_selected <= '1' when OPB_select = '1' and
                             OPB_ABus(0 to C_OPB_AWIDTH-9) =
                             C_BASEADDR(0 to C_OPB_AWIDTH-9)
                        else '0';


    -- Tie unused OPB slave outputs
    AUS_errAck <= '0';
    AUS_retry <= '0';
    AUS_toutSup <= '0';

    -- Tie OPB ack to output enable signal
    AUS_xferAck <= int_aus_dbus_en;

    -- Tie unused audio output
    --AU_SDTI <= '0';

    -- Tie off-FPGA signals to internal buffer signals
    INTERRUPT <= int_interrupt;

    --int_au_sdto0 <= AU_SDTO0;

    -- Generate Audio Transmission Clocks
    int_au_mclk <= opb_clk_count(1);
    int_au_bclk <= opb_clk_count(4);
    int_au_lrck <= NOT opb_clk_count(9);

    opb_clk_count_proc : process(OPB_Clk, OPB_Rst)
    begin
        if OPB_Rst = '1' then
            opb_clk_count <= "00000000000";
        elsif OPB_Clk'event and OPB_Clk='1' then
            opb_clk_count <= opb_clk_count + 1;
        end if;
    end process opb_clk_count_proc;

    -- Gate output to AUS_DBus with int_aus_dbus_en
    gate_aus_dbus_output : process (OPB_Rst, OPB_RNW, int_aus_dbus_en)
    begin
        if OPB_Rst = '1' then
            AUS_DBus(0 to C_OPB_DWIDTH-1) <= (others => '0');
        else
            if int_aus_dbus_en = '1' then
                AUS_DBus(0 to C_OPB_DWIDTH-1) <= int_sample(C_OPB_DWIDTH-1 downto 0);
            else
                AUS_DBus(0 to C_OPB_DWIDTH-1) <= (others => '0');
            end if;
        end if;
    end process gate_aus_dbus_output;

    -- Send interrupt pulse upon receiving sample
    -- SMALL HACK: Using int_au_mclk to reset the interrupt
    send_interrupt : process (int_au_mclk, int_au_lrck, int_interrupt)
    begin
        if int_au_mclk='1' then
            int_interrupt <= '0';
        elsif int_au_lrck'event and int_au_lrck='1' then
            int_interrupt <= '1';
        end if;
    end process send_interrupt;

    -- Sequential part of the FSM
    fsm_seq : process(OPB_Clk, OPB_Rst)
    begin
        if OPB_Rst = '1' then          -- Asynchronous reset to Idle state
            present_state <= Idle;
        elsif OPB_Clk'event and OPB_Clk = '1' then     -- Positive edge OPB_Clk
            present_state <= next_state;               -- advance to next state
        end if;
    end process fsm_seq;

    -- Combinational part of the FSM
    fsm_comb : process (present_state, int_selected)
    begin
        int_aus_dbus_en <= '0';
	    
        case present_state is
    
            when Idle =>
                int_aus_dbus_en <= '0';
                if int_selected='1' then
                    next_state <= Xfer;
                else
                    next_state <= Idle;
                end if;

            when Xfer =>
                int_aus_dbus_en <= '1';
                next_state <= Idle;

            when others =>
                int_aus_dbus_en <= '0';
                next_state <= Idle;

        end case;
    end process fsm_comb;

end behavioral;


