-------------------------------------------------------------------------------
--
-- Audio Controller OPB Peripheral
--
-- Benjamin Dweck
-- bjd2102@columbia.edu
--
-- Oliver Irwin
-- omi3@columbia.edu
--
-------------------------------------------------------------------------------




---------------------------------------------------------------------------
--
-- Clock Divider
--
-- Notes: Clock output starts out low after reset
--
-- Used to divide OPB_Clk to generate AU_CCLK
--
---------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;

entity clock_divider is

  generic (
    FACTOR : integer := 10);    -- Clock divide-by factor
    
  port (
    clk_in : in std_logic;      -- Input clock
    rst : in std_logic;         -- Reset signal
    clk_out : out std_logic);   -- Output clock

end clock_divider;

architecture arch of clock_divider is

  signal count : integer := 0;
  signal output : std_logic := '0';

begin

  clk_out <= output;

  divide: process(clk_in, rst)
  begin
    if (rst = '1') then
      output <= '0';
      count <= 0;
    elsif (clk_in'event and clk_in='1') then
      count <= count + 1;
      
      if (count = (FACTOR/2)-1) then
        output <= NOT output;
        count <= 0;
      end if;
      
    end if;
  end process divide;

end arch;



---------------------------------------------------------------------------
--
-- 5-bit Unsigned Negative Edge Triggered Up Counter
-- with Asynchronous Reset
--
-- Used to keep track of number of AU_CCLK cycles
--
---------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;

entity counter is
  port(
    clk, reset : in  std_logic;
    count      : out std_logic_vector(4 downto 0)
  );
end counter;

architecture arch of counter is

  signal tmp: std_logic_vector(4 downto 0);

begin

  process (clk, reset)
  begin
    if (reset = '1') then
      tmp <= "00000";
    elsif (clk'event and clk = '0') then
      tmp <= tmp + 1;
    end if;
  end process;

  count <= tmp;

end arch;



---------------------------------------------------------------------------
--
-- 16-bit Negative Edge Triggered Right Shift Register
-- with Active High Load and Serial Out
-- 
-- Used to latch onto data to be output to the audio codec and
-- shift it out serially.
--
---------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;

entity shift_out is
  port(
    C, ALOAD : in  std_logic;
    D        : in  std_logic_vector(15 downto 0);
    SO       : out std_logic
  );
end shift_out;

architecture arch of shift_out is

  signal tmp : std_logic_vector(15 downto 0);

begin

  process (C, ALOAD, D)
  begin
    if (ALOAD='1') then
      tmp <= D;
    elsif (C'event and C='0') then
      tmp <= '0' & tmp(15 downto 1);
    end if;
  end process;

  SO <= tmp(0);

end arch;



---------------------------------------------------------------------------
--
-- 8-bit Positive Edge Triggered Right Shift Register
-- with Serial In and Parallel Out
-- 
-- Used to receive data serially from the audio codec
--
---------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;

entity shift_in is
  port(
    SI : in  std_logic;
    C  : in  std_logic;
    D  : out std_logic_vector(7 downto 0)
  );
end shift_in;

architecture arch of shift_in is

  signal tmp : std_logic_vector(7 downto 0);

begin

  process (C)
  begin
    if (C'event and C='1') then
      tmp <= SI & tmp(7 downto 1);
    end if;
  end process;

  D <= tmp;

end arch;



---------------------------------------------------------------------------
--
-- Audio Peripheral
--
---------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;

entity opb_audio_cntlr is
  
  generic (
    C_OPB_AWIDTH : integer                   := 32;
    C_OPB_DWIDTH : integer                   := 32;
    C_BASEADDR   : std_logic_vector(0 to 31) := X"00000000";
    C_HIGHADDR   : std_logic_vector(0 to 31) := X"FFFFFFFF");

  port (
    -- OPB Input Signals
    OPB_Clk     : in  std_logic;
    OPB_Rst     : in  std_logic;
    OPB_ABus    : in  std_logic_vector(0 to C_OPB_AWIDTH-1);
    OPB_BE      : in  std_logic_vector(0 to C_OPB_DWIDTH/8-1);
    OPB_DBus    : in  std_logic_vector(0 to C_OPB_DWIDTH-1);
    OPB_RNW     : in  std_logic;
    OPB_select  : in  std_logic;
    OPB_seqAddr : in  std_logic;        -- Sequential Address

    -- OPB Output Signals
    Sln_DBus    : out std_logic_vector(0 to C_OPB_DWIDTH-1);
    Sln_errAck  : out std_logic;        -- (unused)
    Sln_retry   : out std_logic;        -- (unused)
    Sln_toutSup : out std_logic;        -- Timeout suppress
    Sln_xferAck : out std_logic;        -- Transfer acknowledge

    -- Audio IO Signals
    AU_CSN      : out   std_logic;                      -- Audio Chip Select (Active LOW)
    PB_D        : inout std_logic_vector(15 downto 0);  -- Off-FPGA Peripheral
                                                        -- Data Bus
    
--    AU_CCLK     : out std_logic;        -- Audio Cntl Clock
--    AU_CDTI     : out std_logic;        -- Audio Cntl Data In  (TO Codec)
--    AU_CDTO     : in  std_logic);       -- Audio Cntl Data Out (FROM Codec)

end opb_audio_cntlr;


architecture behavioral of opb_audio_cntlr is

  -- Audio Address and Data Bus Widths
  constant AU_AWIDTH : integer := 8;    -- Width of audio controller address
  constant AU_DWIDTH : integer := 8;    -- Width of audio controller data word

  -- Internal buffer/utillity signals
  signal int_au_csn            : std_logic := '1';    -- Used as audio codec chip select (active-low)
  signal int_au_cclk           : std_logic;           -- Used as serial clock signal to audio codec
  signal int_au_cclk_count     : std_logic_vector(4 downto 0);      -- Used to keep track of # bits tx/rx to audio codec
  signal int_au_cclk_count_rst : std_logic;                     -- To reset cclk counter
  signal int_au_cdti           : std_logic;                                       -- AU_CDTI buffer signal
  signal int_au_cdti_buff      : std_logic_vector(15 downto 0);  -- Input to CDTI shift reg
  signal int_au_cdti_load      : std_logic;                                       -- CDTI shift reg load signal
  signal int_au_cdto_buff      : std_logic_vector(AU_DWIDTH-1 downto 0);    -- Data to be sent to the OPB bus from CDTO shift reg

  -- Internal utillity signals
  signal int_selected      : std_logic;         -- Internal chip_select signal
  signal int_output_enable : std_logic;         -- Enable output from peripheral to OPB bus

  -- State constants
  -- Critical: Sln_xferAck is generated directly from state bit 0!
  constant STATE_BITS : integer := 2;
  constant Idle       : std_logic_vector(0 to STATE_BITS-1) := "00";
  constant Transfer   : std_logic_vector(0 to STATE_BITS-1) := "01";
  constant Ack        : std_logic_vector(0 to STATE_BITS-1) := "11";

  signal present_state, next_state : std_logic_vector(0 to STATE_BITS-1);

  -- Clock divider (used with OPB_Clk to generate AU_CCLK)
  component clock_divider is
    generic (FACTOR : integer := 10);     -- Divide 50 MHz OPB_Clk => 5 MHz AU_CCLK
    port (
      clk_in  : in  std_logic;
      rst     : in  std_logic;
      clk_out : out std_logic); 
  end component;

  -- CCLK Counter
  component counter is
    port(
      clk   : in  std_logic;
      reset : in  std_logic;
      count : out std_logic_vector(4 downto 0));
  end component;

  -- Transmit Register (AU_CDTI)
  component shift_out is
    port (
      C, ALOAD : in  std_logic;
      D        : in  std_logic_vector(15 downto 0);
      SO       : out std_logic);
  end component;
  
  -- Receive Register (AU_CDTO)
  component shift_in is
    port (
      SI : in  std_logic;
      C  : in  std_logic;
      D  : out std_logic_vector(7 downto 0));
  end component;

  -- Output buffer for AU_CSN
  component OBUF_F_8
    port (
      O : out std_ulogic;               -- Out to Audio codec
      I : in  std_ulogic);              -- In from Audio codec
  end component;

  -- Tristate output buffer for AU_CCLK and AU_CDTI
  component IOBUF_F_8
    port (
      O : out std_ulogic;               -- Out from the buffer
      I : in  std_ulogic;               -- In to the buffer
      IO: inout std_logic;              -- In/Out to pin
      T : in  std_ulogic);              -- Active-low output enable
  end component;
  
begin  -- behavioral

  -- A / 6 Clock Divider to generate CCLK
  clkdiv: clock_divider
    generic map (FACTOR => 12)
    port map (OPB_Clk, int_au_csn, int_au_cclk);

  -- CCLK Counter
  ccounter: counter
    port map (int_au_cclk, int_au_cclk_count_rst, int_au_cclk_count);

  -- Transmit Register
  cdti_buff: shift_out
    port map (int_au_cclk, int_au_cdti_load, int_au_cdti_buff, int_au_cdti);
    
  -- Receive Register
  cdto_buff: shift_in
    port map (PB_D(2), int_au_cclk, int_au_cdto_buff);

  -- Output buffers for output signals
  au_csn_obuff: OBUF_F_8
    port map (
      O => AU_CSN,
      I => int_au_csn);

  au_cclk_iobuff: IOBUF_F_8
    port map (
      O  => null,
      I  => int_au_cclk,
      IO => PB_D(0),
      T  => int_au_csn);

  au_cdti_iobuff: IOBUF_F_8
    port map (
      O  => null,
      I  => int_au_cdti,
      IO => PB_D(1),
      T  => int_au_csn);

  dbus_iobuff_gen: for i in 3 to 15 generate
    dbus_iobuff : IOBUF_F_8
      port map (
        O  => null,
        I  => '0',
        IO => PB_D(i),
        T  => int_au_csn);
  end generate;
      
  end generate dbus_iobuff_gen;
  -- Produce chip select signal by decoding OPB address and checking OPB_select
  int_selected <=
    '1' when OPB_select = '1' and
        OPB_ABus(0 to C_OPB_AWIDTH-AU_AWIDTH-1) =
        C_BASEADDR(0 to C_OPB_AWIDTH-AU_AWIDTH-1)
    else '0';

  -- Unused outputs
  Sln_errAck  <= '0';
  Sln_retry   <= '0';
  Sln_DBus(0 to C_OPB_DWIDTH-AU_DWIDTH-1) <= (others => '0');

  -- Tie OPB_ABus and OPB_DBus to the AU_CDTI buffer
  int_au_cdti_buff(0) <= '1';           -- Set Op Code bits
  int_au_cdti_buff(1) <= '1';
  int_au_cdti_buff(2) <= NOT OPB_RNW;
  int_au_cdti_buff(3) <= OPB_ABus(31);  -- Set Register Address bits
  int_au_cdti_buff(4) <= OPB_ABus(30);
  int_au_cdti_buff(5) <= OPB_ABus(29);
  int_au_cdti_buff(6) <= OPB_ABus(28);
  int_au_cdti_buff(7) <= '0';
  int_au_cdti_buff(8) <= OPB_DBus(31);   -- Set Control Data bits
  int_au_cdti_buff(9) <= OPB_DBus(30);
  int_au_cdti_buff(10) <= OPB_DBus(29);
  int_au_cdti_buff(11) <= OPB_DBus(28);
  int_au_cdti_buff(12) <= OPB_DBus(27);
  int_au_cdti_buff(13) <= OPB_DBus(26);
  int_au_cdti_buff(14) <= OPB_DBus(25);
  int_au_cdti_buff(15) <= OPB_DBus(24);

  -- Tie the 0'th state bit to Sln_xferAck
  Sln_xferAck <= present_state(0);
  
  -- Process to qualify OPB output using int_output_enable
  register_opb_outputs: process (OPB_Rst, OPB_RNW, int_output_enable)
  begin
    if OPB_Rst = '1' then
      Sln_DBus(C_OPB_DWIDTH-AU_DWIDTH to C_OPB_DWIDTH-1) <= (others => '0');
    else
      if int_output_enable = '1' and OPB_RNW = '1' then
        Sln_DBus(C_OPB_DWIDTH-AU_DWIDTH to C_OPB_DWIDTH-1) <= int_au_cdto_buff;
      else
        Sln_DBus(C_OPB_DWIDTH-AU_DWIDTH to C_OPB_DWIDTH-1) <= (others => '0');
      end if;
    end if;
  end process register_opb_outputs;

  -- Sequential part of the FSM
  fsm_seq : process(OPB_Clk, OPB_Rst)
  begin
    if OPB_Rst = '1' then          -- Asynchronous reset to Idle state
      present_state <= Idle;
    elsif OPB_Clk'event and OPB_Clk = '1' then   -- Positive edge OPB_Clk
      present_state <= next_state;               -- advance to next state
    end if;
  end process fsm_seq;

  -- Combinational part of the FSM
  fsm_comb : process(OPB_Clk, present_state, int_selected, int_au_cclk_count)
  begin
  
    case present_state is
    
      when Idle =>
        Sln_toutSup <= '0';            -- Disable Supress-Timeout
        int_au_csn <= '1';             -- Deselect Audio Codec
        int_au_cdti_load <= '1';       -- Open input to CDTI shift reg
        int_output_enable <= '0';      -- Disable output to Sln_DBus
        int_au_cclk_count_rst <= '1';  -- Reset AU_CCLK counter
        if int_selected = '1' then
          next_state <= Transfer;
        else
          next_state <= Idle;
        end if;  
        
      when Transfer =>
          if int_selected = '1' then
            Sln_toutSup <= '1';            -- Supress OPB Timeout
            int_au_cdti_load <= '0';       -- Latch audio codec address/data from OPB
            int_au_cclk_count_rst <= '0';  -- Start counter

            if int_au_cclk_count = "10000" then  -- IF last (16th) CCLK cycle elapsed...
              int_au_csn <= '1';                -- Deselect audio codec
              int_output_enable <= '1';         -- Enable Sln_DBus output
              next_state <= Ack;                -- Ack transfer
            else                           -- ELSE...
              int_au_csn <= '0';           -- Keep audio codec selected
              int_output_enable <= '0';    -- Keep Sln_DBus output supressed
              next_state <= Transfer;      -- Continue transfering data to audio codec
            end if;

          else                      -- If deselected by OPB master => Idle
            Sln_toutSup <= '0';
            int_au_csn <= '1';
            int_au_cdti_load <= '1';
            int_output_enable <= '0';
            int_au_cclk_count_rst <= '1';
            next_state <= Idle;
          end if;
      
      when Ack =>                  -- Send ACK
        Sln_toutSup <= '1';        -- Keep timeout supressed
        int_au_csn <= '1';         -- Deselect audio codec
        int_au_cdti_load <= '0';
        int_au_cclk_count_rst <= '0';
        int_output_enable <= '1';  -- Enable OPB output
        next_state <= Idle;

      when others =>            -- ELSE => Idle
        Sln_toutSup <= '0';
        int_au_csn <= '1';
        int_au_cdti_load <= '1';
        int_output_enable <= '0';
        int_au_cclk_count_rst <= '1';
        next_state <= Idle;
        
    end case;
    
  end process fsm_comb;
  
end behavioral;
