/*-----------------------------
# CSEE 4840 Embedded System Design
#
# SOBA Server
#
#  Team Warriors: Avraham Shinnar  as1619@columbia.edu
#                 Benjamin Dweck  bjd2102@columbia.edu
#                 Oliver Irwin    omi3@columbia.edu
#                 Sean White      sw2061@columbia.edu
#
-----------------------------*/

#include "ether.h"

#include "xparameters.h"
#include "xbasic_types.h"
#include "xio.h"
#include "xintc_l.h"

/* function prototypes, internal helpers */
static BYTE send(WORD *data, WORD addr, WORD dmalen);
static BYTE transmit(WORD sendlen);

/* note that at one point we had lots more functions.  the code
   indicates where they were. Many of them were flattened for code
   space reasons (the compiler had some inlining issues, possibly due
   to out smashing the stack). */

/* static data */

/* stored without the one's complement */
/* note: ip checksum can actually be constant!! */
/* all the pseudo headers and most of the headers on the udp_checksum
   can be calculated in advance.  we just need to ask ethereal what it
   should be :).  since time_stamp and sequence number are increments,
   we can just add (with carries) these increments to the new initial
   values each new packet.  so all we need is the update stuff in append. */
/* static unsigned long udp_checksum; */

/* the real note: udp checksum can actually be (and now is) 0.  This
   means that it is ignored.  We originally wrote code to calculate a
   proper udp checksum (it is still in the code, just commented out).
   we later commented it out and decided to use 0 due to code space
   limitations.  (this also allowed us to put back in diagnostics).

   note to future people: smashing your stack is a bad idea.
 */
static WORD cur_payload_addr;

struct {
  WORD padding;
  WORD sequence_number;
  unsigned long time_stamp;
} dynamic_data;

/* #define UDP_INIT_CHECKSUM (~((WORD)(0xaa78))); */
/* #define UDP_INIT_CHECKSUM (~((WORD)(0xb15d))); */

static BYTE header[HEADER_SIZE] = {
  /* create the static portion of the packets */
  /* MAC header (14 bytes) */
  /*   Destination MAC address (6 bytes): fixed */
  /* 0-5 * 0x00,0x0C, 0xF1, 0x73, 0x4C, 0x9C, 
   */  
   /* Micro4 MAC */
     0x00,0x07,0xE9,0x43,0x8A,0x38,
  /*   Source MAC address (6 bytes): fixed */
  /* 6 */ 0x00,
  /* 7 */ 0x0D,
  /* 8 */ 0x60,
  /* 9 */ 0x7F,
  /* 10 */0xF9,
  /* 11 */0xAF,
  /*    Length (2 bytes): fixed */
  /* used as a type*/
  /* 12 */ 0x08,
  /* 13 */ 0x00,
  /*  IP header (20 bytes) */
  /*    Version (4 bits): fixed */
  /*    IHL (4 bits): fixed */
  /* 14 */ 0x45,
  /*    TOS (1 byte): fixed */
  /* 15 */ 0x00,
  /*    Total Length (2 bytes): fixed */
  /* 16 */ IP_LENGTH>>8,
  /* 17 */ IP_LENGTH & (0xff),
  /*    Identification (2 bytes): fixed */
  /* 18 */ 0x00,
  /* 19 */ 0x00,
  /*    Flags (3 bits): fixed */
  /*    Fragment Offset (13 bits): fixed */
  /* 20 */ 0x00,
  /* 21 */ 0x00,
  /*    TTL (1 byte): fixed */
  /* for now, this should die on contact*/
  /* 22 */ 0x04,
  /*    Protocol (1 byte): fixed */
  /* UDP=17 */
  /* 23 */ 0x11,
  /*    Header Checksum (2 bytes): updated. see update_ip_checksum */
   // Micro4
      0x8A,0x63,
  /*    Source IP (4 bytes): fixed */
  /* 26 */ 0x80,
  /* 27 */ 0x3B,
  /* 28 */ 0x95,
  /* 29 */ 0xA2,
  /*    Destination IP (4 bytes): fixed */
  // Micro4
      128,59,144,169,
  ///* 30 */ 128,
  ///* 31 */ 59, 
  ///* 32 */ 144,
  ///* 33 */ 168,
  /* UDP header (6 Bytes) */
  /*   Source Port (2 bytes): fixed */
  /* 34 */ 0x0B,
  /* 35 */ 0xE2,
  /*   Destination Port (2 bytes): fixed */
     /* in decimal: 3042 */
  /* 36 */ 0x0B,
  /* 37 */ 0xE2,
  /*   Length (2 bytes): fixed */
  /* 38 */ UDP_LENGTH>>8,
  /* 39 */ UDP_LENGTH & 0xff,
  /*   Checksum (2 bytes): updated */
  /* 40 */ 0,
  /* 41 */ 0,
  /* RTP header (12 bytes) */
  /*   Version (2 bits): fixed */
  /* 10 */
  /*   Padding (1 bit): fixed */
  /* 0 */
  /*   Extension (1 bit): fixed */
  /* 0 */
  /*   CSRC count (4 bits): fixed */
  /* 0 */
  /* 42 */ 0x80,
  /*   Marker bit (1 bit): fixed */
  /* 0 */
  /*   Payload type (7 bits): fixed */
  /* L16=10 (2 channel( (see
     http://www.networksorcery.com/enp/protocol/rtp.htm) */
  /* 43 */ 0x0a,
  /*   Sequence number (16 bits): updated */
  /* 44 */ 0x00,
  /* 45 */ 0x00,
  /*   Time stamp (32 bits): updated */
  /* 46 */ 0x00,
  /* 47 */ 0x00,
  /* 48 */ 0x00,
  /* 49 */ 0x00,
  /*   SSRC (32 bits): fixed */
  /* should be a random value: I don't think this is what is meant :)
     -- since we don't change our source transport address and don't
     handle multiple synchronization source within the same RTP
     channel, this should not be a problem. */
  /* 50 */ 0x42,
  /* 51 */ 0x42,
  /* 52 */ 0x42,
  /* 53 */ 0x42
  /*   CSRC list (0 bits): fixed */
  /* we are evil and don't give proper attribution to sources. */
};


/* a simple wait function that burns clock cycles. */
void wait(int mult){
  volatile int j=0, i=1000000;
  for(; i > 0; --i) {
    for(; mult > 0; --mult) {
      // a smart compiler with aggresive inliner should unroll this
      ++j;
    }
  }
}

/* 
   INITIALIZATION
*/

/* Diagnostics based on page 31 of the Ethernet Controller manual
   Write on two pages first and then read to avoid being fooled by
   data latched in both write and read.
*/

int diagnostics() {
  //  outnic(CMDR, 0x21);                // stop AX88796
  wait(10);
  
  outnic(CMDR,0x61);                
  outnic(PSTART, 0x4E);

  outnic(CMDR,0x21);                
  outnic(PSTOP, 0x3E);
  
  outnic(CMDR,0x61);   
  if(innic(PSTART) != 0x4e)
    return 1;

  outnic(CMDR,0x21);                 // switch to page 0
  if(innic(PSTOP) != 0x3e)
    return 2;

  if(innic(0x16*2) != 0x15)
    return 3;

  if(innic(0x12*2) != 0x0c)
    return 4;

  if(innic(0x13*2) != 0x12)
    return 5;
  
  return 0;
}

/* initializes the ethernet card and our data structures. sets up the
   static part of the packet header in memory. */
BYTE init() {
  int ret;

  outnic(NE_RESET, innic(NE_RESET));  // trigger a reset
  wait(2);
  if ((innic(ISR) & 0x80) == 0)       // Report if failed
    {
      print("  Ethernet card failed to reset!\r\n");
      print("Ethernet NIC not present or not initializing correctly\r\n");
      return 1;
    }
  else
    {
      dprint("Ethernet card reset successful!\r\n");
      /*      ether_reset(); */ /* Reset Ethernet card, */

      /* Write 21h to Command Register to abort current DMA operations. */
      outnic(CMDR, 0x21);
      /* Wait 2 milliseconds (timout for inter-frame gap timer) */
      wait(10);
      /* Write 01h to Data Control Register to enable 16-bit word transfers. */
      outnic(DCR, 0x01);
      /* Write 00h to both Remote Byte Count Registers to zero out DMA counter. */
      outnic(RBCR0, 0x00);
      outnic(RBCR1, 0x00);
      /* Write 00h to Interrupt Mask Register to mask interrupts. */
      outnic(IMR, 0x00);
      /* Write ffh to Interrupt Status Register to clear interrupt flags */
      outnic(ISR, 0xff);
      /* Write 20h to Receive Configuration Register to put NIC in monitor mode */
      outnic(RCR, 0x20);
      /* Write 02h to Transmit Configuration Register to put NIC in loop-back mode */
      outnic(TCR, 0x02);
      /* Set RX Start and Stop, Boundary, and TX start page. */
      outnic(PSTART, RXSTART);
      outnic(PSTOP, RXSTOP);
      outnic(BNRY, (BYTE)(RXSTOP-1));
      outnic(TPSR, TXSTART);
      /* Reset interrupt mask and flags. */
      outnic(ISR, 0xFF);   // clear interrupt status register
      outnic(IMR, 0x00);   // Mask completion irq
      /* Write 22h to Command Register to start NIC */
      outnic(CMDR, 0x22);
      /* Write 00h to Transmit Configuration Register to set normal transmit operation */
      outnic(TCR, 0x00);
      
      dprint("Ethernet card intialization complete!\r\n");
    }

  /* init_packet_setup(); */
  dynamic_data.sequence_number = 0;
  dynamic_data.time_stamp = 0;

/*   udp_checksum = UDP_INIT_CHECKSUM; */
  cur_payload_addr = PAYLOAD_START_ADDRESS;


  /* now lets initialize the buffers. */
  return send((WORD *)header, PACKET_START_ADDRESS, 54);//HEADER_SIZE);
}


/* actually transmits a (filled in) packet. */
static BYTE transmit(WORD sendlen)
{
  int j;
  outnic(TPSR, TXSTART); // set Transmit Page Start Register
  outnic(TBCR0, (sendlen&0xff)); // set Transmit Byte Count
  outnic(TBCR1, (sendlen>>8));
 
  outnic(CMDR, CR_COMPLETE|CR_TXP); // start transmission
  j=1000;
  while(j-->0 && !(innic(ISR) & ISR_PTX));
  if(!j){
    return 1;
  }

  outnic(ISR,0xFF);
  return 0;
}


/* low level function that does dma to send data to card. */
/* based on code from Josh's group. */
static BYTE send(WORD *data, WORD addr, WORD dmalen){
  int i, j, h;
  WORD counter;
  WORD word;

  counter = dmalen>>1;

  outnic(RSAR0, (addr&0xff)); // set DMA starting address
  outnic(RSAR1, (addr>>8));

  outnic(ISR, 0xFF); // clear ISR

  outnic(RBCR0, (dmalen&0xff)); // set Remote DMA Byte Count
  outnic(RBCR1, (dmalen>>8));

  outnic(CMDR, CR_WRITE|CR_START); // start the DMA write - 0x12

  // change order of MS/LS since DMA
  // writes LS byte in 15-8, and MS byte in 7-0
  for(i=0; i<counter; i++){
    word = (data[i]<<8)|(data[i]>>8);
    outnic(DATAPORT, word);
  }

  if(!(innic(ISR) & ISR_RDC)){
    //    print("Data - DMA did not finish\r\n");
    return 1;
  }

  outnic(CMDR,CR_COMPLETE); 

  return 0;
}

/* external interface */
/* takes a 2 word sample, and sends it out.  sends packet if needed */
/* 0 return is success */
BYTE output_sample(WORD *sample) {
  BYTE ret = 0;
/*   WORD check; */

  /*   ret = append(sample); */
   ret = send(sample, cur_payload_addr, 4);
  
   /* if(ret)
      return ret; */
  cur_payload_addr+=4;
  /* update checksums */

/*   udp_checksum += *(sample+1); */
/*   udp_checksum += *(sample); */
/*   while (udp_checksum>>16) */
/*     udp_checksum = (udp_checksum & 0xffff) + (udp_checksum >> 16); */

  if(cur_payload_addr >= PACKET_END_ADDRESS) {
    /* finalize_packet() */
    /* take the ones complement, as well as shoving it into a word */
/*     while (udp_checksum>>16) */
/*       udp_checksum = (udp_checksum & 0xffff) + (udp_checksum >> 16); */

/*     check = ~((WORD)(udp_checksum&0xffff)); */
/*     check = 0; */
/*     ret = send(&check, UDP_CHECKSUM_ADDRESS, 2); */
/*     if(ret) { */
/*       print("Error writing udp checksum\r\n"); */
/*       return ret; */
/*     } */
    ret = send(((WORD *)&dynamic_data)+1, RTP_SEQNUM_ADDRESS, 6);

    if(ret) {
      print("Error writing rtp header data\r\n");
    }
 
    if(ret) {
      print("Error finalizing packet. dropping.\r\n");
    } else {
      ret = transmit(PACKET_SIZE);
      //   print("probable success transmitting packet. happy! happy!\r\n");
    }

    /*     next_packet_setup(); */
    ++dynamic_data.sequence_number;
    dynamic_data.time_stamp += 20;

/*     udp_checksum = dynamic_data.sequence_number + (unsigned short)((dynamic_data.time_stamp) >> 16) + (unsigned short)((dynamic_data.time_stamp)&0xffff); */
/*     udp_checksum += UDP_INIT_CHECKSUM; */
/*     while (udp_checksum>>16) */
/*       udp_checksum = (udp_checksum & 0xffff) + (udp_checksum >> 16); */
    cur_payload_addr=PAYLOAD_START_ADDRESS;
  }
  
  return ret;
}
