/* * Device driver for the VGA video generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_ball.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree vga_ball.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "top_module.h"

#define DRIVER_NAME "top_module"

/* Device registers */
#define BALL_X(x) ((x))
#define BALL_Y(x) ((x) + 1)

/*
 * Information about our device
 */
struct vga_ball_dev
{
    struct resource res;    /* Resource: our registers */
    void __iomem *virtbase; /* Where registers can be accessed in memory */
    VGA_BALL_ARG vla;       /* Last parameter */
} dev;

// read data
static unsigned int camera_read(void)
{
	return ioread32((dev.virtbase));
}

static void write_ball_position(vga_ball_pos_t *position)
{
  //iowrite16(position->x, BALL_X(dev.virtbase));
  //iowrite16(position->y, BALL_Y(dev.virtbase));

    iowrite16(100, BALL_X(dev.virtbase));
    iowrite16(0, BALL_Y(dev.virtbase));

    dev.vla.position = *position;
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long top_module_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{

    VGA_BALL_ARG vla;
    unsigned int info;

    switch (cmd)
    {
    case VGA_BALL_WRITE:
	
        if (copy_from_user(&vla, (VGA_BALL_ARG *)arg,
                           sizeof(VGA_BALL_ARG)))
            return -EACCES;
        write_ball_position(&vla.position);

        break;

    case VGA_BALL_READ:
      // vla = dev.vla;
        if (copy_to_user((VGA_BALL_ARG *)arg, &dev.vla,
                         sizeof(VGA_BALL_ARG)))

            return -EACCES;
        break;

    case CAMERA_READ:

        info = camera_read();

        if (copy_to_user((unsigned int *)arg, &info,
                         sizeof(unsigned int)))
            return -EACCES;
        break;

    default:
        return -EINVAL;
    }

    return 0;
}

/* The operations our device knows how to do */
static const struct file_operations top_module_fops = {
    .owner = THIS_MODULE,
    .unlocked_ioctl = top_module_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice top_module_misc_device = {
    .minor = MISC_DYNAMIC_MINOR,
    .name = DRIVER_NAME,
    .fops = &top_module_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init top_module_probe(struct platform_device *pdev)
{
    int ret;

    /* Register ourselves as a misc device: creates /dev/top_module */
    ret = misc_register(&top_module_misc_device);

    /* Get the address of our registers from the device tree */
    ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
    if (ret)
    {
        ret = -ENOENT;
        goto out_deregister;
    }

    /* Make sure we can use these registers */
    if (request_mem_region(dev.res.start, resource_size(&dev.res),
                           DRIVER_NAME) == NULL)
    {
        ret = -EBUSY;
        goto out_deregister;
    }

    /* Arrange access to our registers */
    dev.virtbase = of_iomap(pdev->dev.of_node, 0);
    if (dev.virtbase == NULL)
    {
        ret = -ENOMEM;
        goto out_release_mem_region;
    }


    return 0;

out_release_mem_region:
    release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
    misc_deregister(&top_module_misc_device);
    return ret;
}

/* Clean-up code: release resources */
static int top_module_remove(struct platform_device *pdev)
{
    iounmap(dev.virtbase);
    release_mem_region(dev.res.start, resource_size(&dev.res));
    misc_deregister(&top_module_misc_device);
    return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id top_module_of_match[] = {
    {.compatible = "csee4840,top_module-1.0"},
    {},
};
MODULE_DEVICE_TABLE(of, top_module_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver top_module_driver = {
    .driver = {
        .name = DRIVER_NAME,
        .owner = THIS_MODULE,
        .of_match_table = of_match_ptr(top_module_of_match),
    },
    .remove = __exit_p(top_module_remove),
};

/* Called when the module is loaded: set things up */
static int __init top_module_init(void)
{
    pr_info(DRIVER_NAME ": init\n");
    return platform_driver_probe(&top_module_driver, top_module_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit top_module_exit(void)
{
    platform_driver_unregister(&top_module_driver);
    pr_info(DRIVER_NAME ": exit\n");
}

module_init(top_module_init);
module_exit(top_module_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("VGA ball driver");
