module button_press (input logic[25:0] horp_time,
                     input logic[25:0] holdup_time,
                     input logic button,
    			     input logic clk, //50 MHz Clock input
                     output logic hold,
                     output logic holdup,
                     output logic letgo
   			     ); 
						  
	logic [25:0] cnt;
    enum logic [1:0] {letgo_s, init, horp, hold_s} state;
	
	always_ff@(posedge clk) begin
        case (state)
            init: begin state <= letgo_s; end
            
            horp: begin
                if (button == 1'b1) begin
                    state <= letgo_s;
                    cnt <= 0;
                end  
                else if (button == 1'b0 && cnt >= horp_time) begin
                    state <= hold_s;
                    cnt <= 0;
                end
                else begin
                    state <= horp;
                    cnt <= cnt + 1'b1;
                end
            end

            hold_s: begin
                if (button == 1'b1) begin
                    state <= letgo_s;
                    cnt <= 0;
                end  
                else if (button == 1'b0 && cnt >= holdup_time) begin
                    state <= hold_s;
                    cnt <= 0;
                end
                else begin
                    cnt <= cnt + 1'b1;
                    state <= hold_s;
                end
            end

            letgo_s: begin  
                if (button == 1'b0) begin
                    state <= horp;
                    cnt <= 0;
                end
                else  state <= letgo_s;
            end 
            default: state <= letgo_s;
        endcase
    end
	
	always_comb begin
        case (state)
            init: begin hold = 0; holdup = 0; letgo = 0; end  
            horp: begin
                if (cnt >= horp_time) begin hold = 1; holdup = 0; letgo = 0; end
                else begin hold = 0; holdup = 0; letgo = 0;  end 
            end 
            hold_s: begin
                if (cnt >= holdup_time) begin hold = 1; holdup = 1; letgo = 0; end
                else begin hold = 1; holdup = 0; letgo = 0;  end 
            end
            letgo_s: begin hold = 0; holdup = 0; letgo = 1; end
            default: begin hold = 0; holdup = 0; letgo = 1; end
        endcase
	end
	  
endmodule


module switch_db    (input logic[25:0] db_time,
                     input logic switch,
    			     input logic clk, //50 MHz Clock input
				     output logic sw
				     ); 
						  
	logic [25:0] cnt;
    logic prev_switch;
    enum logic {letgo_s, debounce} state;

	
	always_ff@(posedge clk) begin
        case (state)
            debounce: begin
                if (cnt == db_time) begin
                    state <= letgo_s;
                    cnt <= 0; 
                end
                else begin
                    cnt <= cnt + 1'b1;
                end
            end
            
            letgo_s: begin  
                if (switch != prev_switch) begin
                    state <= debounce;
                    prev_switch <= switch;
                    cnt <= 0;
                end
                else  begin state <= letgo_s; prev_switch <= switch; end
            end 
            default: state <= letgo_s;
        endcase
    end
	
	always_comb begin
        case (state)
            debounce: begin 
                if (cnt == db_time) begin sw = prev_switch; end
                else begin  sw = ~prev_switch;  end 
            end 
            letgo_s: begin   sw = prev_switch; end
            default: begin sw = prev_switch; end
        endcase
	end
	  
endmodule





module hex7seg(input logic  [11:0] a,
	           output logic [6:0] x,
               output logic [6:0] y,
               output logic [6:0] z
               );
            logic [11:0] tens;
            logic blank;
            
            hex7seg_tens tens_place(.a(tens), .blank(blank), .y(y), .z(z));

			 always_comb begin
                if (a == 12'd300) begin
                    x = 7'b011_0000;
                    tens = a - 12'd300;
                    blank =1'b0;
                end
                else if (a >= 12'd200 && a < 12'd300) begin
                    x = 7'b010_0100;
                    tens = a - 12'd200;
                    blank = 1'b0;
                end
                else if (a >= 12'd100 && a < 12'd200) begin
                    x = 7'b111_1001;
                    tens = a - 12'd100;
                    blank = 1'b0;
                end
                else begin
                    x = 7'b111_1111;
                    tens = a;
                    blank = 1'b1;
                end	   
            end
endmodule



module hex7seg_tens(input logic  [11:0] a,
               input logic blank,
               output logic [6:0] y,
               output logic [6:0] z
               );
            logic [11:0] ones;
            
            hex7seg_ones ones_place(.a(ones), .z(z));

			always_comb begin
                if (a >= 12'd90 && a < 12'd100) begin
                    y = 7'b001_0000;
                    ones = a - 12'd90;
                end
                else if (a >= 12'd80 && a < 12'd90) begin
                    y = 7'b000_0000;
                    ones = a - 12'd80;
                end
                else if (a >= 12'd70 && a < 12'd80) begin
                    y = 7'b111_1000;
                    ones = a - 12'd70;
                end
                else if (a >= 12'd60 && a < 12'd70) begin
                    y = 7'b000_0010;
                    ones = a - 12'd60;
                end
                else if (a >= 12'd50 && a < 12'd60) begin
                    y = 7'b001_0010;
                    ones = a - 12'd50;
                end
                else if (a >= 12'd40 && a < 12'd50) begin
                    y = 7'b001_1001;
                    ones = a - 12'd40;
                end
                else if (a >= 12'd30 && a < 12'd40) begin
                    y = 7'b011_0000;
                    ones = a - 12'd30;
                end
                else if (a >= 12'd20 && a < 12'd30) begin
                    y = 7'b010_0100;
                    ones = a - 12'd20;
                end
                else if (a >= 12'd10 && a < 12'd20) begin
                    y = 7'b111_1001;
                    ones = a - 12'd10;
                end
                else begin
                    ones = a;
                    if (blank) begin
                        y = 7'b111_1111;
                    end
                    else begin
                        y = 7'b100_0000;
                    end
                end
            end
endmodule

module hex7seg_ones(input logic  [11:0] a,
               output logic [6:0] z
               );
            
			always_comb begin
                case(a)
				  12'd0:    z = 7'b100_0000;
				  12'd1:    z = 7'b111_1001;
				  12'd2:    z = 7'b010_0100;
				  12'd3:    z = 7'b011_0000;
				  12'd4:    z = 7'b001_1001;
				  12'd5:    z = 7'b001_0010;
				  12'd6:    z = 7'b000_0010;
				  12'd7:    z = 7'b111_1000;
				  12'd8:    z = 7'b000_0000;
				  12'd9:    z = 7'b001_0000;
			      default: z = 7'b111_1111;
				endcase
            end

endmodule



module user_interface( 
         input logic        CLOCK_50,  // 50 MHz Clock input
	     
	     input logic [3:0] 	KEY, // Pushbuttons; KEY[0] is rightmost

	     input logic [9:0] 	SW, // Switches; SW[0] is rightmost

         input logic        reset,
            
         input logic        address,

         input logic        read,

         input logic        chipselect,

	     // 7-segment LED displays; HEX0 is rightmost
	     output logic [6:0] HEX0, HEX1, HEX2, HEX3, HEX4, HEX5,

	     output logic [9:0] LEDR, // LEDs above the switches; LED[0] on right
        
         output logic [15:0] readdata
	     );

    logic 			clk;
    logic [25:0] db_time;
    logic [25:0] horp_time;
    logic [25:0] holdup_time;
    logic hold0, holdup0, letgo0;
    logic hold1, holdup1, letgo1;
    logic hold2, holdup2, letgo2;
    logic hold3, holdup3, letgo3;
    logic sw9, sw8;
    assign db_time = 26'd1_000_000; //20 ms
    assign horp_time = 26'd50_000_000; //1 s
    assign holdup_time = 26'd2_500_000; //50 ms
    assign clk = CLOCK_50;
 
    button_press b3(.horp_time(horp_time), .holdup_time(holdup_time),
                    .button(KEY[3]),
                    .clk(CLOCK_50),
                    .hold(hold3), .holdup(holdup3), .letgo(letgo3)
                    );
    button_press b2(.horp_time(horp_time), .holdup_time(holdup_time),
                    .button(KEY[2]),
                    .clk(CLOCK_50),
                    .hold(hold2), .holdup(holdup2), .letgo(letgo2)
                    );

    button_press b1(.horp_time(horp_time), .holdup_time(holdup_time),
                    .button(KEY[1]),
                    .clk(CLOCK_50),
                    .hold(hold1), .holdup(holdup1), .letgo(letgo1)
                    );

    button_press b0(.horp_time(horp_time), .holdup_time(holdup_time),
                    .button(KEY[0]),
                    .clk(CLOCK_50),
                    .hold(hold0), .holdup(holdup0), .letgo(letgo0)
                    );
    switch_db s9(.db_time(db_time),
                 .switch(SW[9]),
                 .clk(CLOCK_50),
                 .sw(sw9)
                 );
    switch_db s8(.db_time(db_time),
                 .switch(SW[8]),
                 .clk(CLOCK_50),
                 .sw(sw8)
                 );
   			
    enum logic [2:0] {init, playback, bpm, step, button_step, button_bpm, button_track, dummy} state, prev_state;
    logic [15:0] bpm_count;
    logic [7:0]  step_count;
    logic [7:0]  track_count;
    logic        playback_mode;
    logic [11:0] left_hex;
    logic [11:0] right_hex;
    logic add;

    assign LEDR[4] = state == button_step;
    assign LEDR[5] = state == button_bpm;
    assign LEDR[6] = state == button_track;
    assign LEDR[7] = state == dummy;


    hex7seg bpm_press  (.a(left_hex),  .x(HEX5), .y(HEX4), .z(HEX3));
    hex7seg step_press (.a(right_hex), .x(HEX2), .y(HEX1), .z(HEX0));

    always_ff @(posedge clk)
    begin 
        if (reset) begin
            bpm_count <= 16'd100; 
            step_count <= 8'd1;
            track_count <= 8'd1;
            state <= init;
            prev_state <= init;
        end
                
        case (state)
        init: begin
            bpm_count <= 16'd100; step_count <= 8'd1; track_count <= 8'd1;
            if (sw9) begin state <= playback; end
            else if (!sw9 && sw8) state <= bpm;
            else if (!sw9 && !sw8) state <= step;
            else state <= step;
        end
        playback: begin
            if (sw9) state <= playback;
            else if (!sw9 && sw8) state <= bpm;
            else if (!sw9 && !sw8) state <= step;
            else state <= playback; 
        end
        bpm: begin 
            if (KEY[1] == 1'b0) begin state <= button_bpm; prev_state <= bpm; add = 1; end
            else if (KEY[0] == 1'b0) begin state <= button_bpm; prev_state <= bpm; add = 0; end 
            else if (sw9) begin state <= playback; end
            else if (!sw9 && sw8) state <= bpm;
            else if (!sw9 && !sw8) state <= step;
            else state <= bpm; 
        end
        step: begin
            if (KEY[3] == 1'b0) begin state <= button_step; prev_state <= step; add <= 1; end
            else if (KEY[2] == 1'b0) begin state <= button_step; prev_state <= step; add <= 0; end 
            else if (KEY[1] == 1'b0) begin state <= button_track; prev_state <= step; add <= 1; end
            else if (KEY[0] == 1'b0) begin state <= button_track; prev_state <= step; add <= 0; end 
            else if (sw9) state <= playback;
            else if (!sw9 && sw8) state <= bpm;
            else if (!sw9 && !sw8) state <= step;
            else state <= step;
        end
        button_step: begin
            //add or sub only once (check if prev state is step)
            if (prev_state == step) begin
                prev_state <= button_step;
                state <= button_step;
                
                if (add == 1) begin
                    if (step_count == 8'd16) step_count <= 8'd1;
                    else                     step_count <= step_count + 8'd1;
                end
                else begin
                    if (step_count == 8'd1)   step_count <= 8'd16;
                    else                      step_count <= step_count - 8'd1;
                end
            end
            else begin
                prev_state <= button_step;
                state <= button_step;
                if ((letgo3 && add == 1) || (letgo2 && add == 0)) begin
                    state <= step;
                end 
            end

        end 
        button_bpm: begin
            //if prev state was bpm (just once before holding) or hold and
            //holdup and prev state == button_bpm, add or sub
            if ((prev_state == bpm) || 
                (hold1 && holdup1 && prev_state == button_bpm && add == 1) ||
                (hold0 && holdup0 && prev_state == button_bpm && add == 0))  begin 
                
                prev_state <= button_bpm;
                state <= button_bpm;
               
                if (add == 1) begin
                    if (bpm_count == 16'd300) bpm_count <= 16'd40;
                    else                      bpm_count <= bpm_count + 16'd1;
                end
                else begin
                    if (bpm_count == 16'd40) bpm_count <= 16'd300;
                    else                     bpm_count <= bpm_count - 16'd1;
                end
            end
            else begin
                prev_state <= button_bpm;
                state <= button_bpm;
                if ((letgo1 && add == 1) || (letgo0 && add == 0)) begin
                    state <= bpm;
                end
            end
        end        

        button_track: begin
            //add or sub only once (check if prev state is step)
            if (prev_state == step) begin
                prev_state <= button_track;
                state <= button_track;
                
                if (add == 1) begin
                    if (track_count == 8'd4)  track_count <= 8'd1;
                    else                      track_count <= track_count + 8'd1;
                end
                else begin
                    if (track_count == 8'd1)   track_count <= 8'd4;
                    else                       track_count <= track_count - 8'd1;
                end
            end
            else begin
                prev_state <= button_track;
                state <= button_track;
                if ((letgo1 && add == 1) || (letgo0 && add == 0)) begin
                    state <= step;
                end 
            end
        end 
        
        dummy:   state <= init;        
        default: state <= init;
        endcase
	end	
	
	always_comb 
	begin
        if (reset) begin
            readdata[15:0] = 16'b0;
            playback_mode = 1'b0;
            left_hex = {4'b0, step_count[7:0]};   
            right_hex = {4'b0, track_count[7:0]};  
        end

        else if (chipselect && read) begin
            case (address)
                1'b0: begin readdata[15:0] = {playback_mode, bpm_count[14:0]}; end
                1'b1: begin readdata[15:0] = {track_count, step_count}; end
                default: readdata[15:0] = 16'b0;
            endcase

            case (state)
                init:           begin left_hex = {4'b0, step_count[7:0]};   right_hex = {4'b0, track_count[7:0]}; playback_mode = 1'b0; end
                playback:       begin left_hex = {4'b0, step_count[7:0]};   right_hex = {4'b0, track_count[7:0]}; playback_mode = 1'b1; end
                bpm:            begin left_hex = 12'd400;                   right_hex = bpm_count[11:0];          playback_mode = 1'b0; end
                step:           begin left_hex = {4'b0, step_count[7:0]};   right_hex = {4'b0, track_count[7:0]}; playback_mode = 1'b0; end
                button_step:    begin left_hex = {4'b0, step_count[7:0]};   right_hex = {4'b0, track_count[7:0]}; playback_mode = 1'b0; end
                button_bpm:     begin left_hex = 12'd400;                   right_hex = bpm_count[11:0];          playback_mode = 1'b0; end
                button_track:   begin left_hex = {4'b0, step_count[7:0]};   right_hex = {4'b0, track_count[7:0]}; playback_mode = 1'b0; end
                dummy:          begin left_hex = 12'd400;                   right_hex = 12'd400;                  playback_mode = 1'b0; end
                default:        begin left_hex = {4'b0, step_count[7:0]};   right_hex = {4'b0, track_count[7:0]}; playback_mode = 1'b0; end
            endcase
	    end

        else begin
            readdata[15:0] = 16'b0;
            case (state)
                init:           begin left_hex = {4'b0, step_count[7:0]};   right_hex = {4'b0, track_count[7:0]}; playback_mode = 1'b0; end
                playback:       begin left_hex = {4'b0, step_count[7:0]};   right_hex = {4'b0, track_count[7:0]}; playback_mode = 1'b1; end
                bpm:            begin left_hex = 12'd400;                   right_hex = bpm_count[11:0];          playback_mode = 1'b0; end
                step:           begin left_hex = {4'b0, step_count[7:0]};   right_hex = {4'b0, track_count[7:0]}; playback_mode = 1'b0; end
                button_step:    begin left_hex = {4'b0, step_count[7:0]};   right_hex = {4'b0, track_count[7:0]}; playback_mode = 1'b0; end
                button_bpm:     begin left_hex = 12'd400;                   right_hex = bpm_count[11:0];          playback_mode = 1'b0; end
                button_track:   begin left_hex = {4'b0, step_count[7:0]};   right_hex = {4'b0, track_count[7:0]}; playback_mode = 1'b0; end
                dummy:          begin left_hex = 12'd400;                   right_hex = 12'd400;                  playback_mode = 1'b0; end
                default:        begin left_hex = {4'b0, step_count[7:0]};   right_hex = {4'b0, track_count[7:0]}; playback_mode = 1'b0; end
            endcase
        end
    end

endmodule 
