/* game_logic.c
 *
 * Runs the main game logic, on the FPGA or in emulation
 *
 * Patrick Cronin, Dan Ivanovich, & Kiryl Beliauski
 * Columbia University CSEE 4840 - Embedded Systems
 */

#include "colors.h"
#include "fbputchar.h"
#include "global_consts.h"
#include "guitar_reader.h"
#include "guitar_state.h"
#include "helpers.h"
#include "song_data.h"
#include "sprites.h"
#include "vga_emulator.h"
#include "vga_framebuffer.h"

#include <SDL2/SDL_blendmode.h>
#include <fcntl.h>
#include <linux/fb.h>
#include <math.h>
#include <ncurses.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

// Set this to 1 to emulate the game. Will not attempt to connect to VGA/drivers
int EMULATING_VGA = 1;

int SCREEN_LINE_LENGTH;
int vga_framebuffer_fd, guitar_fd;
unsigned char *framebuffer;
pthread_mutex_t framebuffer_mutex = PTHREAD_MUTEX_INITIALIZER,
                controller_mutex = PTHREAD_MUTEX_INITIALIZER;
guitar_state controller_state;

struct {
  int green;
  int red;
  int yellow;
  int blue;
  int orange;
} color_cols_x = {15, 45, 75, 105, 135};

void *update_guitar_state(void *arg) {
  (void)arg; // Suppress unused warning

  while (1) {
    char *guitar_hex = read_note(guitar_fd);
    char *binary_string = hex_string_to_binary(guitar_hex);

    guitar_state note;
    set_note_guitar(&note, binary_string);

    pthread_mutex_lock(&controller_mutex);
    controller_state = note;
    pthread_mutex_unlock(&controller_mutex);

    usleep(16667); // 60 Hz refresh rate
  }
  return NULL;
}

void *update_framebuffer(void *arg) {
  (void)arg; // Suppress warning

  while (1) {
    pthread_mutex_lock(&framebuffer_mutex);
    for (int pixel_row = 0; pixel_row < WINDOW_HEIGHT; pixel_row++) {
      for (int pixel_col = 0; pixel_col < WINDOW_WIDTH; pixel_col++) {
        unsigned char *pixel =
            framebuffer + (pixel_row * WINDOW_WIDTH + pixel_col) * 4;
        RGB pixel_rgb = {pixel[2], pixel[1], pixel[0]};

        vga_framebuffer_arg_t vfba;
        vfba.pixel_writedata = pixel_writedata(get_color_from_rgb(pixel_rgb),
                                               pixel_row, pixel_col);

        if (ioctl(vga_framebuffer_fd, VGA_FRAMEBUFFER_UPDATE, &vfba)) {
          perror("ioctl(VGA_FRAMEBUFFER_UPDATE) failed");
        }
      }
    }
    pthread_mutex_unlock(&framebuffer_mutex);
  }
  return NULL;
}

void set_note(note_row *note_state, const char *binary_string) {
  if (note_state == NULL || binary_string == NULL) {
    return; // Error handling: Ensure note_state and binary_string are not NULL
  }

  // Convert the binary string to integer values
  int green = binary_string[7] - '0';
  int red = binary_string[6] - '0';
  int yellow = binary_string[5] - '0';
  int blue = binary_string[4] - '0';
  int orange = binary_string[3] - '0';

  // Assign the values to the struct fields
  note_state->green = green;
  note_state->red = red;
  note_state->yellow = yellow;
  note_state->blue = blue;
  note_state->orange = orange;
}

int hit_notes(guitar_state controller_state, note_row notes) {
  return controller_state.green == notes.green &&
         controller_state.red == notes.red &&
         controller_state.yellow == notes.yellow &&
         controller_state.blue == notes.blue &&
         controller_state.orange == notes.orange;
}

int main() {
  // Color definitions (hardcoded).
  // Inspired by https://oaksstudio.itch.io/guitarheroui, recreated from scratch
  circle_colors green_colors = {.white = palette[WHITE],
                                .light_gray = palette[LIGHT_GREEN],
                                .middle_gray = palette[MIDDLE_GREEN],
                                .dark_gray = palette[DARK_GREEN]};

  circle_colors red_colors = {.white = palette[WHITE],
                              .light_gray = palette[LIGHT_RED],
                              .middle_gray = palette[MIDDLE_RED],
                              .dark_gray = palette[DARK_RED]};

  circle_colors yellow_colors = {.white = palette[WHITE],
                                 .light_gray = palette[LIGHT_YELLOW],
                                 .middle_gray = palette[MIDDLE_YELLOW],
                                 .dark_gray = palette[DARK_YELLOW]};

  circle_colors blue_colors = {.white = palette[WHITE],
                               .light_gray = palette[LIGHT_BLUE],
                               .middle_gray = palette[MIDDLE_BLUE],
                               .dark_gray = palette[DARK_BLUE]};

  circle_colors orange_colors = {.white = palette[WHITE],
                                 .light_gray = palette[LIGHT_ORANGE],
                                 .middle_gray = palette[MIDDLE_ORANGE],
                                 .dark_gray = palette[DARK_ORANGE]};
  // 32 bits/pixel = 4 B/pixel
  unsigned char *next_frame;
  pthread_t fb_update_thread, guitar_thread;
  VGAEmulator emulator;

  init_guitar_state(&controller_state);

  if (EMULATING_VGA) {
    printf("Running in VGA EMULATION MODE\n");
  }

  if ((framebuffer = malloc(WINDOW_WIDTH * WINDOW_HEIGHT * 4)) == NULL) {
    perror("Error allocating framebuffer!\n");
    return 1;
  }

  SCREEN_LINE_LENGTH = WINDOW_WIDTH * 4;

  if ((next_frame = malloc(WINDOW_WIDTH * WINDOW_HEIGHT * 4)) == NULL) {
    perror("Error allocating next_frame!\n");
    return 1;
  }

  // Set black background by default
  memset(framebuffer, 0, WINDOW_WIDTH * WINDOW_HEIGHT * 4);
  // Load necessary sprites into memory
  sprite GH_circle_base = load_sprite("sprites/GH-Circle.png");
  sprite GH_logo = load_sprite("sprites/GH-Logo.png");
  // Generate the sprites for the notes
  generated_circles note_circles =
      generate_circles(GH_circle_base, green_colors, red_colors, yellow_colors,
                       blue_colors, orange_colors);
  // Generate the sprites for the indicators to play:
  green_colors.white = BACKGROUND_COLOR;
  red_colors.white = BACKGROUND_COLOR;
  yellow_colors.white = BACKGROUND_COLOR;
  blue_colors.white = BACKGROUND_COLOR;
  orange_colors.white = BACKGROUND_COLOR;
  generated_circles play_circles_released =
      generate_circles(GH_circle_base, green_colors, red_colors, yellow_colors,
                       blue_colors, orange_colors);
  green_colors.white = green_colors.dark_gray;
  red_colors.white = red_colors.dark_gray;
  yellow_colors.white = yellow_colors.dark_gray;
  blue_colors.white = blue_colors.dark_gray;
  orange_colors.white = orange_colors.dark_gray;
  generated_circles play_circles_held =
      generate_circles(GH_circle_base, green_colors, red_colors, yellow_colors,
                       blue_colors, orange_colors);

  // Set up VGA emulator. Requires libsdl2-dev
  if (EMULATING_VGA) {
    if (VGAEmulator_init(&emulator, framebuffer, &controller_state))
      return 1;
  } else {
    // Set up VGA framebuffer connection
    if ((vga_framebuffer_fd = open("/dev/vga_framebuffer", O_WRONLY)) == -1) {
      perror("could not open /dev/vga_framebuffer\n");
      return -1;
    }

    if ((guitar_fd = open("/dev/note_reader", O_RDONLY)) == -1) {
      perror("could not open /dev/note_reader\n");
      return -1;
    }

    if (pthread_create(&fb_update_thread, NULL, &update_framebuffer, NULL) !=
        0) {
      perror("pthread_create(fb_update_thread) failed\n");
      return 1;
    }

    if (pthread_create(&guitar_thread, NULL, update_guitar_state, NULL) != 0) {
      perror("pthread_create(fb_update_thread) failed\n");
      return 1;
    }
  }

  note_row song_rows[NUM_NOTE_ROWS];

  char line[9]; // Buffer to store each line (8 characters + null terminator)
  FILE *file = fopen("single_note_commaless.txt", "r");

  int i = 0;
  while (fgets(line, sizeof(line), file) != NULL) {
    // Remove the newline character if present
    if (line[strlen(line) - 1] == '\n') {
      line[strlen(line) - 1] = '\0';
    }
    if (strlen(line) == 8) {
      // printf("note: %s\n", line);
      note_row note_row;
      set_note(&note_row, line);
      song_rows[i++] = note_row;
    }
  }

  fclose(file);

  int current_bottom_row_idx = 0, num_note_rows = 100;
  double current_bottom_row_Y = 0;
  int note_duration = round((60.0 / SONG_BPM) / NOTES_PER_MEASURE * 1000);

  // The Y coordinate of the middle of the guitar state line
  int guitar_state_line_Y = WINDOW_HEIGHT - 24;
  // How many pixels of "margin" (top and bottom) to apply to each note
  int note_row_veritcal_padding = 12;
  // THe total height including the 24x24 px sprite and the margin
  int note_height_px = 24 + 2 * note_row_veritcal_padding;
  // How many pixels each note row has to move down the screen in one ms
  double note_row_pixels_per_ms = (double)(note_height_px) / note_duration;

  printf("---SONG INFORMATION---\n");
  printf("BPM: %d\n", SONG_BPM);
  printf("Beat duration: %dms\n", note_duration);
  printf("Note row pixels/ms: %f\n", note_row_pixels_per_ms);

  // Start menu
  pthread_mutex_lock(&framebuffer_mutex);
  draw_sprite(GH_logo, framebuffer, WINDOW_WIDTH / 2, 192);

  fbputs(framebuffer, "Press", 8, 2, palette[WHITE]);
  fbputs(framebuffer, "Green", 9, 2, palette[GREEN]);
  fbputs(framebuffer, "To Play", 10, 1, palette[WHITE]);

  pthread_mutex_unlock(&framebuffer_mutex);

  while (1) {
    // Press green to continue
    pthread_mutex_unlock(&controller_mutex);
    if (controller_state.green) {
      pthread_mutex_unlock(&controller_mutex);
      break;
    }
    pthread_mutex_unlock(&controller_mutex);

    usleep(1000);
  }

  long long song_start_time = current_time_in_ms();
  long long last_draw_time = song_start_time;
  long long last_score_adjust_time = song_start_time;

  // For accuracy calculation
  int num_strummed = 0, num_hit = 0;
  double current_accuracy = 0;

  while (1) {
    // Fresh start
    memset(next_frame, 0, WINDOW_WIDTH * WINDOW_HEIGHT * 4);

    long long time_delta = current_time_in_ms() - last_draw_time;
    last_draw_time = current_time_in_ms();

    for (int row_on_screen = 0;
         row_on_screen < WINDOW_HEIGHT / note_height_px + 1; row_on_screen++) {
      if (current_bottom_row_idx + row_on_screen >= num_note_rows)
        break; // We've run out of notes

      note_row row = song_rows[current_bottom_row_idx + row_on_screen];

      int row_y = round(current_bottom_row_Y - note_height_px * row_on_screen);

      if (row.green)
        draw_sprite(note_circles.green, next_frame, color_cols_x.green, row_y);
      if (row.red)
        draw_sprite(note_circles.red, next_frame, color_cols_x.red, row_y);
      if (row.yellow)
        draw_sprite(note_circles.yellow, next_frame, color_cols_x.yellow,
                    row_y);
      if (row.blue)
        draw_sprite(note_circles.blue, next_frame, color_cols_x.blue, row_y);
      if (row.orange)
        draw_sprite(note_circles.orange, next_frame, color_cols_x.orange,
                    row_y);
    }

    current_bottom_row_Y += note_row_pixels_per_ms * time_delta;

    pthread_mutex_lock(&controller_mutex);
    if (controller_state.strum) {
      long long strum_time = current_time_in_ms();

      // Is the bottom note in a playable range, and did we try?
      if (current_bottom_row_Y <= guitar_state_line_Y + 12 &&
          current_bottom_row_Y >= guitar_state_line_Y - 12)
        if (hit_notes(controller_state, song_rows[current_bottom_row_idx])) {
          // We hit the note!
          if (strum_time - last_score_adjust_time <= 150) {
            // We barely missed the previous note. Undo that
            num_strummed--;
          }
          current_bottom_row_idx++;
          current_bottom_row_Y -= (24 + 2 * note_row_veritcal_padding);

          num_strummed++;
          num_hit++;
          last_score_adjust_time = current_time_in_ms();
          goto accuracy_adjust;
        }

      // We missed
      if (strum_time - last_score_adjust_time <= 150)
        // This is the same strum as last time, don't punish them again
        goto accuracy_adjust;

      num_strummed++;
      last_score_adjust_time = current_time_in_ms();

    accuracy_adjust:
      current_accuracy = (num_hit * 100.0) / num_strummed;
    }

    // Draw the Guitar state line
    draw_sprite(controller_state.green ? play_circles_held.green
                                       : play_circles_released.green,
                next_frame, color_cols_x.green, guitar_state_line_Y);
    draw_sprite(controller_state.red ? play_circles_held.red
                                     : play_circles_released.red,
                next_frame, color_cols_x.red, guitar_state_line_Y);
    draw_sprite(controller_state.yellow ? play_circles_held.yellow
                                        : play_circles_released.yellow,
                next_frame, color_cols_x.yellow, guitar_state_line_Y);
    draw_sprite(controller_state.blue ? play_circles_held.blue
                                      : play_circles_released.blue,
                next_frame, color_cols_x.blue, guitar_state_line_Y);
    draw_sprite(controller_state.orange ? play_circles_held.orange
                                        : play_circles_released.orange,
                next_frame, color_cols_x.orange, guitar_state_line_Y);
    pthread_mutex_unlock(&controller_mutex);

    // Score bar
    memset(next_frame, 0, WINDOW_WIDTH * 4 * 31);
    char score_line[11];
    if ((int)round(current_accuracy) < 10)
      sprintf(score_line, "Score: %d%%", (int)round(current_accuracy));
    else if ((int)round(current_accuracy) < 100)
      sprintf(score_line, "Score:%d%%", (int)round(current_accuracy));
    else
      // No colon
      sprintf(score_line, "Score%d%%", (int)round(current_accuracy));
    fbputs(next_frame, score_line, 0, 0, palette[WHITE]);
    memset(next_frame + WINDOW_WIDTH * 4 * 31, 255, WINDOW_WIDTH * 4);

    // Is it time to shift the buffer because a note has gone off-screen?
    if (round(current_bottom_row_Y) >= WINDOW_HEIGHT + 6) {
      // The bottom row is off screen
      current_bottom_row_idx++;
      current_bottom_row_Y -= (24 + 2 * note_row_veritcal_padding);
      // This counts as a miss
      num_strummed++;
      current_accuracy = (num_hit * 100.0) / num_strummed;

      if (current_bottom_row_idx >= num_note_rows + 2) {
        // We are done with the game
        break;
      }
    }

    // Push next frame to buffer
    pthread_mutex_lock(&framebuffer_mutex);
    memcpy(framebuffer, next_frame, WINDOW_WIDTH * WINDOW_HEIGHT * 4);
    pthread_mutex_unlock(&framebuffer_mutex);
  }

  // Game ends
  pthread_mutex_lock(&framebuffer_mutex);
  memset(framebuffer, 0, WINDOW_WIDTH * WINDOW_HEIGHT * 4);

  char *score_1 = (int)round(current_accuracy) < 100 ? "Score:" : "Score";
  char score_2[5];

  if ((int)round(current_accuracy) < 10)
    sprintf(score_2, " %d%%", (int)round(current_accuracy));
  else if ((int)round(current_accuracy) < 100)
    sprintf(score_2, "%d%%", (int)round(current_accuracy));
  else
    // No colon
    sprintf(score_2, "%d%%", (int)round(current_accuracy));

  fbputs(framebuffer, "Final", 0, 2, palette[WHITE]);
  fbputs(framebuffer, "Accuracy", 1, 0, palette[WHITE]);
  fbputs(framebuffer, score_1, 2, 0, palette[WHITE]);
  fbputs(framebuffer, score_2, 2, 6, palette[GREEN]);
  fbputs(framebuffer, "Thanks", 12, 1, palette[WHITE]);
  fbputs(framebuffer, "For", 13, 2, palette[WHITE]);
  fbputs(framebuffer, "Playing!", 14, 0, palette[WHITE]);

  // Try reading a high score
  FILE *hs_file = fopen("high_score.txt", "r");

  int high_score;

  if (hs_file == NULL) {
    high_score = 0;
  } else {
    fscanf(hs_file, "%d", &high_score);
    fclose(hs_file);
  }

  char best_score[11];
  sprintf(best_score, "Best: %d%%", high_score);

  int your_score = (int)round(current_accuracy);

  if (your_score > high_score) {
    fbputs(framebuffer, best_score, 6, 0, palette[WHITE]);
    fbputs(framebuffer, "NewRecord", 7, 0, palette[GREEN]);
    high_score = your_score;
  } else
    fbputs(framebuffer, best_score, 7, 0, palette[WHITE]);

  // Open the file for writing
  hs_file = fopen("high_score.txt", "w");

  if (hs_file != NULL) {
    // Write the new high score to the file
    fprintf(hs_file, "%d", high_score);
    fclose(hs_file);
  } else {
    // Error handling if unable to open file for writing
    printf("Error: Unable to open high score file for writing.\n");
  }

  pthread_mutex_unlock(&framebuffer_mutex);

  usleep(150000);

  if (EMULATING_VGA)
    VGAEmulator_destroy(&emulator);
  // Clear sprites
  unload_sprite(GH_circle_base);
  unload_sprite(GH_logo);
  unload_sprites(note_circles);
  unload_sprites(play_circles_released);
  unload_sprites(play_circles_held);

  if (EMULATING_VGA)
    free(framebuffer);

  return 0;
}
