# File: sv_gen.py 
# Author: Yunzhou Li
# Time: Spring 2024
# Description: Generate SystemVerilog code block of message display
import math

# Parameters
#	- txt: message text
#	- h_start: message start hcount [0,640)
# 			   if h_start=-1, the message will be placed at the center.
#	- v_start: message start vcount [0,480)
#			   if v_start=-1, the message will be placed at the center.
#	- width_size_order: describes font width, the final font width is
# 						2^width_size_order*8
#	- height_size_order: describes font height, the final font height is
# 						2^height_size_order*5
#	- group: message group name, if None, set to default

# Define maximum number of messages in a message group
# You must ensure that log2(num_groups) + GROUP_MAX_MSG_COUNT<=16
GROUP_MAX_MSG_COUNT = 10

# Add your message here
message_list = [{
	"txt":"GOMOKU",
	"h_start":128,
	"v_start":128,
	"width_size_order":3,
	"height_size_order":4,
	"group":"menu",
},{
	"txt":"START PVP",
	"h_start":-1,
	"v_start":250,
	"width_size_order":1,
	"height_size_order":2,
	"group":"menu",
},{
	"txt":"START PVE",
	"h_start":-1,
	"v_start":280,
	"width_size_order":1,
	"height_size_order":2,
	"group":"menu",
},{
	"txt":"CREATE LAN",
	"h_start":-1,
	"v_start":310,
	"width_size_order":1,
	"height_size_order":2,
	"group":"menu",
},{
	"txt":"JOIN LAN",
	"h_start":-1,
	"v_start":340,
	"width_size_order":1,
	"height_size_order":2,
	"group":"menu",
},{
	"txt":"EXIT",
	"h_start":-1,
	"v_start":370,
	"width_size_order":1,
	"height_size_order":2,
	"group":"menu",
},

{
	"txt":"Regret",
	"h_start":520,
	"v_start":355,
	"width_size_order":1,
	"height_size_order":2,
	"group":"ingame_options",
},{
	"txt":"HINT",
	"h_start":536,
	"v_start":385,
	"width_size_order":1,
	"height_size_order":2,
	"group":"ingame_options",
},{
	"txt":"RESIGN",
	"h_start":520,
	"v_start":415,
	"width_size_order":1,
	"height_size_order":2,
	"group":"ingame_options",
},{
	"txt":"EXIT",
	"h_start":536,
	"v_start":445,
	"width_size_order":1,
	"height_size_order":2,
	"group":"ingame_options",
},

{
	"txt":"P1",
	"h_start":583,
	"v_start":139,
	"width_size_order":1,
	"height_size_order":2,
	"group":"ingame_players",
},{
	"txt":"P2",
	"h_start":583,
	"v_start":309,
	"width_size_order":1,
	"height_size_order":2,
	"group":"ingame_players",
},{
	"txt":"AI",
	"h_start":583,
	"v_start":309,
	"width_size_order":1,
	"height_size_order":2,
	"group":"ingame_players",
},

]

def group_messages(message_list):
	groups = {}
	for msg in message_list:
		if msg.get('group',None)==None:
			msg['group']="default"
		if groups.get(msg['group'],None)==None:
			groups[msg['group']] = []
		groups[msg['group']].append(msg)
	return groups

msg_cnt = len(message_list)

msg_groups = group_messages(message_list)

code = '''\
	/*******************************************************/
	/**** Message display block, generated by sv_gen.py ****/
	/*******************************************************/

'''
for key in msg_groups:
    code+='''\
    // Message visibility and display signal of group %s
	logic [%d:0] msg_visible_%s;
	logic [%d:0] msg_display_%s;
'''%(key,len(msg_groups[key])-1,key,len(msg_groups[key])-1,key)

code+='''
	// Selected message index, 0 means none is selected
	logic [%d:0] msg_selected;
	// Whether the current message is selected
	logic cur_msg_selected;
	// Font pixel index of a line, [0,8)
	logic [2:0] font_pix_idx;
	logic [7:0] font_addr;
	logic [7:0] font_val;
	soc_system_font font(.address(font_addr),.clk(clk),.clken(1),.reset_req(0),.readdata(font_val));

	always_ff @(posedge clk) begin\
'''%(math.ceil(math.log2(msg_cnt)))

msg_idx = 0
group_no = 0
for key in msg_groups:   
	group_no+=1 
	for group_idx,msg in enumerate(msg_groups[key]):
		txt=msg['txt']
		h_start=msg['h_start']
		v_start=msg['v_start']
		msg_idx=msg_idx+1
		width_size_order=msg['width_size_order']
		height_size_order=msg['height_size_order']

		idx = []
		for c in txt.upper():
			i = ord(c)-ord('A')
			if ord(c) >= ord('0') and ord(c) <= ord('9'):
				i=ord(c)-ord('0')+26
			if c==' ':
				i=36
			elif c=='!':
				i=37
			elif c=='?':
				i=38
			elif c==',':
				i=39
			elif c=='.':
				i=40
			elif c=='+':
				i=41
			elif c=='-':
				i=42
			elif c=='*':
				i=43
			elif c=='/':
				i=44
			idx.append(i)

		seg_width = pow(2,width_size_order)
		seg_height = pow(2,height_size_order)
		str_len = len(txt)
		str_len_order = math.ceil(math.log2(str_len))

		if v_start == -1:
			v_start = (480-seg_height*5)/2
		if h_start == -1:
			h_start = (640-seg_width*8*str_len)/2

		code +='''
		/**
		* message: %s
		* group %d: %s
		* group_index: %d
		* select_index: %d
		* h_start: %d
		* v_start: %d
		* font_width: %d
		* font_height: %d
		*/
		if((hcount[10:1] >= 10'd%d) && (hcount[10:1] < 10'd%d) && (vcount >= 10'd%d) && (vcount < 10'd%d) && msg_visible_%s[%d]) begin
			msg_display_%s[%d] <= 1;
			cur_msg_selected <= (msg_selected==%d);
'''%(txt,group_no,key,group_idx,msg_idx,h_start,v_start,8*seg_width,5*seg_height,\
		h_start, h_start+8*seg_width*str_len, v_start, v_start+5*seg_height, key,group_idx,\
		key,group_idx,\
		msg_idx)

		code +='''\
			case((hcount[10:1]-%d)>>%d)
'''%(h_start,width_size_order+3)

		for i in range(str_len):
			code +='''\
				8'd%d: font_addr <= 8'd%d+((vcount[9:0]-%d)>>%d);
'''%(i,idx[i]*5,v_start,height_size_order)

		code +='''\
				default:;
			endcase
			font_pix_idx <= (hcount[10:1]-%d)>>%d;
		end
		else
			msg_display_%s[%d] <= 0;
'''%(h_start,width_size_order,key,group_idx)

code +='''	end

	/*******************************************************/
	/************** Message display block end **************/
	/*******************************************************/
	
		/*
		* You may need to put the following code in the combinational logic.
'''
for key in msg_groups:
    code+='''\
			// Draw %s message
			if(msg_display_%s && font_val[font_pix_idx]) begin
				// Font selected			
				if(cur_msg_selected)
					{VGA_R, VGA_G, VGA_B} = 24'h00ffff;
				else
					{VGA_R, VGA_G, VGA_B} = 24'h000000;
			end
'''%(key,key)
code+='''\
		*/
'''

code +='''
		/*
		* You may need to put the following code in the avalon interface.
			case(writedata[%d:%d])
'''%(15,GROUP_MAX_MSG_COUNT)
group_no=0
for key in msg_groups:
    group_no+=1
    code+='''\
				%d'h%d:msg_visible_%s <= writedata[%d:0];
'''%(16-GROUP_MAX_MSG_COUNT,group_no,key,len(msg_groups[key])-1)
code+='''\
				default:;
			endcase
		*/
'''

print(code)

with open("display.sv","w") as f:
    f.write(code)