/*
 * Userspace program that communicates with the cat_invaders device driver
 * through ioctls
 *
 * Stephen A. Edwards
 * Columbia University
 */

#include <stdio.h>
#include "cat_invaders.h"
#include "interface.h"
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

int cat_invaders_fd;
cat_invaders_arg_t vla;

void init_driver(){
    static const char filename[] = "/dev/cat_invaders";
    if ( (cat_invaders_fd = open(filename, O_RDWR)) == -1) {
        fprintf(stderr, "could not open %s\n", filename);
    }
    printf("driver init");
}


void set_dog_pos(unsigned short value){
    vla.dog_p_x = value;

    if (ioctl(cat_invaders_fd, WRITE_DOG_POSITION , &vla)) {
      perror("ioctl(WRITE_DOG_POSITION) failed");
    }
    //printf("Write DOG X_LSB: 0x%X X_MSB: 0x%X\n", reg_lsb, reg_msb);
}


void set_projectile_dog_position(unsigned short x, unsigned short y, unsigned short visibility){
    vla.projectile_dog.p_x = x;
    vla.projectile_dog.p_y = (y |= (visibility << 15));

    if (ioctl(cat_invaders_fd, WRITE_PROJECTILE_DOG , &vla)) {
      perror("ioctl(WRITE_PROJECTILE_DOG) failed");
    }

    //printf("Write X_LSB: 0x%X X_MSB: 0x%X Y:0x%X\n", reg_lsb, reg_msb, y);

}

void set_projectile_sprite(unsigned char number, unsigned short x, unsigned short y, unsigned short visibility){
    
    switch (number){
        case 0:   vla.projectile_cat_group.c0.x = x;
                  vla.projectile_cat_group.c0.y = (y |= (visibility << 15)) ;
                  if (ioctl(cat_invaders_fd, WRITE_PROJECTILE_CAT_0 , &vla)) {
                  perror("ioctl(WRITE_PROJECTILE_SPRITES_0) failed");}
                  break;
        case 1:  vla.projectile_cat_group.c1.x = x;
                  vla.projectile_cat_group.c1.y = (y |= (visibility << 15)) ;
                  if (ioctl(cat_invaders_fd, WRITE_PROJECTILE_CAT_1 , &vla)) {
                  perror("ioctl(WRITE_PROJECTILE_SPRITES_1) failed");}
                  break;
        case 2:   vla.projectile_cat_group.c2.x = x;
                  vla.projectile_cat_group.c2.y = (y |= (visibility << 15)) ;
                  if (ioctl(cat_invaders_fd, WRITE_PROJECTILE_CAT_2 , &vla)) {
                  perror("ioctl(WRITE_PROJECTILE_SPRITES_2) failed");}
                  break;
        case 3:   vla.projectile_cat_group.c3.x = x;
                  vla.projectile_cat_group.c3.y = (y |= (visibility << 15)) ;
                  if (ioctl(cat_invaders_fd, WRITE_PROJECTILE_CAT_3 , &vla)) {
                  perror("ioctl(WRITE_PROJECTILE_SPRITES_3) failed");}
                  break;
        default: break;
    }
    

    //printf("Write S%X: X:0x%X  Y:0x%X\n", number, *x, *y);
}

void set_sprite_pos(unsigned short x, unsigned short y){
    vla.cat_matrix_pos.x = x;
    vla.cat_matrix_pos.y = y;

    if (ioctl(cat_invaders_fd, WRITE_CAT_MATRIX_POSITION , &vla)) {
      perror("ioctl(WRITE_SPRITE_POSITION) failed");}

    //printf("Write SPRITE POS: X: 0x%X  Y:0x%X\n", *x, *y);
}


void set_ufo_pos(unsigned short x, unsigned short vis){
    vla.mystery_pos_x = (x |= (vis << 15))  ;
    if (ioctl(cat_invaders_fd, WRITE_MYSTERY_POSITION , &vla)) {
      perror("ioctl(WRITE_UFO_POSITION) failed");}

    //printf("Write UFO POS: X_LSB: 0x%X X_MSB: 0x%X\n", reg_lsb, reg_msb);
}



void set_status(unsigned short level, unsigned short gameStatus, unsigned short
life){
    vla.status = (((level&15) << 4)) | ((gameStatus&3) << 2) | (life&3);  
    if (ioctl(cat_invaders_fd, WRITE_STATUS , &vla)) {perror("ioctl(WRITE_lives) failed");}
}


void set_level(unsigned short value){
    unsigned short status = vla.status;
    status &= 0xFF0F; 
    status |= ((value&15) << 4);  
    vla.status = status;  
    if (ioctl(cat_invaders_fd, WRITE_STATUS , &vla)) {perror("ioctl(WRITE_level) failed");}
}

void set_lives(unsigned short value){
    unsigned short status = vla.status;
    status &= 0xFFFC; 
    status |= (value & 3);
    vla.status = status;  
    if (ioctl(cat_invaders_fd, WRITE_STATUS , &vla)) {perror("ioctl(WRITE_lives) failed");}
}

void set_gameStatus(unsigned short value){
    unsigned short status = vla.status;
    status &= 0xFFF3;
    status |= (value&3)<<2;
    vla.status = status;
    if (ioctl(cat_invaders_fd, WRITE_STATUS , &vla)) {perror("ioctl(WRITE_gameStatus) failed");}
}

void set_score(unsigned short score){
    unsigned short units = score % 10;
    unsigned short tens = (score / 10) % 10;
    unsigned short hundreds = (score / 100) % 10;
    vla.score = hundreds | (tens << 4) | (units << 8);
    if (ioctl(cat_invaders_fd, WRITE_SCORE , &vla)) {perror("ioctl(WRITE_SCORE ) failed");}
   // printf("Write score: %X\n", *score);
}

void set_dog_ani(unsigned short state){
    vla.dog_ani = state;
    if (ioctl(cat_invaders_fd, WRITE_DOG_ANIMATION , &vla)) {perror("ioctl(WRITE_ANIMATION ) failed");}
    //printf("Write animation target: %X, direction: %X, state:%X\n", target, *direction, *state)
}

void set_cat_ani(unsigned short state){
    vla.cat_ani = state;
    if (ioctl(cat_invaders_fd, WRITE_CAT_ANIMATION , &vla)) {perror("ioctl(WRITE_ANIMATION ) failed");}
    //printf("Write animation target: %X, direction: %X, state:%X\n", target, *direction, *state)
}

void set_barrier(unsigned short number, unsigned short status){
    unsigned short barrier = (number << 3) | status;
    vla.barrier_mat = barrier;
    if (ioctl(cat_invaders_fd, WRITE_BARRIER , &vla)) {perror("ioctl(WRITE_BARRIER ) failed");}

    //printf("Write Barrier: %X, state:%X\n", *number, *status);
}
void set_sprite_matrix(unsigned char target, unsigned short status){
    vla.cat_mat = (target << 3) | status;
    
    if (ioctl(cat_invaders_fd, WRITE_CAT_MATRIX , &vla)) {perror("ioctl(WRITE_CAT_MATRIX) failed"); }
}

void play_background(unsigned char volume){
    //vla.audio.background = (0x20 | volume);
    vla.audio.background = 0xf0;

    if (ioctl(cat_invaders_fd, WRITE_BG , &vla)) {perror("ioctl(WRITE_BG ) failed");}
    //printf("Play Background: %X\n", *state);
}

void play_bark(unsigned char volume){
    vla.audio.bark = 0x00;
    if (ioctl(cat_invaders_fd, WRITE_BARK , &vla)) {perror("ioctl(WRITE_BARK ) failed");}
    vla.audio.bark = (0x20 | volume);
    
    if (ioctl(cat_invaders_fd, WRITE_BARK , &vla)) {perror("ioctl(WRITE_BARK ) failed");}
    //printf("Play Bark: %X\n", *state);
}

void play_meow(unsigned char volume){
    vla.audio.meow = 0x00;
    if (ioctl(cat_invaders_fd, WRITE_MEOW , &vla)) {perror("ioctl(WRITE_MEOW ) failed");}
    vla.audio.meow = (0x20|volume);
    if (ioctl(cat_invaders_fd, WRITE_MEOW , &vla)) {perror("ioctl(WRITE_MEOW ) failed");}
    //printf("Play Meow: %X\n", *state);
}

void set_speed(unsigned char speed){
    vla.audio.speed = speed;
    if (ioctl(cat_invaders_fd, WRITE_SPEED , &vla)) {perror("ioctl(WRITE_MEOW ) failed");}
    //printf("Play Meow: %X\n", *state);
}

void set_explosion(unsigned short x, unsigned short y, unsigned short visibility){
    vla.explosion.x = x;
    vla.explosion.y = y|(visibility << 15);
    if (ioctl(cat_invaders_fd, WRITE_EXPLOSION , &vla)) {perror("ioctl(WRITE_EXPLOSION ) failed");}
}

void clear_irq(unsigned short irq){
    vla.irq = irq;
    if (ioctl(cat_invaders_fd, CLEAR_IRQ , &vla)) {perror("ioctl(WRITE_MEOW ) failed");}
    //printf("Play Meow: %X\n", *state);
}


unsigned short get_test_reg(void){
    if (ioctl(cat_invaders_fd, READ_TEST_REG , &vla)) {perror("ioctl(READ_TEST_REG ) failed");}
    
    printf("Test Reg: %X\n", vla.test_reg);
    return vla.test_reg;
}

unsigned short get_frame_reg(void){
    if (ioctl(cat_invaders_fd, READ_FRAME_REG , &vla)) {perror("ioctl(READ_FRAME_REG ) failed");}
    
    return vla.frame_reg;
}

// unsigned char get_bullet(void){
//   if (ioctl(cat_invaders_fd, READ_MISC_0 , &vla)) {
//       perror("get bullet failed");
//   }
//   return vla.misc_0;
// }

// /* For those which has both x and y position, {0xFF, 0xFF} by default*/


// unsigned short get_dog_pos(void){
//   if (ioctl(cat_invaders_fd, READ_DOG_POSITION , &vla)) {
//       perror("Get dog position failed");
//   }
//   unsigned short pos = (vla.dog_pos.x_msb<<2) | vla.dog_pos.x_lsb;
//   return pos;
// }


// Position get_projectile_dog(void){
//   Position pos = {0xFF, 0xFF};
//   if (ioctl(cat_invaders_fd, READ_PROJECTILE_DOG , &vla)) {
//       perror("get projectile failed");
//   }

//   pos.x =  (unsigned short)(vla.projectile_dog.x_msb << 2) | vla.projectile_dog.x_lsb; 
//   pos.y = vla.projectile_dog.y;

//   return pos;
// }

// Position get_projectile_sprites(unsigned char target){
//   Position pos = {0xFF, 0xFF};

//   if (ioctl(cat_invaders_fd, READ_PROJECTILE_SPRITES , &vla)) {
//       perror("get projectile sprites failed");
//   }
//   switch (target){
//     case 0x00: pos.x = (unsigned char)vla.projectile_sprites.s0.x; 
//             pos.y = vla.projectile_sprites.s0.y; 
//             break;
//     case 0x01: pos.x = (unsigned char)vla.projectile_sprites.s1.x; 
//             pos.y = vla.projectile_sprites.s1.y; 
//             break;
//     case 0x02: pos.x = (unsigned char)vla.projectile_sprites.s2.x; 
//             pos.y = vla.projectile_sprites.s2.y; 
//             break;
//     case 0x03: pos.x = (unsigned char)vla.projectile_sprites.s3.x; 
//             pos.y = vla.projectile_sprites.s3.y; 
//             break;
//     default: break;
//   }

//   return pos;
// }

// Position get_sprite_pos(void){
//    Position pos = {0xFF, 0xFF};
//   if (ioctl(cat_invaders_fd, READ_SPRITE_POSITION , &vla)) {
//       perror("Get sprites pos failed");
//   }
//   pos.x = vla.sprite_pos.x;
//   pos.y = vla.sprite_pos.y;
//   return pos;
// }

// unsigned short get_ufo_pos(void){

//   if (ioctl(cat_invaders_fd, READ_UFO_POSITION , &vla)) {
//       perror("Get UFO pos failed");
//   }
//   return ((vla.ufo_pos.x_msb << 2) | vla.ufo_pos.x_lsb); 
// }

// unsigned char get_misc1(void){
//   if (ioctl(cat_invaders_fd, READ_MISC_1 , &vla)) {
//       perror("get misc1 failed");
//   }
//   return vla.misc_1;
// }

// unsigned short get_score(void){
//   if (ioctl(cat_invaders_fd, READ_SCORE , &vla)) {
//       perror("get score failed");
//   }
//   return (unsigned short)(vla.score.score_msb << 2) & vla.score.score_lsb;
// }

// unsigned char get_barrier(unsigned char target){

//   if (ioctl(cat_invaders_fd, READ_BARRIER , &vla)) {
//       perror("get barrier state failed");
//   }
//   unsigned char state = 0xFF;
//   switch (target){
//     case 0x00: state = vla.barrier.b0.row_0; break;
//     case 0x01: state = vla.barrier.b1.row_0; break;
//     case 0x02: state = vla.barrier.b2.row_0; break;
//     case 0x03: state = vla.barrier.b3.row_0; break;
//     default: break;
//   }
//   return state;
// }

// unsigned char get_animation_dog(void){
//   if (ioctl(cat_invaders_fd, READ_ANIMATION , &vla)) {
//       perror("get dog animation state failed");
//   }
//   return vla.animation.dog_ani;
// }

// unsigned char get_animation_cat(void){
//   if (ioctl(cat_invaders_fd, READ_ANIMATION , &vla)) {
//       perror("get cat animation state failed");
//   }
//   return vla.animation.cat_ani;
// }

// unsigned char get_background(void){
//     if (ioctl(cat_invaders_fd, READ_BG , &vla)) {
//       perror("get background audio state failed");
//   }
//   return vla.audio.background;
// }
// unsigned char get_bark(void){
//     if (ioctl(cat_invaders_fd, READ_BARK , &vla)) {
//       perror("get bark audio state failed");
//   }
//   return vla.audio.background;
// }
// unsigned char get_meow(void){
//     if (ioctl(cat_invaders_fd, READ_MEOW , &vla)) {
//       perror("get meow audio state failed");
//   }
//   return vla.audio.meow;
// }
