/* * Device driver for the VGA video and audio generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod cat_invaders.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree cat_invaders.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "cat_invaders.h"
#include <linux/interrupt.h>
#include <linux/ioport.h>

#define DRIVER_NAME "cat_invaders"

/* Device registers */
#define DOG_POS(x) (x)
#define PROJECT_D_X(x) ((x)+2)
#define PROJECT_D_Y(x) ((x)+4)
#define PROJECT_C0_X(x) ((x)+6)
#define PROJECT_C0_Y(x) ((x)+8)
#define PROJECT_C1_X(x) ((x)+10)
#define PROJECT_C1_Y(x) ((x)+12)
#define PROJECT_C2_X(x) ((x)+14)
#define PROJECT_C2_Y(x) ((x)+16)
#define PROJECT_C3_X(x) ((x)+18)
#define PROJECT_C3_Y(x) ((x)+20)
#define CAT_ARRAY_POS_X(x) ((x)+22)
#define CAT_ARRAY_POS_Y(x) ((x)+24)
#define MYSTERY_POS_X(x) ((x)+26)
#define STATUS(x) ((x)+28)
#define SCORE(x) ((x)+30)
#define DOG_ANI(x) ((x)+32)
#define CAT_ANI(x) ((x)+34)
#define EXPLOSION_X(x) ((x)+36)
#define EXPLOSION_Y(x) ((x)+38)
#define BARRIER_MAT(x) ((x)+(46*2))
#define CAT_MAT(x) ((x)+(41*2))
#define AUDIO_BG(x) ((x)+84)
#define AUDIO_MEOW(x) ((x)+86)
#define AUDIO_BARK(x) ((x)+88)
#define AUDIO_SPEED(x) ((x)+90)
#define EOF_IRQ(x) ((x)+94)

#define TEST_REG(x)    ((x)+(48*2))
#define FRAME_REG(x)   ((x)+(49*2))



#define UINPUT_INT_NUM 72
/*
 * Information about our device
 */
struct cat_invaders_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
	unsigned short dog_p_x;
  	projectile_dog_t projectile_dog;
 	projectile_cat_group_t projectile_cat_group;
  	cat_matrix_position_t cat_matrix_pos;
    explosion_t explosion;
    unsigned short mystery_pos_x;
  	unsigned short status;
  	unsigned short score;
  	unsigned short dog_ani;
  	unsigned short cat_ani;
  	unsigned short barrier_mat; 
  	unsigned short cat_mat;
  	audio_t audio;
  	unsigned short irq;
    unsigned short test_reg;
    unsigned short frame_reg;
} dev;

/* IRQ STUFF*/
static DECLARE_WAIT_QUEUE_HEAD(interrupt_wq);
static int interrupt_flag = 0;
static DEFINE_SPINLOCK(interrupt_flag_lock);

/*functions*/

/*dog position*/
static void write_dog_pos(unsigned short *value)
{
	iowrite16(*value, DOG_POS(dev.virtbase));
	dev.dog_p_x = *value;
}

/*projectile dog*/
static void write_projectile_dog(projectile_dog_t *pos)
{
	iowrite16(pos->p_x, PROJECT_D_X(dev.virtbase));
	iowrite16(pos->p_y, PROJECT_D_Y(dev.virtbase));
    dev.projectile_dog = *pos;
}

/*projectile cats*/
static void write_projectile_cats(projectile_cat_group_t *pro_cats, unsigned char num)
{
	switch(num){
		case 0:	iowrite16(pro_cats->c0.x, PROJECT_C0_X(dev.virtbase)); 
    			iowrite16(pro_cats->c0.y, PROJECT_C0_Y(dev.virtbase));
				dev.projectile_cat_group.c0 = pro_cats->c0; break;
		case 1:	iowrite16(pro_cats->c1.x, PROJECT_C1_X(dev.virtbase));
    			iowrite16(pro_cats->c1.y, PROJECT_C1_Y(dev.virtbase));
				dev.projectile_cat_group.c1 = pro_cats->c1; break;
		case 2:	iowrite16(pro_cats->c2.x, PROJECT_C2_X(dev.virtbase));
    			iowrite16(pro_cats->c2.y, PROJECT_C2_Y(dev.virtbase));
				dev.projectile_cat_group.c2 = pro_cats->c2; break;
		case 3:	iowrite16(pro_cats->c3.x, PROJECT_C3_X(dev.virtbase));
    			iowrite16(pro_cats->c3.y, PROJECT_C3_Y(dev.virtbase));
				dev.projectile_cat_group.c3 = pro_cats->c3; break;
	default: break;
	}
    
}

/*cat matrix position*/
static void write_cat_pos(cat_matrix_position_t *pos)
{
	iowrite16(pos->x, CAT_ARRAY_POS_X(dev.virtbase));
    iowrite16(pos->y, CAT_ARRAY_POS_Y(dev.virtbase));
	dev.cat_matrix_pos = *pos;
}

/*ufo position*/
static void write_ufo_pos(unsigned short *value)
{
	iowrite16(*value, MYSTERY_POS_X(dev.virtbase));
	dev.mystery_pos_x = *value;
}

/* misc 1*/
static void write_status(unsigned short *status)
{
	iowrite16(*status, STATUS(dev.virtbase));
	dev.status = *status;
}

/* score*/
static void write_score(unsigned short *score)
{
	iowrite16(*score, SCORE(dev.virtbase));
	dev.score = *score;
}

/* dog animation*/
static void write_dog_animation(unsigned short *dog_ani)
{
	iowrite16(*dog_ani, DOG_ANI(dev.virtbase));
	dev.dog_ani = *dog_ani;
}

/* cat animation*/
static void write_cat_animation(unsigned short *cat_ani)
{
	iowrite16(*cat_ani, CAT_ANI(dev.virtbase));
	dev.cat_ani = *cat_ani;
}

/* barrier*/
static void write_barrier(unsigned short *barrier)
{
	iowrite16(*barrier, BARRIER_MAT(dev.virtbase));
	dev.barrier_mat = *barrier;
}

/* cat matrix*/
static void write_cat_matrix(unsigned short *value)
{
	iowrite16(*value, CAT_MAT(dev.virtbase));
	dev.cat_mat = *value;
}

static void write_explosion(explosion_t *value)
{
    iowrite16(value->x, EXPLOSION_X(dev.virtbase));
	iowrite16(value->y, EXPLOSION_Y(dev.virtbase));
	dev.explosion = *value;
}

/* audio*/
static void write_bg(unsigned short *value)
{
	iowrite16(*value, AUDIO_BG(dev.virtbase));
	dev.audio.background = *value;
}
static void write_bark(unsigned short *value)
{
    iowrite16(*value, AUDIO_BARK(dev.virtbase));
	dev.audio.bark = *value;
}
static void write_meow(unsigned short *value)
{
    iowrite16(*value, AUDIO_MEOW(dev.virtbase));
	dev.audio.meow = *value;
}
static void write_speed(unsigned short *value)
{
    iowrite16(*value, AUDIO_SPEED(dev.virtbase));
	dev.audio.speed = *value;
}
static void clear_irq(unsigned short *value)
{
    iowrite16(*value,  EOF_IRQ(dev.virtbase));
	dev.irq = *value;
}

static void read_test_reg(unsigned short *value) {
   *value = ioread16(TEST_REG(dev.virtbase));
}

static void read_frame_reg(unsigned short *value) {
   *value = ioread16(FRAME_REG(dev.virtbase));
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long cat_invaders_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	cat_invaders_arg_t vla;
	switch (cmd) {

	case WRITE_DOG_POSITION:
		
		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_dog_pos(&vla.dog_p_x);
		break;

    case WRITE_PROJECTILE_DOG:
		
		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_projectile_dog(&vla.projectile_dog);
		break;

    case WRITE_PROJECTILE_CAT_0:

		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_projectile_cats(&vla.projectile_cat_group, 0);
		break;

	case WRITE_PROJECTILE_CAT_1:

		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_projectile_cats(&vla.projectile_cat_group, 1);
		break;

	case WRITE_PROJECTILE_CAT_2:

		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_projectile_cats(&vla.projectile_cat_group, 2);
		break;

	case WRITE_PROJECTILE_CAT_3:

		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_projectile_cats(&vla.projectile_cat_group, 3);
		break;

    case WRITE_CAT_MATRIX_POSITION:

		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_cat_pos(&vla.cat_matrix_pos);
		break;

    case WRITE_MYSTERY_POSITION:

		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_ufo_pos(&vla.mystery_pos_x);
		break;


    case WRITE_STATUS:

		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_status(&vla.status);
		break;
	
    case WRITE_SCORE:

		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_score(&vla.score);
		break;

    case WRITE_DOG_ANIMATION:

		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_dog_animation(&vla.dog_ani);
		break;

    case WRITE_CAT_ANIMATION:

		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_cat_animation(&vla.cat_ani);
		break;

    case WRITE_BARRIER:
	
		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_barrier(&vla.barrier_mat);
		break;
	
	case WRITE_CAT_MATRIX:
	
		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_cat_matrix(&vla.cat_mat);
		break;

    case WRITE_BG:
		
		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_bg(&vla.audio.background);
		break;

	case WRITE_BARK:
		
		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_bark(&vla.audio.bark);
		break;

	case WRITE_MEOW:
		
		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_meow(&vla.audio.meow);
		break;

	case WRITE_SPEED:
		
		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_speed(&vla.audio.speed);
		break;
	
	case CLEAR_IRQ:
		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		clear_irq(&vla.irq);
		break;
 
	case WRITE_EXPLOSION:
		if (copy_from_user(&vla, (cat_invaders_arg_t *) arg, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		write_explosion(&vla.explosion);
		break;

    case READ_TEST_REG:
        
        read_test_reg(&vla.test_reg);
		if (0 != copy_to_user((cat_invaders_arg_t *) arg, &vla, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		break;

    case READ_FRAME_REG:
        read_frame_reg(&vla.frame_reg);
		if (0 != copy_to_user((cat_invaders_arg_t *) arg, &vla, sizeof(cat_invaders_arg_t)))
		    return -EACCES;
		break;

	default:
		return -EINVAL;
	}
	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations cat_invaders_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = cat_invaders_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice cat_invaders_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &cat_invaders_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init cat_invaders_probe(struct platform_device *pdev)
{
        //cat_invaders_color_t beige = { 0xf9, 0xe4, 0xb7 };
		//audio_t audio_begin = { 0x00, 0x00, 0x00 };
	int ret;

	/* Register ourselves as a misc device: creates /dev/cat_invaders */
	ret = misc_register(&cat_invaders_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}
	
	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&cat_invaders_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int cat_invaders_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&cat_invaders_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id cat_invaders_of_match[] = {
	{ .compatible = "csee4840,cat_invaders-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, cat_invaders_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver cat_invaders_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(cat_invaders_of_match),
	},
	.remove	= __exit_p(cat_invaders_remove),
};

/* Called when the module is loaded: set things up */
static int __init cat_invaders_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&cat_invaders_driver, cat_invaders_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit cat_invaders_exit(void)
{
	platform_driver_unregister(&cat_invaders_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(cat_invaders_init);
module_exit(cat_invaders_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("cat_invaders driver");
