#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "./game_driver.h"

#define SIZE 4
int board[SIZE][SIZE];
int score;
int best_score;
int frame1 [16][4];
int frame2 [16][4];
int frame3 [16][4];
int frame4 [16][4];
int frame5 [16][4];
int frame6 [16][4];
int game_on = 1;
tile_t to_load[16];

void init_board() {
    // initialize the board
    for (int i = 0; i < 4; i++) {
        for (int j = 0; j < 4; j++) {
            board[i][j] = 0;
        }
    }
}

void init_frame() {
    // initialize the frames
    for (int i = 0; i < SIZE; i++) {
        for (int j = 0; j < SIZE; j++) {
            frame1[i*4+j][0] = i*100+100;
            frame2[i*4+j][0] = i*100+100;
            frame3[i*4+j][0] = i*100+100;
            frame4[i*4+j][0] = i*100+100;

            frame1[i*4+j][1] = j*100+100;
            frame2[i*4+j][1] = j*100+100;
            frame3[i*4+j][1] = j*100+100;
            frame4[i*4+j][1] = j*100+100;

            frame1[i*4+j][2] = 4;
            frame2[i*4+j][2] = 4;
            frame3[i*4+j][2] = 4;
            frame4[i*4+j][2] = 4;
        }
    }
}

void init_game() {
    score = 0; 
    best_score = 0;
    init_board();
}

void restart() {
    init_frame();
    init_board();
    if (score > best_score) {
        best_score = score;
    }
    score = 0;
}

void print_board() {
    // print the current board layout to the console (only for test purpose)

    for (int i = 0; i < SIZE; i++) {
        for (int j = 0; j < SIZE; j++) {
            printf("%d  ", board[i][j]);
        }
        printf("\n");
    }
}

void generate_tile() {
    for (int i = 0; i < 16; i++) {
        for (int j = 0; j < 4; j++) {
            frame5[i][j] = frame4[i][j];
            frame6[i][j] = frame4[i][j];
        }
    }
    int emp_cells[16];
    int num_empty = 0;
    for (int i = 0; i < 16; i++) {
        if (board[i/4][i%4] == 0) {
            emp_cells[num_empty] = i;
            num_empty ++;
        }
    }

    // randomly generate a 2 or 4 in empty space
    int rand_idx = emp_cells[rand() % num_empty];
    int val = (rand() % 2 + 1) * 2;
    board[rand_idx/4][rand_idx%4] = val;


    // generate tile animation happens in the fifth and sixth frame, where the tile zooms out from 50% size to 100% size
    frame5[rand_idx][0] = rand_idx/4 * 100 + 100;
    frame6[rand_idx][0] = rand_idx/4 * 100 + 100;
    frame5[rand_idx][1] = rand_idx%4 * 100 +100;
    frame6[rand_idx][1] = rand_idx%4 * 100 +100;
    frame5[rand_idx][2] = 2;
    frame6[rand_idx][2] = 4;
    frame5[rand_idx][3] = val;
    frame6[rand_idx][3] = val;


    printf("tile generated at (%d, %d) \n", rand_idx/4, rand_idx%4);
}

void shift(int dest, int orig) {
    int origX = orig/4*100+100;
    int origY = orig%4*100+100;
    int destX = dest/4*100+100;
    int destY = dest%4*100+100;
    int origVal = board[orig/4][orig%4];

    //slide animation
    //in first and second frame, shrink the target tile, then enlarge it
    frame1[dest][2] = 2;
    frame2[dest][2] = 3;

    //resize the target tile to its original size and orientation in frame3, 4
    frame3[dest][0] = destX;
    frame3[dest][1] = destY;
    frame3[dest][2] = 4;
    frame3[dest][3] = origVal;

    frame4[dest][0] = destX;
    frame4[dest][1] = destY;
    frame4[dest][2] = 4;
    frame4[dest][3] = origVal;

    //slide the origin tile to the target tile's position and resize it in the process to create the sliding animation
    if (origX < destX) {
        frame1[orig][0] = (destX-origX)/3+origX;
        frame2[orig][0] = (destX-origX)/2+origX;
    }   
    else {
        frame1[orig][0] = (origX-destX)/3+destX;
        frame2[orig][0] = (origX-destX)/2+destX;
    } 
    if (origY < destY) {
        frame1[orig][1] = (destY-origY)/3+origY;
        frame2[orig][1] = (destY-origY)/2+origY;
    }
    else {
        frame1[orig][1] = (origY-destY)/3+destY;
        frame2[orig][1] = (origY-destY)/2+destY;
    }

    //in the third, and fourth frame, the original tile reached the target position and replace it with the target tile to complete the slide
    frame3[orig][0] = origX;
    frame3[orig][1] = origY;
    frame4[orig][0] = origX;
    frame4[orig][1] = origY;

    //size of the original tile in the four frames (50%-75%-100%-100%)
    frame1[orig][2] = 2;
    frame2[orig][2] = 3;
    frame3[orig][2] = 4;
    frame4[orig][2] = 4;

    //the value of the original tile, in the first two frames is its original value during the slide and return to 0 after the slide to complete the shift
    frame1[orig][3] = origVal;
    frame2[orig][3] = origVal;
    frame3[orig][3] = 0;
    frame4[orig][3] = 0;
}

void load(int[][] frame, tile_t[] to_load) {
    for (int i = 0; i < 16; i++){
        tile_t tmp;
        tmp.x = (unsigned) frame[i][0];
        tmp.y = (unsigned) frame[i][1];
        tmp.size = (unsigned char) frame[i][2];
        tmp.type = (unsigned char) sqrt(frame[i][3])-1;
    }
}

void render_frames() {
    game_arg_t ga;
    //load score and frames in sequence and sent to the hardware
    ga.current_score.score = score;
    if (ioctl(game_fd, GAME_WRITE_CURRENT_SCORE, &ga)) {
        perror("ioctl(GAME_WRITE_TILES) failed");
        return;
    }

    ga.best_score.score = best_score;
    if (ioctl(game_fd, GAME_WRITE_BEST_SCORE, &ga)) {
        perror("ioctl(GAME_WRITE_TILES) failed");
        return;
    }

    load(frame1, to_load);
    memcpy(ga.tiles, to_load, 16 * sizeof(tile_t));
    if (ioctl(game_fd, GAME_WRITE_TILES, &ga)) {
        perror("ioctl(GAME_WRITE_TILES) failed");
        return;
    }
    usleep(400000);

    load(frame2, to_load);
    memcpy(ga.tiles, to_load, 16 * sizeof(tile_t));
    if (ioctl(game_fd, GAME_WRITE_TILES, &ga)) {
        perror("ioctl(GAME_WRITE_TILES) failed");
        return;
    }
    usleep(400000);

    load(frame3, to_load)
    memcpy(ga.tiles, to_load, 16 * sizeof(tile_t));
    if (ioctl(game_fd, GAME_WRITE_TILES, &ga)) {
        perror("ioctl(GAME_WRITE_TILES) failed");
        return;
    }
    usleep(400000);

    load(frame4, to_load)
    memcpy(ga.tiles, to_load, 16 * sizeof(tile_t));
    if (ioctl(game_fd, GAME_WRITE_TILES, &ga)) {
        perror("ioctl(GAME_WRITE_TILES) failed");
        return;
    }
    usleep(400000);

    load(frame5, to_load)
    memcpy(ga.tiles, to_load, 16 * sizeof(tile_t));
    if (ioctl(game_fd, GAME_WRITE_TILES, &ga)) {
        perror("ioctl(GAME_WRITE_TILES) failed");
        return;
    }
    usleep(400000);

    load(frame6, to_load)
    memcpy(ga.tiles, to_load, 16 * sizeof(tile_t));
    if (ioctl(game_fd, GAME_WRITE_TILES, &ga)) {
        perror("ioctl(GAME_WRITE_TILES) failed");
        return;
    }
    usleep(400000);
}



void merge(int dest, int orig) {
    int origX = orig/4*100+100;
    int origY = orig%4*100+100;
    int destX = dest/4*100+100;
    int destY = dest%4*100+100;

    //merge animation
    //shift towards the target tile in the third frame and shrink the tile during the process to create the animation
    if (destX < origX) {
        frame3[orig][0] = (origX-destX)/2 + destX;
    }
    else {
        frame3[orig][0] = (destX-origX)/2 + origX;
    }
    if (destY < origY) {
        frame3[orig][1] = (origY-destY)/2 + destY;
    }
    else {
        frame3[orig][1] = (destY-origY)/2 + origY;
    }
    frame3[orig][2] = 3;
    frame3[orig][3] = board[orig/4][orig%4];

    //finishing merge in the fourth frame and resize the finished tile to its original size
    frame4[orig][0] = origX;
    frame4[orig][1] = origY;
    frame4[orig][2] = 4;
    frame4[orig][3] = 0;

    //shrink the target file in the third frame
    frame3[dest][0] = destX;
    frame3[dest][1] = destY;
    frame3[dest][2] = 3;
    frame3[dest][3] = board[dest/4][dest%4];

    //resize the target tile in fourth frame to its original size completing the merge
    frame4[dest][0] = destX;
    frame4[dest][1] = destY;
    frame4[dest][2] = 4;
    frame4[dest][3] = board[dest/4][dest%4]*2;

}


void shift_left() {
    init_frame();
    int valid_move = 0;
    for (int i = 0; i < SIZE; i++) {
        int k = 0;
        int first_zero = 1;
        for (int j = 0; j < SIZE; j++) {
            if (first_zero) {
                if (board[i][j] == 0) {
                    k = j;
                    first_zero = 0;
                }

            }
            else {
                if (board[i][j] == 0 || j == k) {
                    continue;
                }
                else {
                    // shift animation
                    shift(i*4+k-1, i*4+j-1);

                    //if the current tile holds a non-zero number, 
                    // then switch it with the first zero tile to make the shift happen
                    board[i][k] = board[i][j];
                    board[i][j] = 0;
                    k ++;
                    valid_move = 1;
                }
            }
        }
    }

    for (int i = 0; i < SIZE; i++) {
        for (int j = 3; j > 0; j--) {
            if (board[i][j] != 0 && board[i][j] == board[i][j-1]) {
                // a valid move has been made
                valid_move = 1;

                //merge animation
                merge(i*4+j-2, i*4+j-1);
                
                //update the value of the board due to merge
                board[i][j-1] *= 2;
                board[i][j] = 0;

                //update score
                score += board[i][j-1];
                j = j - 1;
            }
        }
    }

    if (!valid_move) {
        // no tiles has been changed, the move is invalid such the board cannot shift in this direction
        printf("invalid move \n");
        return;
    }
    generate_tile();
    render_frames();
}

void shift_right() {
    init_frame();
    int valid_move = 0;

    for (int i = 0; i < SIZE; i++) {
        int k = SIZE-1;
        int first_zero = 1;
        for (int j = SIZE-1; j >= 0; j--) {
            if (first_zero) {
                if (board[i][j] == 0) {
                    k = j;
                    first_zero = 0;
                }
            }
            else {
                if (board[i][j] == 0 || j == k) {
                    continue;
                }
                else {
                    valid_move = 1;

                    //shift animation
                    shift(i*4+k-1, i*4+j-1);

                    //update the board status
                    board[i][k] = board[i][j];
                    board[i][j] = 0;
                    k --;
                }
            }
        }
    }

    for (int i = 0; i < SIZE; i++) {
        for (int j = 0; j < SIZE - 1; j++) {
            if (board[i][j] == board[i][j+1]) {
                valid_move = 1;

                //merge animation
                merge(i*4+j, i*4+j-1);

                //update the values in the board
                board[i][j+1] *= 2;
                board[i][j] = 0;
                

                //update the score
                score += board[i][j-1];

                //update the pointer to avoid double merge
                j = j+1;
            }
        }
    }
    
    if (!valid_move) {
        // no tiles has been changed, the move is invalid such the board cannot shift in this direction
        printf("invalid move \n");
        return;
    }

    generate_tile();
    render_frames();
}

void shift_up() {
    init_frame();
    int valid_move = 0;

    for (int j = 0; j < SIZE; j++) {
        int k = 0;
        int first_zero = 1;
        for (int i = 0; i < SIZE; i++) {
            if (first_zero) {
                if (board[i][j] == 0) {
                    k = i;
                    first_zero = 0;
                }
            }
            else {
                 if (board[i][j] == 0 || i == k) {
                    continue;
                }
                else {
                    //a valid move has been made
                    valid_move = 1;

                    //shift animation
                    shift(k*4+j-1, i*4+j-1);

                    //update the board
                    board[k][j] = board[i][j];
                    board[i][j] = 0;
                    k ++;
                }
            }
           
        }
    }

    for (int j = 0; j < SIZE; j++) {
        for (int i = SIZE-1; i > 0; i--) {
            if (board[i][j] == board[i-1][j]) {
                valid_move = 1;
                
                //merge animation
                merge((i-1)*4+j-1, i*4+j-1);

                //update the value of the board due to merge
                board[i-1][j] *= 2;
                board[i][j] = 0;

                //update score
                score += board[i-1][j];
                i = i-1;
            }
        }
    }
    if (!valid_move) {
        // no tiles has been changed, the move is invalid such the board cannot shift in this direction
        printf("invalid move \n");
        return;
    }
    generate_tile();
    render_frames();
}

void shift_down() {
    init_frame();
    int valid_move = 0;

    for (int j = 0; j < SIZE; j++) {
        int k = SIZE-1;
        int first_zero = 1;
        for (int i = SIZE-1; i >= 0; i--) {
            if (first_zero) {
                if (board[i][j] == 0) {
                    k = i;
                    first_zero = 0;
                }
            }
            else {
                if (board[i][j] == 0 || i == k) {
                    continue;
                }
                else {
                    valid_move = 1;
                    //shift animation
                    shift(k*4+j-1, i*4+j-1);
                    
                    board[k][j] = board[i][j];
                    board[i][j] = 0;
                    k --;
                }
            }
        }
    }

    for (int j = 0; j < SIZE; j++) {
        for (int i = 0; i < SIZE - 1; i++) {
            if (board[i][j] != 0 && board[i][j] == board[i+1][j]) {
                valid_move = 1;
                merge((i+1)*4+j-1, i*4+j-1);

                //update the board accordingly
                board[i+1][j] *= 2;
                
                board[i][j] = 0;

                //update score
                score += board[i+1][j];
                i = i + 1;
            }
        }
    }
    if (!valid_move) {
        // no tiles has been changed, the move is invalid such the board cannot shift in this direction
        printf("invalid move \n");
        return;
    }
    generate_tile();
    render_frames();
}


int game_over () {
    // return 1 if the game is over, 0 if not
    for (int i = 0; i < SIZE; i++) {
        for (int j = 0; j < SIZE; j++) {
            if (board[i][j] == 0) {
                return 0;
            }
            if (i != SIZE - 1) {
                if (board[i][j] == board[i+1][j]) {
                    return 0;
                }
            }
            if (j != SIZE - 1) {
                if (board[i][j] == board[i][j+1]) {
                    return 0;
                }
            }
        }
    }
    return 1;
}


void main() {
    // testing the code in c environment (test purpose only)
    init_game();
    int first_in = 1;
    while (game_on) {
        char in;
        if (first_in) {
            generate_tile();
            print_board();
        }
        first_in = 0;
        printf("Enter direction (u, d, l, r) to move the tile or q to quit game: ");
        scanf(" %c", &in);
        
        switch(in) {
            case 'q':
                game_on = 0;
                return;
            case 'u':
                shift_up();
                break;
            case 'd':
                shift_down();
                break;
            case 'l':
                shift_left();
                break;
            case 'r':
                shift_right();
                break;
            default:
                printf("Invalid input \n");
        }
        printf("current board: \n");
        print_board();
        if (game_over()) {
            restart();
            generate_tile();
            print_board();
        }
    }
}