/* * Device driver for the 2048 video & audio generator
 *
 * checkpatch.pl --file --no-tree game_driver.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "game_driver.h"

#define DRIVER_NAME "2048_driver"

/* Device registers */
#define TILE(x, i) ((x) + i * sizeof(tile_t))
#define CURRENT_SCORE(x) (TILE(x, 16))
#define BEST_SCORE(x) (CURRENT_SCORE(x) + sizeof(current_score_t))

/*
 * Information about our device
 */
struct game_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
        tile_t tiles[16];
		current_score_t current_score;
		best_score_t best_score;
} dev;

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
static void write_tile(tile_t *tile_i, int i)
{
	unsigned tile = (tile_i->x) + 
					(tile_i->y << 10) + 
					(tile_i->size << 19) + 
					(tile_i->type << 22);
	iowrite32(tile, TILE(dev.virtbase, i));
	dev.tiles[i] = *tile_i;
}

static void write_current_score(current_score_t *current_score)
{
	unsigned score_state_audio = (current_score->score) + 
								 (current_score->state << 20) + 
								 (current_score->audio << 22);
	iowrite32(score_state_audio, BEST_SCORE(dev.virtbase));
	dev.current_score = *current_score;
}

static void write_best_score(best_score_t *best_score)
{
	unsigned score = (best_score->score);
	iowrite32(score, BEST_SCORE(dev.virtbase));
	dev.best_score = *best_score;
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long game_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	game_arg_t ga;

	if (copy_from_user(&ga, (game_arg_t *) arg, sizeof(game_arg_t)))
		return -EACCES;

	switch (cmd) {
	case GAME_WRITE_TILES:
		for (int i = 0; i < 16; i++)
			write_tile(&ga.tiles[i], i);
		break;

	case GAME_WRITE_CURRENT_SCORE:
		write_current_score(&ga.current_score);
		break;

	case GAME_WRITE_BEST_SCORE:
		write_best_score(&ga.best_score);
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations game_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = game_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice game_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &game_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init game_probe(struct platform_device *pdev)
{
	current_score_t current_score = {0, 0, 0};
	best_score_t best_score = {0};

	int ret;

	/* Register ourselves as a misc device: creates /dev/game */
	ret = misc_register(&game_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	/* Set an initial score */
    write_current_score(&current_score);
	write_best_score(&best_score);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&game_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int game_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&game_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id game_of_match[] = {
	{ .compatible = "csee4840,game-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, game_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver game_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(game_of_match),
	},
	.remove	= __exit_p(game_remove),
};

/* Called when the module is loaded: set things up */
static int __init game_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&game_driver, game_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit game_exit(void)
{
	platform_driver_unregister(&game_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(game_init);
module_exit(game_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("VGA ball driver");
