/*
 * Avalon memory-mapped peripheral that generates VGA
 *
 * Stephen A. Edwards
 * Columbia University
 */

module sprite_rom (
  input wire logic      clk, rst,
  input logic [9:0]     vc, 
  input logic [10:0]	  hc, 
  input logic [19:0]    current_score, best_score,
  input logic [31:0]    tiles_0, tiles_1, tiles_2, tiles_3, 
                        tiles_4, tiles_5, tiles_6, tiles_7,
                        tiles_8, tiles_9, tiles_10, tiles_11,
                        tiles_12, tiles_13, tiles_14, tiles_15, 
  output logic          drawing, transparent,
  output logic [23:0]   pix_out
  );
  // parameter a = 323(vertical), AA = 382(horizantal,x2ed), b = 18, BB = 156;
  // parameter a = 191, AA = 1292, b = 78, BB = 36;
  // parameter a = 382, AA = 1292, b = 78, BB = 36, bg_x = 1, bg_y = 1;
  // parameter a = 382, AA = 1292, b = 34, BB = 12, bg_x = 1, bg_y = 51;
  // parameter start_x = 550, start_y = 51, STA = 92, st = 191;
  // parameter note_y_1 = 161, note_y_2 = 225, note_y_3 = 289, note_y_4 = 353;
  // parameter note_y_5 = 161, note_y_6 = 225, note_y_7 = 289, note_y_8 = 353;
  // parameter note_y_9 = 161, note_y_10 = 225, note_y_11 = 289, note_y_12 = 353;
  // parameter note_y_13 = 161, note_y_14 = 225, note_y_15 = 289, note_y_16 = 353;
  // //--------indicater
  // parameter hm = 30, HMM = 60, ind = 35, INDD = 50;
  // parameter hm_x = 1075, hm_y_1 = 162, hm_y_2 =226, hm_y_3 = 290, hm_y_4 = 354;
  // parameter indi_x = 1020, indi_y_1 = 160, indi_y_2 = 224, indi_y_3 = 288, indi_y_4 = 352;
  // parameter finalline_x = 1101, finalline_y = 99, FLL = 8, fl = 286;
  // //--------num
  // parameter num_x = 580;
  // parameter num_y_3 = 258, num_y_2 = 223, num_y_1 = 188, num_y_0 = 153;
  // parameter num_height = 13, num_width = 15 ; // h:13 w:15 

  logic         tile_active, frame_finish, num_active;
  logic [1:0]   num_0_output, num_1_output, num_2_output, num_3_output, 
                num_4_output, num_5_output, num_6_output, num_7_output, 
                num_8_output, num_9_output;
  logic [2:0]   tile_2_output, tile_4_output, tile_8_output, tile_16_output,
                tile_32_output, tile_64_output, tile_128_output,
                tile_256_output, tile_512_output, tile_1024_output, 
                tile_2048_output, tile_4096_output, tile_8192_output, 
                tile_16384_output, tile_32768_output, tile_65536_output;
  logic [3:0]   bg_output, digit, tile_type;
  logic [7:0]   color_addr;
  logic [11:0]  num_addr;
  logic [12:0]  tile_addr;
  logic [18:0]  bg_addr;
  logic [23:0]  color_output;
  
  // sprite state machine
  enum {
    IDLE,      // screen
    WAIT,
    DRAW_BG,
    DRAW_TILE,
    RESET,
    DRAW_NUM
  } state;
  
  tile_2 tile2(.address(tile_addr),.clock(clk),.q(tile_2_output));
  tile_4 tile4(.address(tile_addr),.clock(clk),.q(tile_4_output));
  tile_8 tile8(.address(tile_addr),.clock(clk),.q(tile_8_output));
  tile_16 tile16(.address(tile_addr),.clock(clk),.q(tile_16_output));
  tile_32 tile32(.address(tile_addr),.clock(clk),.q(tile_32_output));
  tile_64 tile64(.address(tile_addr),.clock(clk),.q(tile_64_output));
  tile_128 tile128(.address(tile_addr),.clock(clk),.q(tile_128_output));
  tile_256 tile256(.address(tile_addr),.clock(clk),.q(tile_256_output));
  tile_512 tile512(.address(tile_addr),.clock(clk),.q(tile_512_output));
  tile_1024 tile1024(.address(tile_addr),.clock(clk),.q(tile_1024_output));
  tile_2048 tile2048(.address(tile_addr),.clock(clk),.q(tile_2048_output));
  tile_4096 tile4096(.address(tile_addr),.clock(clk),.q(tile_4096_output));
  tile_8192 tile8192(.address(tile_addr),.clock(clk),.q(tile_8192_output));
  tile_16384 tile16384(.address(tile_addr),.clock(clk),.q(tile_16384_output));
  tile_32768 tile32768(.address(tile_addr),.clock(clk),.q(tile_32768_output));
  tile_65536 tile65536(.address(tile_addr),.clock(clk),.q(tile_65536_output));

  background bg(.address(bg_addr),.clock(clk),.q(bg_output));

  num_0 num0(.address(num_addr),.clock(clk),.q(num_0_output));
  num_1 num1(.address(num_addr),.clock(clk),.q(num_1_output));
  num_2 num2(.address(num_addr),.clock(clk),.q(num_2_output));
  num_3 num3(.address(num_addr),.clock(clk),.q(num_3_output));
  num_4 num4(.address(num_addr),.clock(clk),.q(num_4_output));
  num_5 num5(.address(num_addr),.clock(clk),.q(num_5_output));
  num_6 num6(.address(num_addr),.clock(clk),.q(num_6_output));
  num_7 num7(.address(num_addr),.clock(clk),.q(num_7_output));
  num_8 num8(.address(num_addr),.clock(clk),.q(num_8_output));
  num_9 num9(.address(num_addr),.clock(clk),.q(num_9_output));

  color cl(.address(color_addr), .clock(clk), .q(color_output));

  always_comb begin
  
    // TODO:
    /* 
     * 45 is half of our default tile size, how to implement the scale of tiles
     * remains to be determined, after we figure out how to do that, just 
     * replace all '45'
     */

    tile_active = ((hc >= tiles_0[9:0] - 45 && hc <= tiles_0[9:0] + 45 
                 && vc >= tiles_0[18:10] - 45 && vc <= tiles_0[18:10] + 45) || 

                   (hc >= tiles_1[9:0] - 45 && hc <= tiles_1[9:0] + 45 
                 && vc >= tiles_1[18:10] - 45 && vc <= tiles_1[18:10] + 45) || 

                   (hc >= tiles_2[9:0] - 45 && hc <= tiles_2[9:0] + 45 
                 && vc >= tiles_2[18:10] - 45 && vc <= tiles_2[18:10] + 45) || 

                   (hc >= tiles_3[9:0] - 45 && hc <= tiles_3[9:0] + 45 
                 && vc >= tiles_3[18:10] - 45 && vc <= tiles_3[18:10] + 45) || 

                   (hc >= tiles_4[9:0] - 45 && hc <= tiles_4[9:0] + 45 
                 && vc >= tiles_4[18:10] - 45 && vc <= tiles_4[18:10] + 45) ||

                   (hc >= tiles_5[9:0] - 45 && hc <= tiles_5[9:0] + 45 
                 && vc >= tiles_5[18:10] - 45 && vc <= tiles_5[18:10] + 45) || 

                   (hc >= tiles_6[9:0] - 45 && hc <= tiles_6[9:0] + 45 
                 && vc >= tiles_6[18:10] - 45 && vc <= tiles_6[18:10] + 45) || 

                   (hc >= tiles_7[9:0] - 45 && hc <= tiles_7[9:0] + 45 
                 && vc >= tiles_7[18:10] - 45 && vc <= tiles_7[18:10] + 45) || 

                   (hc >= tiles_8[9:0] - 45 && hc <= tiles_8[9:0] + 45 
                 && vc >= tiles_8[18:10] - 45 && vc <= tiles_8[18:10] + 45) ||

                   (hc >= tiles_9[9:0] - 45 && hc <= tiles_9[9:0] + 45 
                 && vc >= tiles_9[18:10] - 45 && vc <= tiles_9[18:10] + 45) || 

                   (hc >= tiles_10[9:0] - 45 && hc <= tiles_10[9:0] + 45 
                 && vc >= tiles_10[18:10] - 45 && vc <= tiles_10[18:10] + 45) || 

                   (hc >= tiles_11[9:0] - 45 && hc <= tiles_11[9:0] + 45 
                 && vc >= tiles_11[18:10] - 45 && vc <= tiles_11[18:10] + 45) || 

                   (hc >= tiles_12[9:0] - 45 && hc <= tiles_12[9:0] + 45 
                 && vc >= tiles_12[18:10] - 45 && vc <= tiles_12[18:10] + 45) ||

                   (hc >= tiles_13[9:0] - 45 && hc <= tiles_13[9:0] + 45 
                 && vc >= tiles_13[18:10] - 45 && vc <= tiles_13[18:10] + 45) || 

                   (hc >= tiles_14[9:0] - 45 && hc <= tiles_14[9:0] + 45 
                 && vc >= tiles_14[18:10] - 45 && vc <= tiles_14[18:10] + 45) || 

                   (hc >= tiles_15[9:0] - 45 && hc <= tiles_15[9:0] + 45 
                 && vc >= tiles_15[18:10] - 45 && vc <= tiles_15[18:10] + 45));

    frame_finish = hc == 1280 && vc == 480; // maybe need modify? 
     
    num_active = (hc >= 400 && hc <= 640 &&
                  vc >= 176 && vc <= 224 ||
                 
                  hc >= 400 && hc <= 640 &&
                  vc >= 336 && vc <= 384);
  end

  always_ff @(posedge clk) begin
   pix_out <= color_output;
    if (rst) begin
      state <= IDLE;
      color_addr <= 0;
      drawing <= 0;
      transparent <= 0;
      bg_addr <= 0;
      tile_addr <= 0;
      num_addr <= 0;
    end
    else begin
      case (state)
        IDLE: begin
          if (tile_active || num_active) begin
            state <= WAIT;
          end else begin
            state <= IDLE;
          end
        end

        RESET: begin
          state <= IDLE;
          color_addr <= 0;
          drawing <= 0;
          transparent <= 0;
          bg_addr <= 0;
          tile_addr <= 0; 
          num_addr <= 0;
        end

        WAIT: begin
          if(frame_finish) state <= RESET;
          else begin
            if (tile_active) begin
              state <= DRAW_TILE;
            end
            else if (num_active) begin
              state <= DRAW_NUM;
            end
            else begin
              state <= DRAW_BG;
            end
          end
          pix_out <= 0;
          drawing <= 0;
          transparent <= 0;
        end
        // DRAW_NUM: begin
      //  if(!num_active) state <= WAIT;
      //  else state <= DRAW_NUM;

        // Determine which digit to draw based on hc and vc values
      //  if (vc >= 176 && vc <= 224) begin
        // Current score display
     //   if (hc >= 400 && hc <= 640) begin
      //  num_addr = (hc - 400) + ((vc - 176) << 5);
      // case (hc)
      //  400: digit = (current_score & 20'hf0000) >> 16;
      //  448: digit = (current_score & 20'h0f000) >> 12;
      //  496: digit = (current_score & 20'h00f00) >> 8;
      //  544: digit = (current_score & 20'h000f0) >> 4;
      //  592: digit = (current_score & 20'h0000f);
      // endcase
      // end
      // end else if (vc >= 336 && vc <= 384) begin
       // Best score display
      // if (hc >= 400 && hc <= 640) begin
      // num_addr = (hc - 400) + ((vc - 336) << 5);
     // case (hc)
       //  400: digit = (best_score & 20'hf0000) >> 16;
       //  448: digit = (best_score & 20'h0f000) >> 12;
       //  496: digit = (best_score & 20'h00f00) >> 8;
       //  544: digit = (best_score & 20'h000f0) >> 4;
       //  592: digit = (best_score & 20'h0000f);
      //  endcase
     //  end
       //end

        // Map digit to color address
      // case(digit)
      // 0: color_addr = num_0_output + 144;
      // 1: color_addr = num_1_output + 144;
      // 2: color_addr = num_2_output + 144;
      // 3: color_addr = num_3_output + 144;
      // 4: color_addr = num_4_output + 144;
      // 5: color_addr = num_5_output + 144;
      // 6: color_addr = num_6_output + 144;
      // 7: color_addr = num_7_output + 144;
      // 8: color_addr = num_8_output + 144;
     //  9: color_addr = num_9_output + 144;
      // endcase
      // drawing <= 1;
      // transparent <= (color_addr == 144);
      // end

        DRAW_BG:begin
          if (tile_active) begin
            state <= DRAW_TILE;
          end
          else if (num_active) begin
            state <= DRAW_NUM;
          end
          else begin
            state <= DRAW_BG;
          end
          bg_addr <= hc[10:1] + ((vc[9:0] << 9) + (vc[9:0] << 7));
          drawing <= 1;
          color_addr = bg_output + 128;
          if (color_addr == 128) begin
            transparent = 1;
          end
          else begin
            transparent = 0;
          end
        end

        DRAW_TILE:begin
          if(!num_active) state <= WAIT;
          else state <= DRAW_NUM;

          if ((hc >= tiles_0[9:0] - 45 && hc <= tiles_0[9:0] + 45 
            && vc >= tiles_0[18:10] - 45 && vc <= tiles_0[18:10] + 45)) begin
            tile_addr = (hc - (tiles_0[9:0] - 45)) + 
                        (vc - (tiles_0[18:10] - 45)) << 6 + 
                        (vc - (tiles_0[18:10] - 45)) << 4 + 
                        (vc - (tiles_0[18:10] - 45)) << 3 + 
                        (vc - (tiles_0[18:10] - 45)) << 1;
            tile_type = tiles_0[25:22];
          end
          else if ((hc >= tiles_1[9:0] - 45 && hc <= tiles_1[9:0] + 45 
            && vc >= tiles_1[18:10] - 45 && vc <= tiles_1[18:10] + 45)) begin
            tile_addr = (hc - (tiles_1[9:0] - 45)) + 
                        (vc - (tiles_1[18:10] - 45)) << 6 + 
                        (vc - (tiles_1[18:10] - 45)) << 4 + 
                        (vc - (tiles_1[18:10] - 45)) << 3 + 
                        (vc - (tiles_1[18:10] - 45)) << 1;
            tile_type = tiles_1[25:22];
          end
          else if ((hc >= tiles_2[9:0] - 45 && hc <= tiles_2[9:0] + 45 
            && vc >= tiles_2[18:10] - 45 && vc <= tiles_2[18:10] + 45)) begin
            tile_addr = (hc - (tiles_2[9:0] - 45)) + 
                        (vc - (tiles_2[18:10] - 45)) << 6 + 
                        (vc - (tiles_2[18:10] - 45)) << 4 + 
                        (vc - (tiles_2[18:10] - 45)) << 3 + 
                        (vc - (tiles_2[18:10] - 45)) << 1;
            tile_type = tiles_2[25:22];
          end
          else if ((hc >= tiles_3[9:0] - 45 && hc <= tiles_3[9:0] + 45 
            && vc >= tiles_3[18:10] - 45 && vc <= tiles_3[18:10] + 45)) begin
            tile_addr = (hc - (tiles_3[9:0] - 45)) + 
                        (vc - (tiles_3[18:10] - 45)) << 6 + 
                        (vc - (tiles_3[18:10] - 45)) << 4 + 
                        (vc - (tiles_3[18:10] - 45)) << 3 + 
                        (vc - (tiles_3[18:10] - 45)) << 1;
            tile_type = tiles_3[25:22];
          end
          else if ((hc >= tiles_4[9:0] - 45 && hc <= tiles_4[9:0] + 45 
            && vc >= tiles_4[18:10] - 45 && vc <= tiles_4[18:10] + 45)) begin
            tile_addr = (hc - (tiles_4[9:0] - 45)) + 
                        (vc - (tiles_4[18:10] - 45)) << 6 + 
                        (vc - (tiles_4[18:10] - 45)) << 4 + 
                        (vc - (tiles_4[18:10] - 45)) << 3 + 
                        (vc - (tiles_4[18:10] - 45)) << 1;
            tile_type = tiles_4[25:22];
          end
          else if ((hc >= tiles_5[9:0] - 45 && hc <= tiles_5[9:0] + 45 
            && vc >= tiles_5[18:10] - 45 && vc <= tiles_5[18:10] + 45)) begin
            tile_addr = (hc - (tiles_5[9:0] - 45)) + 
                        (vc - (tiles_5[18:10] - 45)) << 6 + 
                        (vc - (tiles_5[18:10] - 45)) << 4 + 
                        (vc - (tiles_5[18:10] - 45)) << 3 + 
                        (vc - (tiles_5[18:10] - 45)) << 1;
            tile_type = tiles_5[25:22];
          end
          else if ((hc >= tiles_6[9:0] - 45 && hc <= tiles_6[9:0] + 45 
            && vc >= tiles_6[18:10] - 45 && vc <= tiles_6[18:10] + 45)) begin
            tile_addr = (hc - (tiles_6[9:0] - 45)) + 
                        (vc - (tiles_6[18:10] - 45)) << 6 + 
                        (vc - (tiles_6[18:10] - 45)) << 4 + 
                        (vc - (tiles_6[18:10] - 45)) << 3 + 
                        (vc - (tiles_6[18:10] - 45)) << 1;
            tile_type = tiles_6[25:22];
          end
          else if ((hc >= tiles_7[9:0] - 45 && hc <= tiles_7[9:0] + 45 
            && vc >= tiles_7[18:10] - 45 && vc <= tiles_7[18:10] + 45)) begin
            tile_addr = (hc - (tiles_7[9:0] - 45)) + 
                        (vc - (tiles_7[18:10] - 45)) << 6 + 
                        (vc - (tiles_7[18:10] - 45)) << 4 + 
                        (vc - (tiles_7[18:10] - 45)) << 3 + 
                        (vc - (tiles_7[18:10] - 45)) << 1;
            tile_type = tiles_7[25:22];
          end
          else if ((hc >= tiles_8[9:0] - 45 && hc <= tiles_8[9:0] + 45 
            && vc >= tiles_8[18:10] - 45 && vc <= tiles_8[18:10] + 45)) begin
            tile_addr = (hc - (tiles_8[9:0] - 45)) + 
                        (vc - (tiles_8[18:10] - 45)) << 6 + 
                        (vc - (tiles_8[18:10] - 45)) << 4 + 
                        (vc - (tiles_8[18:10] - 45)) << 3 + 
                        (vc - (tiles_8[18:10] - 45)) << 1;
            tile_type = tiles_8[25:22];
          end
          else if ((hc >= tiles_9[9:0] - 45 && hc <= tiles_9[9:0] + 45 
            && vc >= tiles_9[18:10] - 45 && vc <= tiles_9[18:10] + 45)) begin
            tile_addr = (hc - (tiles_9[9:0] - 45)) + 
                        (vc - (tiles_9[18:10] - 45)) << 6 + 
                        (vc - (tiles_9[18:10] - 45)) << 4 + 
                        (vc - (tiles_9[18:10] - 45)) << 3 + 
                        (vc - (tiles_9[18:10] - 45)) << 1;
            tile_type = tiles_9[25:22];
          end
          else if ((hc >= tiles_10[9:0] - 45 && hc <= tiles_10[9:0] + 45 
            && vc >= tiles_10[18:10] - 45 && vc <= tiles_10[18:10] + 45)) begin
            tile_addr = (hc - (tiles_10[9:0] - 45)) + 
                        (vc - (tiles_10[18:10] - 45)) << 6 + 
                        (vc - (tiles_10[18:10] - 45)) << 4 + 
                        (vc - (tiles_10[18:10] - 45)) << 3 + 
                        (vc - (tiles_10[18:10] - 45)) << 1;
            tile_type = tiles_10[25:22];
          end
          else if ((hc >= tiles_11[9:0] - 45 && hc <= tiles_11[9:0] + 45 
            && vc >= tiles_11[18:10] - 45 && vc <= tiles_11[18:10] + 45)) begin
            tile_addr = (hc - (tiles_11[9:0] - 45)) + 
                        (vc - (tiles_11[18:10] - 45)) << 6 + 
                        (vc - (tiles_11[18:10] - 45)) << 4 + 
                        (vc - (tiles_11[18:10] - 45)) << 3 + 
                        (vc - (tiles_11[18:10] - 45)) << 1;
            tile_type = tiles_11[25:22];
          end
          else if ((hc >= tiles_12[9:0] - 45 && hc <= tiles_12[9:0] + 45 
            && vc >= tiles_12[18:10] - 45 && vc <= tiles_12[18:10] + 45)) begin
            tile_addr = (hc - (tiles_12[9:0] - 45)) + 
                        (vc - (tiles_12[18:10] - 45)) << 6 + 
                        (vc - (tiles_12[18:10] - 45)) << 4 + 
                        (vc - (tiles_12[18:10] - 45)) << 3 + 
                        (vc - (tiles_12[18:10] - 45)) << 1;
            tile_type = tiles_12[25:22];
          end
          else if ((hc >= tiles_13[9:0] - 45 && hc <= tiles_13[9:0] + 45 
            && vc >= tiles_13[18:10] - 45 && vc <= tiles_13[18:10] + 45)) begin
            tile_addr = (hc - (tiles_13[9:0] - 45)) + 
                        (vc - (tiles_13[18:10] - 45)) << 6 + 
                        (vc - (tiles_13[18:10] - 45)) << 4 + 
                        (vc - (tiles_13[18:10] - 45)) << 3 + 
                        (vc - (tiles_13[18:10] - 45)) << 1;
            tile_type = tiles_13[25:22];
          end
          else if ((hc >= tiles_14[9:0] - 45 && hc <= tiles_14[9:0] + 45 
            && vc >= tiles_14[18:10] - 45 && vc <= tiles_14[18:10] + 45)) begin
            tile_addr = (hc - (tiles_14[9:0] - 45)) + 
                        (vc - (tiles_14[18:10] - 45)) << 6 + 
                        (vc - (tiles_14[18:10] - 45)) << 4 + 
                        (vc - (tiles_14[18:10] - 45)) << 3 + 
                        (vc - (tiles_14[18:10] - 45)) << 1;
            tile_type = tiles_14[25:22];
          end
          else if ((hc >= tiles_15[9:0] - 45 && hc <= tiles_15[9:0] + 45 
            && vc >= tiles_15[18:10] - 45 && vc <= tiles_15[18:10] + 45)) begin
            tile_addr = (hc - (tiles_15[9:0] - 45)) + 
                        (vc - (tiles_15[18:10] - 45)) << 6 + 
                        (vc - (tiles_15[18:10] - 45)) << 4 + 
                        (vc - (tiles_15[18:10] - 45)) << 3 + 
                        (vc - (tiles_15[18:10] - 45)) << 1;
            tile_type = tiles_15[25:22];
          end

          case(tile_type)
            0: begin
              color_addr = tile_2_output;
            end
            1: begin
              color_addr = tile_4_output + 8;
            end
            2: begin
              color_addr = tile_8_output + 16;
            end
            3: begin
              color_addr = tile_16_output + 24;
            end
            4: begin
              color_addr = tile_32_output + 32;
            end
            5: begin
              color_addr = tile_64_output + 40;
            end
            6: begin
              color_addr = tile_128_output + 48;
            end
            7: begin
              color_addr = tile_256_output + 56;
            end
            8: begin
              color_addr = tile_512_output + 64;
            end
            9: begin
              color_addr = tile_1024_output + 72;
            end
            10: begin
              color_addr = tile_2048_output + 80;
            end
            11: begin
              color_addr = tile_4096_output + 88;
            end
            12: begin
              color_addr = tile_8192_output + 96;
            end
            13: begin
              color_addr = tile_16384_output + 104;
            end
            14: begin
              color_addr = tile_32768_output + 112;
            end
            15: begin
              color_addr = tile_65536_output + 120;
            end
          endcase
          drawing = 1;
          if (color_addr == 0 || color_addr == 8 || 
              color_addr == 16 || color_addr == 24 ||
              color_addr == 32 || color_addr == 40 || 
              color_addr == 48 || color_addr == 56 ||
              color_addr == 64 || color_addr == 72 || 
              color_addr == 80 || color_addr == 88 ||
              color_addr == 96 || color_addr == 104 || 
              color_addr == 112 || color_addr == 120) begin
            transparent = 1;
          end
          else begin
            transparent = 0;
          end
        end
        default: begin
          state <= IDLE;
        end
      endcase
    end
  end
endmodule

module game(input logic         clk,
	          input logic 	      reset,
		        input logic [31:0]  writedata,
	        	input logic 	      write,
            input 		          chipselect,
            input logic [4:0]   address,

            output logic [7:0]  VGA_R, VGA_G, VGA_B,
            output logic 	      VGA_CLK, VGA_HS, VGA_VS,
                                VGA_BLANK_n,
            output logic 	      VGA_SYNC_n
);

  logic [10:0]    hcount;
  logic [9:0]     vcount;
  logic [7:0] 	  background_r, background_g, background_b;

  vga_counters counters(.clk50(clk), .*);

  logic           drawing, transparent;
  logic [19:0]    current_score, best_score;
  logic [23:0]    pix_out;
  logic [31:0]    tiles_0, tiles_1, tiles_2, tiles_3, 
                  tiles_4, tiles_5, tiles_6, tiles_7,
                  tiles_8, tiles_9, tiles_10, tiles_11,
                  tiles_12, tiles_13, tiles_14, tiles_15;

  sprite_rom sr(.clk(clk), .rst(reset), .hc(hcount), .vc(vcount), 
                .tiles_0(tiles_0), .tiles_1(tiles_1), .tiles_2(tiles_2), 
                .tiles_3(tiles_3), .tiles_4(tiles_4), .tiles_5(tiles_5), 
                .tiles_6(tiles_6), .tiles_7(tiles_7), .tiles_8(tiles_8), 
                .tiles_9(tiles_9), .tiles_10(tiles_10), .tiles_11(tiles_11), 
                .tiles_12(tiles_12), .tiles_13(tiles_13), .tiles_14(tiles_14), 
                .tiles_15(tiles_15), .current_score(current_score), 
                .best_score(best_score),

                .drawing(drawing), .transparent(transparent), .pix_out(pix_out));

  always_ff @(posedge clk) begin
    if (reset) begin
      background_r <= 8'h00;
      background_g <= 8'h00;
      background_b <= 8'h00;
      tiles_0 <= 32'h0;
      tiles_1 <= 32'h0;
      tiles_2 <= 32'h0;
      tiles_3 <= 32'h0;
      tiles_4 <= 32'h0;
      tiles_5 <= 32'h0;
      tiles_6 <= 32'h0;
      tiles_7 <= 32'h0;
      tiles_8 <= 32'h0;
      tiles_9 <= 32'h0;
      tiles_10 <= 32'h0;
      tiles_11 <= 32'h0;
      tiles_12 <= 32'h0;
      tiles_13 <= 32'h0;
      tiles_14 <= 32'h0;
      tiles_15 <= 32'h0;
      current_score <= 0;
      best_score <= 0;
    end
    else if (chipselect && write) begin
      case (address)
        6'h0: tiles_0 <= writedata;
        6'h1: tiles_1 <= writedata;
        6'h2: tiles_2 <= writedata;
        6'h3: tiles_3 <= writedata;
        6'h4: tiles_4 <= writedata;
        6'h5: tiles_5 <= writedata;
        6'h6: tiles_6 <= writedata;
        6'h7: tiles_7 <= writedata;
        6'h8: tiles_8 <= writedata;
        6'h9: tiles_9 <= writedata;
        6'ha: tiles_10 <= writedata;
        6'hb: tiles_11 <= writedata;
        6'hc: tiles_12 <= writedata;
        6'hd: tiles_13 <= writedata;
        6'he: tiles_14 <= writedata;
        6'hf: tiles_15 <= writedata;
        6'h10: begin
          current_score <= writedata[19:0];

          // TODO:
          /* 
           * game state and audio still not used, let's take 'breakout' as 
           * reference!
           */

        end
        6'h11: best_score <= writedata[19:0];
      endcase
    end
  end
    
  always_comb begin
    {VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0};
    if (VGA_BLANK_n && !transparent) begin
      if (drawing) begin
        {VGA_R, VGA_G, VGA_B} = pix_out;
      end
      else begin
        {VGA_R, VGA_G, VGA_B} ={background_r, background_g, background_b};
      end
    end
  end
endmodule

module vga_counters(
 input logic 	     clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;
       
   logic endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule
