#include <stdio.h>
#include <stdlib.h>

#include "usbgamepad.h"

#define VID 121 // Vendor ID for the device
#define PID 17 // Product ID for the device

// function to initialize and open gamepad
struct libusb_device_handle *opengamepad(uint8_t *endpoint_address) {
    libusb_device **devs;
    ssize_t num_devs;
    struct libusb_device_descriptor desc;
    struct libusb_device_handle *gamepad = NULL;
    int retval;

    // initialize the libusb library
    retval = libusb_init(NULL);
    if (retval < 0) {
        fprintf(stderr, "libusb_init error: %d\n", retval);
        exit(1);
    }

    // enumerate all the attached USB devices
    num_devs = libusb_get_device_list(NULL, &devs);
    if (num_devs < 0) {
        fprintf(stderr, "Error: libusb_get_device_list failed\n");
        exit(1);
    }

    // only care about the first and only device, confirm this is the expected controller
    libusb_device *dev = devs[0];
    if (libusb_get_device_descriptor(dev, &desc) < 0) 
    {
        fprintf(stderr, "Error: libusb_get_device_descriptor failed\n");
        exit(1);
    }

    if (!((desc.idVendor == VID) && (desc.idProduct == PID))) {
        fprintf(stderr, "Error connecting to gamepad. Unexpected Vendor/Product IDs.\n");
        exit(1);
    }

    // obtain the device address of a functional HID class interface device
    if (desc.bDeviceClass == LIBUSB_CLASS_PER_INTERFACE) 
    {
        struct libusb_config_descriptor *config;
        libusb_get_config_descriptor(dev, 0, &config);
        const struct libusb_interface_descriptor *inter = config->interface[0].altsetting;

        // not checking here if speaking keyboard protocol, but checking if HID class interface
        if (inter->bInterfaceClass == LIBUSB_CLASS_HID)
        {
            // open the usb device
            retval = libusb_open(dev, &gamepad);
	        if (retval != 0) 
            {
                fprintf(stderr, "Error: libusb_open failed: %d\n", retval);
                exit(1);
            }

            // detatch the kernel driver if one is attached to the device
            if (libusb_kernel_driver_active(gamepad, 0))
            {
                libusb_detach_kernel_driver(gamepad, 0);
            }
            libusb_set_auto_detach_kernel_driver(gamepad, 0);

            // claim the device interface
            retval = libusb_claim_interface(gamepad, 0);
            if (retval != 0) 
            {
                fprintf(stderr, "Error: libusb_claim_interface failed: %d\n", retval);
                exit(1);
            }

            // obtain the device endpoint address
            *endpoint_address = inter->endpoint[0].bEndpointAddress;
            libusb_free_device_list(devs, 1);
        }
    }

    return gamepad;
}

// function to read and convert raw gamepad usb data to buttons pressed
char gamepad_state(struct libusb_device_handle *gamepad, uint8_t endpoint_address) {
    int libusb_retval;
    unsigned char data[8];
    int len = 0;
    enum gamepad_state state = NONE;

    // obtain data from the usb device; non-blocking
    libusb_retval = libusb_interrupt_transfer(gamepad, endpoint_address, data, sizeof(data), &len, 0);
    if (libusb_retval < 0) 
    {
        fprintf(stderr, "libusb_interrupt_transfer error %d\n", libusb_retval);
        libusb_release_interface(gamepad, 0);
        libusb_close(gamepad);
        libusb_exit(NULL);
        exit(1);
    }

    // convert buttons pressed on gamepad to gamepad_state
    //  buttons have priority when multiple are clicked together; see gamepad_state enum    
    if (data[6] & 32) {
        state = START;
    } else if (data[6] == 16) {
        state = SELECT;
    } else if ((data[3] == 0) && (data[4] == 255)) {
        state = DOWN_LEFT;
    } else if ((data[3] == 255) && (data[4] == 255)) {
        state = DOWN_RIGHT;
    } else if (data[3] == 0) {
        state = LEFT;
    } else if (data[3] == 255) {
        state = RIGHT;
    } else if (data[4] == 0) {
        state = TOP;
    } else if (data[4] == 255) {
        state = DOWN;
    } else if (data[5] & 32) {
        state = A;
    } else if (data[5] & 64) {
        state = B;
    } else {
        state = NONE;
    }

    return state;
}