/*
 * Avalon memory-mapped peripheral that generates VGA
* gic [29:0
 *
 * Stephen A. Edwards
 * Columbia University
 */

module vga_ball(input logic        clk,
	        input logic 	   reset,
		input logic [31:0]  writedata,
		input logic 	   write,
		input 		   chipselect,
		input logic [5:0]  address,

		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output logic 	   VGA_CLK, VGA_HS, VGA_VS,
		                   VGA_BLANK_n,
		output logic 	   VGA_SYNC_n);

   logic [10:0]	   hcount;
   logic [9:0]     vcount;
   logic [7:0]     background_r, background_g, background_b;
	logic[31:0] background_colors, falling_reg, falling_v_reg, falling_h_reg, tetro_next;
   	logic[31:0]    back0, back1, back2, back3, back4, back5, back6, back7, back8, back9;
   logic [31:0]    back10, back11, back12, back13, back14, back15, back16, back17, back18, back19;
   logic [399:0] rows;

   logic [7:0]     backnext;
   logic [7:0]    falling_sprite, next_sprite;
   logic [9:0]     falling_v;
   logic [9:0]     falling_h;
   logic [1:0]     falling_ori;
   logic [31:0]    curr_score;
   logic [31:0]    high_score;
   logic [3:0]     music_sound;
   logic           music_en;
   logic [0:39] v_start;
   logic [0:29] h_start;

	//system nets
	logic [23:0] assigned_color, block_color;
	logic [15:0] num; 
	logic stitch;
   	logic v_line, v_on;
  	logic h_line, h_on;
	
	//yellow tetromino bitmap, //2D array for 20 pixel per line for 20 lines
	logic [0:19] yell [20];

	//block bitmap ROM
	logic [8:0] block_rom_addr;
	logic [19:0] row;
	logic [1:0] block;

   //create sprite bitmap ROM
   localparam SPR_ROM_DEPTH = (80*80); //sprite width = 80px, sprite height = 80px
   logic [12:0] spr_rom_addr, next_rom_addr; //pixel position
   logic [7:0] spr_rom_data, block_rom_data, next_rom_data; //pixel color
	//logic [7:0] yellow_data, test_data;
	logic [7:0] i1_data, i2_data, i3_data, i4_data;
	logic [7:0] o_data; 
        logic [7:0] t1_data, t2_data, t3_data, t4_data;
        logic [7:0] s1_data, s2_data, s3_data, s4_data;
	logic [7:0] z1_data, z2_data, z3_data, z4_data;
        logic [7:0] j1_data, j2_data, j3_data, j4_data;
        logic [7:0] l1_data, l2_data, l3_data, l4_data;
	logic [7:0] white_data, yellow_data, blue_data;
	logic [7:0] i1N_data, oN_data, t1N_data, s1N_data, z1N_data, j1N_data,l1N_data;

	//develop synchronous memory for all of the sprites
	rom_sync #(
		.INIT_F("i1.mem")
	) i1_rom (clk, spr_rom_addr, i1_data);

	rom_sync #(
		.INIT_F("i2.mem")) i2_rom (clk, spr_rom_addr, i2_data); 
	rom_sync #(
		.INIT_F("i3.mem")
	) i3_rom (clk, spr_rom_addr, i3_data);

	rom_sync #(
		.INIT_F("i4.mem")
	) i4_rom (clk, spr_rom_addr, i4_data);

	rom_sync #(
		.INIT_F("t1.mem")
	) t1_rom (clk, spr_rom_addr, t1_data);

	rom_sync #(
		.INIT_F("t2.mem")
	) t2_rom (clk, spr_rom_addr, t2_data);

	rom_sync #(
		.INIT_F("t3.mem")
	) t3_rom (clk, spr_rom_addr, t3_data);

	rom_sync #(
		.INIT_F("t4.mem")
	) t4_rom (clk, spr_rom_addr, t4_data);

	rom_sync #(
		.INIT_F("o1.mem")
	) o_rom (clk, spr_rom_addr, o_data);

	rom_sync #(
		.INIT_F("s1.mem")
	) s1_rom (clk, spr_rom_addr, s1_data);

	rom_sync #(
		.INIT_F("s2.mem")
	) s2_rom (clk, spr_rom_addr, s2_data);

	rom_sync #(
		.INIT_F("s3.mem")
	) s3_rom (clk, spr_rom_addr, s3_data);

	rom_sync #(
		.INIT_F("s4.mem")
	) s4_rom (clk, spr_rom_addr, s4_data);

	rom_sync #(
		.INIT_F("z1.mem")
	) z1_rom (clk, spr_rom_addr, z1_data);

	rom_sync #(
		.INIT_F("z2.mem")
	) z2_rom (clk, spr_rom_addr, z2_data);

	rom_sync #(
		.INIT_F("z3.mem")
	) z3_rom (clk, spr_rom_addr, z3_data);

	rom_sync #(
		.INIT_F("z4.mem")
	) z4_rom (clk, spr_rom_addr, z4_data);

	rom_sync #(
		.INIT_F("j1.mem")
	) j1_rom (clk, spr_rom_addr, j1_data);

	rom_sync #(
		.INIT_F("j2.mem")
	) j2_rom (clk, spr_rom_addr, j2_data);

	rom_sync #(
		.INIT_F("j3.mem")
	) j3_rom (clk, spr_rom_addr, j3_data);

	rom_sync #(
		.INIT_F("j4.mem")
	) j4_rom (clk, spr_rom_addr, j4_data);

	rom_sync #(
		.INIT_F("l1.mem")
	) l1_rom (clk, spr_rom_addr, l1_data);

	rom_sync #(
		.INIT_F("l2.mem")
	) l2_rom (clk, spr_rom_addr, l2_data);

	rom_sync #(
		.INIT_F("l3.mem")
	) l3_rom (clk, spr_rom_addr, l3_data);

	rom_sync #(
		.INIT_F("l4.mem")
	) l4_rom (clk, spr_rom_addr, l4_data);

	//memory for next block
	rom_sync #(
		.INIT_F("i1.mem")
	) i1_rom2 (clk, next_rom_addr, i1N_data);

	rom_sync #(
		.INIT_F("t1N.mem")
	) t1_rom2 (clk, next_rom_addr, t1N_data);

	rom_sync #(
		.INIT_F("o1N.mem")
	) o_rom2 (clk, next_rom_addr, oN_data);

	rom_sync #(
		.INIT_F("s1.mem")
	) s1_rom2 (clk, next_rom_addr, s1N_data);

	rom_sync #(
		.INIT_F("z1.mem")
	) z1_rom2 (clk, next_rom_addr, z1N_data);

	rom_sync #(
		.INIT_F("j1.mem")
	) j1_rom2 (clk, next_rom_addr, j1N_data);

	rom_sync #(
		.INIT_F("l1.mem")
	) l1_rom2 (clk, next_rom_addr, l1N_data);

	//block ROMs
	rom_sync #(
		.INIT_F("white.mem")
	) white_rom (clk, block_rom_addr, white_data);

	rom_sync #(
		.INIT_F("red.mem")
	) red_rom (clk, block_rom_addr, red_data);

	rom_sync #(
		.INIT_F("blue.mem")
	) blue_rom (clk, block_rom_addr, blue_data);


   //make color assignment decoder
   //color_decode cde(hcount[10:1],vcount, decode_infalling_h,falling_v, spr_rom_addr);
   logic [9:0] decode_h, decode_v;
   logic [7:0] spriteChoice;
   always_comb begin
	decode_h = 10'b0;
	decode_v = 10'b0;
	if (hcount[10:1] >= 10'd440) begin
		decode_h = 10'd480;
		decode_v = 10'd200;
		spriteChoice = next_sprite;
	end
	else begin 
		decode_h = falling_h;
		decode_v = falling_v;
		spriteChoice = falling_sprite;
	end
   end 
   color_decode cde(hcount[10:1], vcount, decode_h, decode_v, spr_rom_addr);
   color_assign cas(spr_rom_data,assigned_color);
   block_decode bdc(hcount[10:1], vcount, block_rom_addr);
   color_assign bas(block_rom_data,block_color);
   color_decode nde(hcount[10:1], vcount,10'd480, 10'd200, next_rom_addr); 
   color_assign can(next_rom_data, next_assigned_color);
	


   //stich background work
   initial begin  // random start values
	v_start = 40'b01100_00101_00110_10011_10101_10101_01111_01101;
	h_start = 30'b10111_01001_00001_10100_00111_01010;
   end
   // paint stitch pattern with 16x16 pixel grid
   always_comb begin
	v_line = (hcount[4:1] == 4'b0000);
	h_line = (vcount[3:0] == 4'b0000);
	v_on = vcount[4] ^ v_start[hcount[10:5]];
	h_on = hcount[5] ^ h_start[vcount[8:4]];
	stitch = (v_line && v_on) || (h_line && h_on);
   end

	vga_counters counters(.clk50(clk), .*);
	
   always_ff @(posedge clk)
     if (reset) begin
	     background_colors <= 32'h001a1c80;
	     falling_reg <= 32'h0;
	     //backZero <= 32'h003a5e3f;
	     background_r <= 8'h1a;
	     background_g <= 8'h1c;
	     background_b <= 8'h80;
	     //write black to the entire gamespace
	     back0 <= 32'h0;
	     back1 <= 32'h0;
	     back2 <= 32'h0;
	     back3 <= 32'h0;
	     back4 <= 32'h0;
	     back5 <= 32'h0;
	     back6 <= 32'h0;
	     back7 <= 32'h0;
	     back8 <= 32'h0;
	     back9 <= 32'h0;
	     back10 <= 32'h0;
	     back11 <= 32'h0;
	     back12 <= 32'h0;
	     back13 <= 32'h0;
	     back14 <= 32'h0;
	     back15 <= 32'h0;
	     back16 <= 32'h0;
	     back17 <= 32'h0;
	     back18 <= 32'h0;
	     back19 <= 32'h0;
	     falling_v_reg <= 32'h0;
	     falling_h_reg <= 32'h0;
	     tetro_next <= 32'h0;
	     //put certain of tetromino in random place
	     //falling_v <= 10'b0000010000;
	     //falling_h <= 10'b1111111111;
	     falling_ori <= 2'b00;
	     //falling_sprite <= 8'd0;
	     backnext <= 3'b000;
	     //set initial scores to 0
	     curr_score <= 32'h0;
	     high_score <= 32'h0;
	     //start with no music on
	     music_sound <= 4'h0;
	     music_en <= 1'b0;
	    
     end else if (chipselect && write) begin 
       case (address)
	       6'd0 : background_colors <= writedata;
	       6'd1 : falling_reg <= writedata;
	       6'd2 : back0 <= writedata;
	       6'd3 : back1 <= writedata;
	       6'd4 : back2 <= writedata;
	       6'd5 : back3 <= writedata;
	       6'd6 : back4 <= writedata;
	       6'd7 : back5 <= writedata;
	       6'd8 : back6 <= writedata;
	       6'd9 : back7 <= writedata;
	       6'd10 : back8 <= writedata;
	       6'd11 : back9 <= writedata;
	       6'd12 : back10 <= writedata;
	       6'd13 : back11 <= writedata;
	       6'd14 : back12 <= writedata;
	       6'd15 : back13 <= writedata;
	       6'd16 : back14 <= writedata;
	       6'd17 : back15 <= writedata;
	       6'd18 : back16 <= writedata;
	       6'd19 : back17 <= writedata;
	       6'd20 : back18 <= writedata;
	       6'd22 : falling_v_reg <= writedata;
	       6'd23 : falling_h_reg <= writedata;
	       6'd24 : back19 <= writedata;
       endcase
     end
	//UPLOAD ALL CODE TO A GITHUBB BRANCH

	//hcount is 11 bits, but you only use the top 10
	logic [3:0] sel;
	logic [31:0] back0cpy;

	always_ff   @(posedge clk) begin
	   if (220 < hcount[10:1] && hcount[10:1] < 240)
			block <= row[19:18];
	   else if (239 < hcount[10:1] && hcount[10:1] < 260) 
			block <= row[17:16];
	   else if (259 < hcount[10:1] && hcount[10:1] < 280)
			block <= row[15:14];
	   else if (279 < hcount[10:1] && hcount[10:1] < 300) 
			block <= row[13:12];
	   else if (299 < hcount[10:1] && hcount[10:1] < 320) 
			block <= row[11:10];
	   else if (319 < hcount[10:1] && hcount[10:1] < 340) 
			block <= row[9:8];
	   else if (339 < hcount[10:1] && hcount[10:1] < 360) 
			block <= row[7:6];
	   else if (359 < hcount[10:1] && hcount[10:1] < 380) 
			block <= row[5:4];
	   else if (379 < hcount[10:1] && hcount[10:1] < 400) 
			block <= row[3:2];
	   else if (399 < hcount[10:1] && hcount[10:1] < 420) 
			block <= row[1:0];
	   else
		   block <= 2'b11;

	  /* case (next_sprite)
		   8'd0: next_rom_data = i1_data;
		   8'd1: next_rom_data = oN_data;
		   8'd2: next_rom_data = t1N_data;
		   8'd3: next_rom_data = s1N_data;
		   8'd4: next_rom_data = z1N_data;
		   8'd5: next_rom_data = j1N_data;
		   8'd6: next_rom_data = l1N_data;
		   default: next_rom_data = i1_data;
	   endcase
*/

	end

   always_comb begin
      {VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0};
	   falling_v = falling_v_reg[9:0]; 
	   falling_h = falling_h_reg[9:0]; 
	   //next_sprite = 8'd0;
	  // back0cpy = backZero[31:0];
	   sel = 4'd0; 
	  // block = 2'd0;
	   row = 20'd0;
	   //assign falling sprite from its 32 bit register
	   falling_sprite = falling_reg[7:0];
	   next_sprite = tetro_next[7:0];
	   //case (falling_sprite)
	   case (spriteChoice)
		   8'd0: begin
			   spr_rom_data = i1_data;
		   end
		   8'd1: begin
			   spr_rom_data = i2_data;
		   end
		   8'd2: begin
			   spr_rom_data = i3_data;
		   end
		   8'd3: spr_rom_data = i4_data;
		   8'd4: spr_rom_data = o_data;
		   8'd5: spr_rom_data = o_data;
		   8'd6: spr_rom_data = o_data;
		   8'd7: spr_rom_data = o_data;
		   8'd8: spr_rom_data = t1_data;
		   8'd9: spr_rom_data = t2_data;
		   8'd10: spr_rom_data = t3_data;
		   8'd11: spr_rom_data = t4_data;
		   8'd12: spr_rom_data = s1_data;
		   8'd13: spr_rom_data = s2_data;
		   8'd14: spr_rom_data = s3_data;
		   8'd15: spr_rom_data = s4_data;
		   8'd16: spr_rom_data = z1_data;
		   8'd17: spr_rom_data = z2_data;
		   8'd18: spr_rom_data = z3_data;
		   8'd19: spr_rom_data = z4_data;
		   8'd20: spr_rom_data = j1_data;
		   8'd21: spr_rom_data = j2_data;
		   8'd22: spr_rom_data = j3_data;
		   8'd23: spr_rom_data = j4_data;
		   8'd24: spr_rom_data = l1_data;
		   8'd25: spr_rom_data = l2_data;
		   8'd26: spr_rom_data = l3_data;
		   8'd27: spr_rom_data = l4_data;
		   default: spr_rom_data = l4_data;
	   endcase
	   /*case (next_sprite)
		   8'd0: next_rom_data = i1_data;
		   8'd1: next_rom_data = oN_data;
		   8'd2: next_rom_data = t1N_data;
		   8'd3: next_rom_data = s1N_data;
		   8'd4: next_rom_data = z1N_data;
		   8'd5: next_rom_data = j1N_data;
		   8'd6: next_rom_data = l1N_data;
		   default: next_rom_data = i1_data;
	   endcase
*/

	   case(block)
		   2'b00: block_rom_data = 8'd8;
		   2'b01: block_rom_data = blue_data;
		   2'b10: block_rom_data = red_data;
		   2'b11: block_rom_data = white_data;
	   endcase

	   if (VGA_BLANK_n) begin
		  if ((hcount[10:1] > 440 && hcount[10:1] < 600) && (vcount > 180 && vcount < 280)) begin
			//draw next tetro box
			if ((hcount[10:1] <= 480 + 79) && (hcount[10:1] >= 480) && (vcount >= 200) && (vcount < 200 + 79)) begin
			 	//assign vga values based upon assigned color
					{VGA_R, VGA_G, VGA_B} = assigned_color; 
			end
			else
				{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
		   end
		   //if in game space
		   else if (hcount[10:1] > 220 && hcount[10:1] < 420 && (vcount > 70 && vcount < 470)) begin
			   //first draw  falling tetromino
			   if ((hcount[10:1] <= falling_h + 79) && (hcount[10:1] >= falling_h) && (vcount >= falling_v) && (vcount < falling_v + 79)) begin
			 	//assign vga values based upon assigned color
				   if (assigned_color != 24'h01_01_01)
					{VGA_R, VGA_G, VGA_B} = assigned_color; 
			   end
			   //now draw the game space background
			   if (vcount >= 450 && vcount < 470) begin
				    row  = back0[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 430 && vcount < 450) begin 
				    row  = back1[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 410 && vcount < 430) begin
				   //{VGA_R, VGA_G, VGA_B} = {{block,block,block,block},8'hff,8'hff};
				   row = back2[31:12];
				   if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   		
			   else if (vcount >= 390 && vcount < 410) begin 
				    row = back3[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 370  && vcount < 390) begin 
				    row = back4[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 350 && vcount < 370) begin 
				    row = back5[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 330 && vcount < 350) begin
				    row = back6[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 310 && vcount < 330) begin
				    row = back7[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 290 && vcount < 310) begin
				    row = back8[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 270 && vcount < 290) begin
				    row = back9[31:12];
			   	    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 250 && vcount < 270) begin
				    row = back10[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 230 && vcount < 250) begin
				    row = back11[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 210  && vcount < 230) begin
				    row = back12[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 190 && vcount < 210) begin
				    row = back13[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 170 && vcount < 190) begin
				    row = back14[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 150 && vcount < 170) begin
				    row = back15[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 130 && vcount < 150) begin
				    row = back16[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 110  && vcount < 130) begin
				    row = back17[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 90 && vcount < 110) begin
				    row = back18[31:12];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			   else if (vcount >= 70 && vcount < 90) begin
				   //just for this row, we use the bottom 20 bits to avoid the issue
				   //with falling v overlapping
				   row = back19[31:12];
				   //row = back19[19:0];
				    if (block != 2'b00) 
					  {VGA_R, VGA_G, VGA_B} = block_color; 
			   end
			  /* else if ((hcount[10:1] <= falling_h + 79) && (hcount[10:1] >= falling_h) && (vcount >= falling_v) && (vcount < falling_v + 79)) begin
			 	//assign vga values based upon assigned color
				   if(assigned_color != 24'h01_01_01)
					{VGA_R, VGA_G, VGA_B} = assigned_color; 
			   end
				  */
			   else 
			   //lastly draw everything else as black
			  	{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
		   end
		   //if not in gamespace
		   else if ((hcount[10:1] > 440 && hcount[10:1] < 600) && (vcount > 20 && vcount < 150)) begin
			//draw score box
			{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
		   end
		  /* else if ((hcount[10:1] > 440 && hcount[10:1] < 600) && (vcount > 180 && vcount < 280)) begin
			//draw next tetro box
			if ((hcount[10:1] <= 480 + 79) && (hcount[10:1] >= 480) && (vcount >= 200) && (vcount < 200 + 79)) begin
			 	//assign vga values based upon assigned color
				   if (next_assigned_color != 24'h01_01_01)
					{VGA_R, VGA_G, VGA_B} = next_assigned_color; 
				   else 
					{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
			end
			else
				{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
		   end
			*/
		   else if ((hcount[10:1] > 220 && hcount[10:1] < 420) && (vcount > 20 && vcount < 60)) begin
			//draw line count box
			{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00};
		   end
		   //indicator lower
		   else if ((hcount[10:1] > 20 && hcount[10:1] < 70) && (vcount > 400 && vcount < 440)) begin
			   {VGA_R, VGA_G, VGA_B} = {tetro_next[31:24], tetro_next[23:16], tetro_next[15:8]};
		   end
		   else if ((hcount[10:1] > 20 && hcount[10:1] < 70) && (vcount > 300 && vcount < 350)) begin
			   {VGA_R, VGA_G, VGA_B} = {next_rom_data, 8'h0, 8'h0};
			    /* case(sel)
				     4'd0: {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
				     4'd1: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};
				     4'd2: {VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};
				     default: {VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00};
			     endcase 
				    */
		   end
		   else begin
			  // {VGA_R, VGA_G, VGA_B} = {background_r, background_g, background_b};
			   VGA_R = (stitch) ? 8'h7d : 8'h62;
			   VGA_G = (stitch) ? 8'hcd : 8'h6e;
			   VGA_B = (stitch) ? 8'hd1 : 8'h65;
//#626e65
//#7dcdd1
		   end
	   end
   end
	       
endmodule

module vga_counters(
 input logic 	     clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);
/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;
       
   logic endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule
