/* * Device driver for the VGA video generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_ball.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree vga_ball.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "vga_ball.h"

#define DRIVER_NAME "vga_ball"

/* Device registers */
#define PIXEL(x) (x)
#define ADDRESS(x) ((x)+1)


/* CHANGED */
#define POS_X(x) ((x)+2)
#define POS_Y(x) ((x)+3)


/*
 * Information about our device
 */
struct cam_data_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
    cam_data_pixel_t pixel;
	cam_address_t address;	
    cam_data_position_t position;
} dev;

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
static void write_pixel(cam_data_pixel_t *pixel)
{
	//printk(KERN_ERR "\n Pixel value: %d \n",(int) pixel->pixel);
	iowrite32(pixel->pixel, PIXEL(dev.virtbase) );
	dev.pixel = *pixel;
	
}

static void write_address(cam_address_t *address)
{
	//printk(KERN_ERR "\n Address value: %d \n",(int) address->address);
	iowrite32(address->address, ADDRESS(dev.virtbase));
	dev.address = *address;	
}

static void write_position(cam_data_position_t *position)
{
	iowrite32(position->pos_x*8, POS_X(dev.virtbase) );
	iowrite32(position->pos_y*8, POS_Y(dev.virtbase) );

	dev.position = *position;
}


/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long cam_data_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	cam_data_arg_t vla;

	switch (cmd) {
	case CAM_DATA_WRITE_PIXEL:
		if (copy_from_user(&vla, (cam_data_arg_t *) arg,
				   sizeof(cam_data_arg_t)))
			return -EACCES;
		write_pixel(&vla.pixel);
		break;
	
	//write pixel position
	case CAM_DATA_WRITE_POSITION:
		if (copy_from_user(&vla, (cam_data_arg_t *) arg,
				   sizeof(cam_data_arg_t)))
			return -EACCES;
		write_position(&vla.position);
		break;

	case CAM_DATA_WRITE_ADDRESS:
		if (copy_from_user(&vla, (cam_data_arg_t *) arg,
					sizeof(cam_data_arg_t)))
			return -EACCES;
		write_address(&vla.address);
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations cam_data_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = cam_data_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice cam_data_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &cam_data_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init cam_data_probe(struct platform_device *pdev)
{
        //cam_data_pixel_t top_corner = 0xFFF; //Just a placeholder
	//cam_address_t intial_address = 0;

//				vga_ball_position initial = { 100, 100 };
	int ret;

	/* Register ourselves as a misc device: creates /dev/vga_ball */
	ret = misc_register(&cam_data_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	/* Set an pixel position */
        //write_pixel(&top_corner);
		//write_address(&initial_address);
//				write_position(&initial);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&cam_data_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int cam_data_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&cam_data_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id cam_data_of_match[] = {
	{ .compatible = "csee4840,vga_ball-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, cam_data_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver cam_data_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(cam_data_of_match),
	},
	.remove	= __exit_p(cam_data_remove),
};

/* Called when the module is loaded: set things up */
static int __init cam_data_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&cam_data_driver, cam_data_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit cam_data_exit(void)
{
	platform_driver_unregister(&cam_data_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(cam_data_init);
module_exit(cam_data_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("Camera driver");
