/* * Device driver for the VGA video generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_ball.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree vga_ball.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include <linux/ioctl.h>
#include "gamelogic.h"

#define DRIVER_NAME "vga_ball"

/* Device registers */
#define LOC(addr, offset) ((addr) + offset)


/*
 * Information about our device
 */
struct vga_ball_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
	location_t locations[16];
} dev;

static void writelocation(location_t loc1, location_t loc2,
				location_t loc3, location_t loc4,
				unsigned char offset)
{
	unsigned long data;
	data = ((loc1.x << 28) & 0xF0000000) | ((loc1.y << 24) & 0x0F000000)
		|((loc2.x << 20) & 0x00F00000) | ((loc2.y << 16) & 0x000F0000)
		| ((loc3.x << 12) & 0x0000F000) | ((loc3.y  << 8) & 0x00000F00)
		| ((loc4.x << 4) & 0x000000F0) | ((loc4.y & 0x0000000F));
	iowrite32(data, LOC(dev.virtbase, offset));
}

static void writedata(location_t cursor, block_t *blocks)
{
	int i;
	
	for (i = 0; i < 3; i++)
		writelocation(blocks[4*i].location, blocks[4*i+1].location,
				blocks[4*i+2].location, blocks[4*i+3].location,
				4*i);
	writelocation(blocks[12].location, blocks[13].location,
			blocks[14].location, cursor, 12);
	for (i = 0; i < 15; i++) {
		dev.locations[i].x = blocks[i].location.x;
		dev.locations[i].y = blocks[i].location.y;
	}
	dev.locations[15].x = cursor.x;
	dev.locations[15].y = cursor.y;
	
}

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
static void write_background(vga_ball_arg_t *vla)
{
	unsigned long data;
	location_t selected;
	unsigned char win;
	writedata(vla->cursor_loc, vla->blocks);
	selected.x = vla->selected.x;
	selected.y = vla->selected.y;
	win = vla->win;
	data = 0x00000000 | ((selected.x << 12) & 0x0000F000) | ((selected.y << 8) & 0x00000F00)
		| win;
	iowrite32(data, dev.virtbase+16);
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long vga_ball_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	vga_ball_arg_t vla;

	switch (cmd) {
	case VGA_BALL_WRITE_BACKGROUND:
		if (copy_from_user(&vla, (vga_ball_arg_t *) arg,
				   sizeof(vga_ball_arg_t)))
			return -EACCES;
		write_background(&vla);
		break;

	case VGA_BALL_READ_BACKGROUND:
		if (copy_to_user((vga_ball_arg_t *) arg, &vla,
				 sizeof(vga_ball_arg_t)))
			return -EACCES;
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations vga_ball_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = vga_ball_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice vga_ball_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &vga_ball_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init vga_ball_probe(struct platform_device *pdev)
{
	vga_ball_arg_t vla;
	int ret;
	int i, num;
	
	num = 15;
	for (i = 0; i < num; i++) {
		vla.blocks[i].block_id = i+1;
		vla.blocks[i].type = 1;
		vla.blocks[i].location.x = i % 4;
		vla.blocks[i].location.y = i / 4; 
	}

	vla.cursor_loc.x = 0;
	vla.cursor_loc.y = 0;
	vla.selected.x = 4;
	vla.selected.y = 4;
	vla.win = 0;
	/* Register ourselves as a misc device: creates /dev/vga_ball */
	ret = misc_register(&vga_ball_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	/* Set an initial color */
        write_background(&vla);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&vga_ball_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int vga_ball_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&vga_ball_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id vga_ball_of_match[] = {
	{ .compatible = "csee4840,vga_ball-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, vga_ball_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver vga_ball_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(vga_ball_of_match),
	},
	.remove	= __exit_p(vga_ball_remove),
};

/* Called when the module is loaded: set things up */
static int __init vga_ball_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&vga_ball_driver, vga_ball_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit vga_ball_exit(void)
{
	platform_driver_unregister(&vga_ball_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(vga_ball_init);
module_exit(vga_ball_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("VGA ball driver");
