/*
 * Userspace program that communicates with the vga_ball device driver
 * through ioctls
 *
 * Stephen A. Edwards
 * Columbia University
 */

#include <stdio.h>
#include "vga_ball.h"
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

#include <pthread.h>
#include <math.h>

//--------- copy from lab2 ----------------------

#include <libusb-1.0/libusb.h>
#include <stdlib.h> 

#define USB_HID_KEYBOARD_PROTOCOL 1

/* Modifier bits */
#define USB_LCTRL  (1 << 0)
#define USB_LSHIFT (1 << 1)
#define USB_LALT   (1 << 2)
#define USB_LGUI   (1 << 3)
#define USB_RCTRL  (1 << 4)
#define USB_RSHIFT (1 << 5)
#define USB_RALT   (1 << 6) 
#define USB_RGUI   (1 << 7)

struct usb_keyboard_packet {
  uint8_t modifiers;
  uint8_t reserved;
  uint8_t keycode[6];
};

/* Find and open a USB keyboard device.  Argument should point to
   space to store an endpoint address.  Returns NULL if no keyboard
   device was found. */
extern struct libusb_device_handle *openkeyboard(uint8_t *);

struct libusb_device_handle *openkeyboard(uint8_t *endpoint_address) {
  libusb_device **devs;
  struct libusb_device_handle *keyboard = NULL;
  struct libusb_device_descriptor desc;
  ssize_t num_devs, d;
  uint8_t i, k;
  
  /* Start the library */
  if ( libusb_init(NULL) < 0 ) {
    fprintf(stderr, "Error: libusb_init failed\n");
    exit(1);
  }

  /* Enumerate all the attached USB devices */
  if ( (num_devs = libusb_get_device_list(NULL, &devs)) < 0 ) {
    fprintf(stderr, "Error: libusb_get_device_list failed\n");
    exit(1);
  }

  /* Look at each device, remembering the first HID device that speaks
     the keyboard protocol */

  for (d = 0 ; d < num_devs ; d++) {
    libusb_device *dev = devs[d];
    if ( libusb_get_device_descriptor(dev, &desc) < 0 ) {
      fprintf(stderr, "Error: libusb_get_device_descriptor failed\n");
      exit(1);
    }

    if (desc.bDeviceClass == LIBUSB_CLASS_PER_INTERFACE) {
      struct libusb_config_descriptor *config;
      libusb_get_config_descriptor(dev, 0, &config);
      for (i = 0 ; i < config->bNumInterfaces ; i++)	       
	for ( k = 0 ; k < config->interface[i].num_altsetting ; k++ ) {
	  const struct libusb_interface_descriptor *inter =
	    config->interface[i].altsetting + k ;
	  if ( inter->bInterfaceClass == LIBUSB_CLASS_HID &&
	       inter->bInterfaceProtocol == USB_HID_KEYBOARD_PROTOCOL) {
	    int r;
	    if ((r = libusb_open(dev, &keyboard)) != 0) {
	      fprintf(stderr, "Error: libusb_open failed: %d\n", r);
	      exit(1);
	    }
	    if (libusb_kernel_driver_active(keyboard,i))
	      libusb_detach_kernel_driver(keyboard, i);
	    libusb_set_auto_detach_kernel_driver(keyboard, i);
	    if ((r = libusb_claim_interface(keyboard, i)) != 0) {
	      fprintf(stderr, "Error: libusb_claim_interface failed: %d\n", r);
	      exit(1);
	    }
	    *endpoint_address = inter->endpoint[0].bEndpointAddress;
	    goto found;
	  }
	}
    }
  }

 found:
  libusb_free_device_list(devs, 1);

  return keyboard;
}
//---------------------


int vga_ball_fd;

/* Set the background color */
void set_background_color(const vga_ball_color_t *c)
{
  vga_ball_arg_t vla;
  vla.background = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_BACKGROUND, &vla)) {
      perror("ioctl(VGA_BALL_SET_BACKGROUND) failed");
      return;
  }
}



int input[5] = {0}; // bool: up, down, left, right, attack
int be_hit = 0; // player be hit, no further damage if > 0, flash player and health

struct item {
  double position[2];
  double vel[2];
  int id;
};


void set_transmit(struct item boss, struct item player, struct item *bullet, struct item *ph, 
	struct item *pbullet, int bh, int result){

	vga_ball_color_t color;

	int i, x, y, id;
	for (i = 0; i < DATA_SIZE; i++){
		color.data[i] = 0;
	}

	color.data[512] = bh*(160/20); // boss health

	if (result == 0){
		x = (int) boss.position[0];
		y = (int) boss.position[1];
		id = boss.id;

		color.data[508] = x+100;
		color.data[509] = y+100;
		color.data[510] = (((x+100)>>4)&0xf0) | ((y+100)>>8);
		color.data[511] = id;

		x = (int) player.position[0];
		y = (int) player.position[1];
		if ((be_hit % 6) < 3)	id = player.id;
		else id = 0;
		

		color.data[504] = x+100;
		color.data[505] = y+100;
		color.data[506] = (((x+100)>>4)&0xf0) | ((y+100)>>8);
		color.data[507] = id;

		for (i = 0; i<5; i++){
			x = (int) ph[i].position[0];
			y = (int) ph[i].position[1];
			if ((be_hit % 6) < 3)	id = ph[i].id;
			else id = 0;

			color.data[4 * (i+1) + 480] = x+100;
			color.data[4 * (i+1) + 481] = y+100;
			color.data[4 * (i+1) + 482] = (((x+100)>>4)&0xf0) | ((y+100)>>8);
			color.data[4 * (i+1) + 483] = id;
		}

		for (i = 0; i < 100; i++){
			x = (int) bullet[i].position[0];
			y = (int) bullet[i].position[1];
			id = bullet[i].id;
			color.data[4 * (i+1) + 0] = x+100;
			color.data[4 * (i+1) + 1] = y+100;
			color.data[4 * (i+1) + 2] = (((x+100)>>4)&0xf0) | ((y+100)>>8);
			color.data[4 * (i+1) + 3] = id;
		}

		for (i = 0; i < 20; i++){
			x = (int) pbullet[i].position[0];
			y = (int) pbullet[i].position[1];
			id = pbullet[i].id;
			color.data[4 * (i+1) + 400] = x+100;
			color.data[4 * (i+1) + 401] = y+100;
			color.data[4 * (i+1) + 402] = (((x+100)>>4)&0xf0) | ((y+100)>>8);
			color.data[4 * (i+1) + 403] = id;
		}
	}
	else if(result == 1){
		int i, j;		
		for (i = 0; i < 3; i++){
			for (j = 0; j < 4; j++){
				x = 180 + 40*i;
				y = 240 + 30*j;
				id = 20 + 5*i + j;
				color.data[4 * (5*i + j +1) + 0] = x+100;
				color.data[4 * (5*i + j +1) + 1] = y+100;
				color.data[4 * (5*i + j +1) + 2] = (((x+100)>>4)&0xf0) | ((y+100)>>8);
				color.data[4 * (5*i + j +1) + 3] = id;
			}		
		}
	}
	else if(result == -1){
		int i, j;		
		for (i = 0; i < 3; i++){
			for (j = 0; j < 4; j++){
				x = 180 + 40*i;
				y = 240 + 30*j;
				id = 35 + 5*i + j;
				color.data[4 * (5*i + j +1) + 0] = x+100;
				color.data[4 * (5*i + j +1) + 1] = y+100;
				color.data[4 * (5*i + j +1) + 2] = (((x+100)>>4)&0xf0) | ((y+100)>>8);
				color.data[4 * (5*i + j +1) + 3] = id;
			}		
		}
	}

	set_background_color(&color);  
}


struct libusb_device_handle *keyboard;
uint8_t endpoint_address;
struct usb_keyboard_packet packet;
int transferred;

pthread_t key_thread;
void *key_thread_f(void *);

void key_input(){
    libusb_interrupt_transfer(keyboard, endpoint_address,
			      (unsigned char *) &packet, sizeof(packet),
			      &transferred, 0);
    if (transferred == sizeof(packet)) {
      int i = 0;

      for (i = 0; i < 5; i++){
	input[i] = 0;
      }
      for (i = 0; i < 6; i++){
	if (packet.keycode[i] == 0x1a) input[0] = 1;
	else if (packet.keycode[i] == 0x16) input[1] = 1;
	else if (packet.keycode[i] == 0x04) input[2] = 1;
	else if (packet.keycode[i] == 0x07) input[3] = 1;
	else if (packet.keycode[i] == 0x2c) input[4] = 1;
      }
      
    }
}

void *key_thread_f(void *ignored)
{
  while (1){
    key_input();
  }
  return NULL;
}

void player_move(struct item *player){
    if ((input[0] == 1) && (player->position[0] > -10)) player->position[0] -= 5;
    if ((input[1] == 1) && (player->position[0] < 460)) player->position[0] += 5;
    if ((input[2] == 1) && (player->position[1] > -10)) player->position[1] -= 5;
    if ((input[3] == 1) && (player->position[1] < 620)) player->position[1] += 5;
}

void bullet_move(struct item *bullet, struct item *pbullet){
	int i;
	for (i = 0; i<100; i++){
		bullet[i].position[0] += bullet[i].vel[0];
		bullet[i].position[1] += bullet[i].vel[1];
	}
	for (i = 0; i<20; i++){
		pbullet[i].position[0] += pbullet[i].vel[0];
		pbullet[i].position[1] += pbullet[i].vel[1];
	}
}

void bullet_clean(struct item *bullet, struct item *pbullet){
	int i;
	for (i = 0; i<100; i++){
		if ((bullet[i].position[0] > 500) || (bullet[i].position[0] < -40)
			|| (bullet[i].position[1] > 660) || (bullet[i].position[1] < -40))
			bullet[i].id = 0;
	}
	for (i = 0; i<20; i++){
		if ((pbullet[i].position[0] > 500) || (pbullet[i].position[0] < -40)
			|| (pbullet[i].position[1] > 660) || (pbullet[i].position[1] < -40))
			pbullet[i].id = 0;
	}
}

void gen_bullet(struct item *bullet, int n, double px, double py, double vx, double vy, int id){
	bullet[n].position[0] = px;
	bullet[n].position[1] = py;
	bullet[n].vel[0] = vx;
	bullet[n].vel[1] = vy;
	bullet[n].id = id;
}


void boss_atk_1(struct item *bullet, int offset, struct item *boss){
	gen_bullet(bullet, offset + 0, boss->position[0] + 10, boss->position[1], 1.5, -1, 3);
	gen_bullet(bullet, offset + 1, boss->position[0] + 10, boss->position[1], 1.5, 0, 2);
	gen_bullet(bullet, offset + 2, boss->position[0] + 10, boss->position[1], 1.5, 1, 4);
}

void boss_atk_2(struct item *bullet, int offset, struct item *boss){
	gen_bullet(bullet, offset + 0, boss->position[0] + 10, boss->position[1], 2.5, -1.5, 17);
	gen_bullet(bullet, offset + 1, boss->position[0] + 10, boss->position[1], 2.5, 0, 16);
	gen_bullet(bullet, offset + 2, boss->position[0] + 10, boss->position[1], 2.5, 1.5, 15);
	gen_bullet(bullet, offset + 3, boss->position[0] + 10, boss->position[1], -2.5, -1.5, 15);
	gen_bullet(bullet, offset + 4, boss->position[0] + 10, boss->position[1], -2.5, 0, 16);
	gen_bullet(bullet, offset + 5, boss->position[0] + 10, boss->position[1], -2.5, 1.5, 17);
	gen_bullet(bullet, offset + 6, boss->position[0] + 10, boss->position[1], 0, 2.5, 18);
	gen_bullet(bullet, offset + 7, boss->position[0] + 10, boss->position[1], 0, -2.5, 18);
}

void boss_atk_3(struct item *bullet, int offset, struct item *boss, int time){
	if (time < 2*30){ }
	else if (time < 3*30){
		gen_bullet(bullet, offset + 0, boss->position[0] + 10, boss->position[1], 15, 0, 2);
		gen_bullet(bullet, offset + 1, boss->position[0] + 10, boss->position[1], 0, -15, 5);
	}
	else if (time < 4*30){
	}
	else if (time < 6*30){
		gen_bullet(bullet, offset + 0, boss->position[0] + 10, boss->position[1], 15, 0, 2);
	}
	else if (time < 7*30){ 
	}
	else if (time < 8*30){
		gen_bullet(bullet, offset + 0, boss->position[0] + 10, boss->position[1], 15, 0, 2);
		gen_bullet(bullet, offset + 1, boss->position[0] + 10, boss->position[1], 0, 15, 5);
	}
	else{ 
	}
}

void boss_atk_4(struct item *bullet, int offset, struct item *boss, int time){
	double ang;
	if (time < 20*30) ang = 0.8 * cos(0.04 * time) + 0.3;
	else if (time < 40*30) ang = 0.8 * cos(0.04 * time) + 0.3 * ((30*30 - time) / (10.0*30) ) ;
	else ang = 0.8 * cos(0.04 * time) - 0.3;

	double vx = 1.5*cos(ang);
	double vy = 1.5*sin(ang);
	
	int n;
	if (ang < (3.14*20/360)) n = 16;
	else if (ang < (3.14*80/360)) n = 15;
	else n = 18;
	gen_bullet(bullet, offset + 0, boss->position[0] + 10, boss->position[1], vx, vy, n);
}

void boss_move_1(struct item *boss){

	if ((boss->vel[0] == 0) && (boss->vel[1] == 0)){ // init
		boss->vel[0] = 0;
		boss->vel[1] = 0.9;
	}

	if ((boss->position[1] > 440) && (boss->vel[1] > 0)) boss->vel[1] = -boss->vel[1];
	if ((boss->position[1] < 200) && (boss->vel[1] < 0)) boss->vel[1] = -boss->vel[1];

}

void boss_move_2(struct item *boss){
	
	if ((boss->vel[0] == 0) && (boss->vel[1] == 0)){ // init
		boss->vel[0] = 0;
		boss->vel[1] = 2;
	}

	if ((boss->position[1] > 440) && (boss->position[0] <= 50) && (boss->vel[1] > 0)) {
		boss->vel[0] = 2;
		boss->vel[1] = 0;
	}
	if ((boss->position[1] > 440) && (boss->position[0] > 280) && (boss->vel[0] > 0)) {
		boss->vel[0] = 0;
		boss->vel[1] = -2;
	}
	if ((boss->position[1] < 200) && (boss->position[0] > 280) && (boss-> vel[1] < 0)) {
		boss->vel[0] = -2;
		boss->vel[1] = 0;
	}
	if ((boss->position[1] < 200) && (boss->position[0] <= 50) && (boss ->vel[0] < 0)) {
		boss->vel[0] = 0;
		boss->vel[1] = 2;
	} 
}

void boss_move_3(struct item *boss, int time){
	if (time < 2*30){ 
		boss->vel[0] = -0.5;
		boss->vel[1] = 0;
	}
	else if (time < 3*30){
		boss->vel[0] = 8;
		boss->vel[1] = 9;
	}
	else if (time < 4*30){
		boss->vel[0] = 0;
		boss->vel[1] = 0;
	}
	else if (time < 6*30){
		boss->vel[0] = 0;
		boss->vel[1] = -9;
	}
	else if (time < 7*30){
		boss->vel[0] = 0;
		boss->vel[1] = 0;
	}
	else if (time < 8*30){
		boss->vel[0] = -8;
		boss->vel[1] = 9;
	}
	else if (time < 10*30){
		boss->vel[0] = 2;
		boss->vel[1] = 0;
	}
	else{
		boss->vel[0] = 0;
		boss->vel[1] = 0;
	}
}

void boss_move_4(struct item *boss, int time){
	double dx, dy, dis;
	if (time < 20*30){ 
		dx = boss->position[0] - 30;
		dy = boss->position[1] - 80;
		dis = dx*dx + dy*dy;
		if (dis < 100) {
			boss->vel[0] = 0;
			boss->vel[1] = 0;
		}
		else{
			boss->vel[0] = -3*dx/sqrt(dis);
			boss->vel[1] = -3*dy/sqrt(dis);
		}
	}
	else if (time < 40*30){
		boss->vel[0] = 0;
		boss->vel[1] = 0.8;
	}
	else { 
		dx = boss->position[0] - 50;
		dy = boss->position[1] - 320;
		dis = dx*dx + dy*dy;
		if (dis < 100) {
			boss->vel[0] = 0;
			boss->vel[1] = 0;
		}
		else{
			boss->vel[0] = -3*dx/sqrt(dis);
			boss->vel[1] = -3*dy/sqrt(dis);
		}
	}
}

void boss_move(struct item *boss){
	boss->position[0] += boss->vel[0];
	boss->position[1] += boss->vel[1];
}


void boss_atk(int game_time, struct item *bullet, struct item *boss){
	static int offset = 0;
	int time = game_time % (95*30);

	if (time < 20*30){
		boss_move_1(boss);
		if (time % (2*30) == 0){
			boss_atk_1(bullet, offset, boss);
			offset += 3;
		}			
		if (offset > (100 - 3)){
			offset = 0;
		}
	}
	else if(time == 20*30) {
		boss->vel[0] = 0;
		boss->vel[1] = 0;
	}
	else if ((time > 20*30) && (time < 35*30)) {
		boss_move_2(boss);
		if (time % (1*30) == 0) {
			boss_atk_2(bullet, offset, boss);
			offset += 8;
		}
		if( offset > (100-8)) offset = 0;
	}
	else if ((time > 35*30) && (time < 45*30)) {
		boss_move_3(boss, time - 35*30);
		if (time % (3) == 0){
			boss_atk_3(bullet, offset, boss, time - 35*30);
			offset += 2;
		}
		if( offset > (100-2)) offset = 0;
	}
	else if ((time > 45*30) && (time < 95*30)) {
		boss_move_4(boss, time - 45*30);
		if (time % (6) == 0){
			boss_atk_4(bullet, offset, boss, time - 45*30);
			offset += 1;
		}
		if( offset > (100-1)) offset = 0;
	}
}

void player_atk(struct item *pbullet, struct item *player){
	static int offset = 0, cd = 0;
	if (cd > 0) cd --;
	if ((input[4] == 1) && (cd == 0)){
		cd = 15;
		pbullet[offset + 0].position[0] = player->position[0];
		pbullet[offset + 0].position[1] = player->position[1];
		pbullet[offset + 0].id = 6;
		
		offset += 1;
		if (offset > (20 - 1)){
			offset = 0;
		}
	}
}


void player_set_health(struct item *ph, int number)
{
	int i=0;
	for(i = 0;i<5;i++)
	{
		if(number >= 2* i + 2)
			ph[i].id = 7;
		else if(number== 2*i+1)
			ph[i].id = 8;
		else
			ph[i].id = 9;
	}
}

int player_hit(int ph, struct item *bullet, struct item *player){
	int i, dx, dy, dis;
	if (be_hit > 0) be_hit --;
	for (i = 0; i<100; i++){
		dx = player->position[0] - bullet[i].position[0];
		dy = player->position[1] - bullet[i].position[1];
		dis = dx*dx + dy*dy;

		if ((bullet[i].id != 0) && (dis < 400)){
			printf("hit, %d\n", ph-1);
			bullet[i].id = 0;
			if (be_hit == 0){
				be_hit = 30;
				return ph - 1;
			}
		} 
	}
	return ph;
}

int boss_hit(int bh, struct item *pbullet, struct item *boss){
	int i, dx, dy, dis;
	for (i = 0; i<20; i++){
		dx = boss->position[0] - pbullet[i].position[0];
		dy = boss->position[1] - pbullet[i].position[1];
		dis = dx*dx + dy*dy;

		if ((pbullet[i].id != 0) && (dis < 400)){
			printf("boss hit, %d\n", bh-1);
			pbullet[i].id = 0;
			return bh - 1;
		} 
	}
	return bh;
}

void reset(struct item *boss, struct item *player, struct item *bullet, 
		struct item *phealth, struct item *pbullet){

	boss->vel[0] = 0;
	boss->vel[1] = 0;
	boss->position[0] = 50;
	boss->position[1] = 320;
	boss->id = 1;

	player->position[0] = 400;
	player->position[1] = 320;
	player->id = 19;

	int i;
	for (i = 0; i<100; i++){
		bullet[i].id = 0;
	}

	for (i = 0; i<5; i++){
		phealth[i].position[0] = 20;
		phealth[i].position[1] = 580 - 20*i;
		phealth[i].id = 7;
	}

	for (i = 0; i<20; i++){
		pbullet[i].vel[0] = -10;
		pbullet[i].vel[1] = 0;
		pbullet[i].id = 0;
	}
}


int main()
{


  /* Open the keyboard */
  if ( (keyboard = openkeyboard(&endpoint_address)) == NULL ) {
    fprintf(stderr, "Did not find a keyboard\n");
    exit(1);
  }

  pthread_create(&key_thread, NULL, key_thread_f, NULL);

  struct item boss, player, bullet[100], phealth[5], pbullet[20];


  vga_ball_arg_t vla;
  static const char filename[] = "/dev/vga_ball";

  printf("VGA ball Userspace program started\n");

  if ( (vga_ball_fd = open(filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename);
    return -1;
  }



  while (1){
	int game_time = 0;
	int player_health = 10;
	int boss_health = 20;
	be_hit = 0;
	reset(&boss, &player, bullet, phealth, pbullet);
	while ((player_health > 0) && (boss_health > 0)) {
		game_time ++;
		boss_atk(game_time, bullet, &boss);
		boss_move(&boss);

		bullet_move(bullet, pbullet);
		bullet_clean(bullet, pbullet);

		player_move(&player);
		player_atk(pbullet, &player);

		player_health = player_hit(player_health, bullet, &player);
		player_set_health(phealth,player_health);

		boss_health = boss_hit(boss_health, pbullet, &boss);

		set_transmit(boss, player, bullet, phealth, pbullet, boss_health, 0);

		usleep(30000);
	}

	if (player_health == 0) {
		printf("YOU LOSE !!!\n");
		set_transmit(boss, player, bullet, phealth, pbullet, boss_health, -1);
		usleep(1000000);
		while(input[4] == 0){}
	}
	else{
		printf("YOU WIN !!!\n");
		set_transmit(boss, player, bullet, phealth, pbullet, boss_health, 1);
		usleep(1000000);
		while(input[4] == 0){}
	}
  }

  /* Terminate the network thread */
  pthread_cancel(key_thread);

  /* Wait for the network thread to finish */
  pthread_join(key_thread, NULL);

  return 0;
}














