/*
 * Avalon memory-mapped peripheral that generates VGA
 *
 * Stephen A. Edwards
 * Columbia University
 */

module vga_ball(input logic        clk,
	        input logic 	   reset,
		input logic [7:0]  writedata,
		input logic 	   write,
		input 		   chipselect,
		input logic [9:0]  address,

		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output logic 	   VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n,
		output logic 	   VGA_SYNC_n);

	logic [10:0]	hcount;
	logic [9:0]     vcount;

	vga_counters counters(.clk50(clk), .*);

	logic [255:0][9:0] sprites_x, sprites_y;
	logic [255:0][5:0] sprites_n;

	logic [9:0]	boss_health;

	always_ff @(posedge clk)
		if (reset) begin
			sprites_x <= {256{10'h0}};
			sprites_y <= {256{10'h0}};
			sprites_n <= {256{6'h0}};
			boss_health <= 10'd500;
		end else if (chipselect && write)
			if (address[9] == 1'h0) begin
				case (address[1:0])
					 2'h0 : sprites_x[address[9:2]][7:0] <= writedata;
					 2'h1 : sprites_y[address[9:2]][7:0] <= writedata;
					 2'h2 : begin
						sprites_x[address[9:2]][9:8] <= writedata[5:4];
						sprites_y[address[9:2]][9:8] <= writedata[1:0];
						end
					 2'h3 : sprites_n[address[9:2]][5:0] <= writedata[5:0];
					default:;
				endcase
			end else if (address == 10'd512) begin
				boss_health[9:2] <= writedata;
			end


	logic [9:0]					line;
	logic [29:0][23:0] 	pattern;
	logic [5:0]					length;
	logic [5:0]					n;
	sprites sprites0(.n_sprite (n), .*);

	logic [9:0] x, y;

	logic [639:0][23:0]	buf_e;
	logic [639:0][23:0]	buf_o;

	// assign n = sprites_n[0];
	// assign x = sprites_x[0];
	// assign y = sprites_y[0];

	logic [3:0] stage;
	logic [8:0] count;
	logic [5:0] pixel;
	logic [9:0] yposition;
	logic done;

	assign yposition = y + {4'd0, pixel} - 10'd100;

	always_ff @(posedge clk) begin
		if(reset) begin
			buf_e <= {640{24'h0}};
			buf_o <= {640{24'h0}};
			stage <= 0;
			count <= 0;
			pixel <= 0;
			done <= 1;
		end else begin
			if (vcount[0]) begin 	// output buffer_odd, edit buffer_even
				if ((hcount[10:1] > 640) & (vcount < 10'd480))
					buf_o <= {640{24'h000000}}; 	// background color

			end else begin 		// output buffer_even, edit buffer_odd
				if ((hcount[10:1] > 640) & (vcount < 10'd480))
					buf_e <= {640{24'h000000}}; 	// background color
			end

			if (hcount == 11'd1) begin
				done <= 0;
				stage <= 0;
				count <= 0;
				pixel <= 0;
			end

			if(~done)begin
				case(stage)
					4'd0 : begin
						n <= sprites_n[count];
						x <= sprites_x[count];
						y <= sprites_y[count];
						pixel <= 0;
						if (vcount >= 10'd479)
							line <= 10'd100 - sprites_x[count];
						else
							line <= vcount + 10'd101 - sprites_x[count];

						stage <= stage + 4'd1;
					end

					4'd1 : begin
						if ((n == 0) | (line >= 40)) begin // not in this line
							pixel <= 6'd0;
							stage <= 4'd0;
							if (count < 9'd256)
								count <= count + 8'd1;
							else
								done <= 1;
						end else begin
							stage <= stage + 4'd1;
						end
					end

					4'd2 : begin
						if (vcount[0]) begin 	// output buffer_odd, edit buffer_even
							if (pattern[pixel] != 24'h0) begin
								if (yposition < 10'd640)
									buf_e[yposition] <= pattern[pixel];
							end

						end else begin 		// output buffer_even, edit buffer_odd
							if (pattern[pixel] != 24'h0) begin
								if (yposition < 10'd640)
									buf_o[yposition] <= pattern[pixel];
							end
						end

						if (pixel < 6'd29) begin
							pixel <= pixel + 1;
							stage <= stage;
						end else begin
							pixel <= 6'd0;
							stage <= 4'd0;
							if (count < 9'd256)
								count <= count + 8'd1;
							else
								done <= 1;
						end


					end
					default:;
				endcase
			end
		end
	end


	always_comb begin

		{VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0};

		if (VGA_BLANK_n )
			if (vcount < 10'd5) begin
				if (hcount[10:1] < boss_health)
					{VGA_R, VGA_G, VGA_B} = 24'hff0000; // health
				else
					{VGA_R, VGA_G, VGA_B} = 24'h555555; // no health
			end
			else if (vcount < 10'd480) begin
				if (vcount[0]) begin	// output buffer_odd, edit buffer_even
					{VGA_R, VGA_G, VGA_B} = buf_o[hcount[10:1]][23:0];

				end else begin		// output buffer_even, edit buffer_odd
					{VGA_R, VGA_G, VGA_B} = buf_e[hcount[10:1]][23:0];

				end
			end
	end

endmodule


module vga_counters(
 input logic 	     clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;
       
   logic endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
/* verilator lint_off WIDTH */
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2); 
/* verilator lint_on WIDTH */

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule



