/* module: RiverRaid Hardware
 * Team - River Raid
 * Rojan Banmali (rb3199)
 * Xinhao Su (xs2413)
 *
 *
 * Avalon memory mapped slave
 *
 *
 * CSEE4840
 * Stephen A. Edwards
 * Columbia University
 */

module vga_ball(
    input logic clk,
    input logic reset,
		input logic [31:0]  writedata,
		input logic write,
		input logic read,
    input       chipselect,
		input logic [7:0]  address,
    input logic [7:0] GPIO,
    input left_chan_ready,
		input right_chan_ready,

		output logic [15:0] sample_data_l,
		output logic sample_valid_l,
		output logic [15:0] sample_data_r,
		output logic sample_valid_r,
    output logic [31:0] readdata,
		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output logic    VGA_CLK, VGA_HS, VGA_VS,
		                VGA_BLANK_n,
		output logic 	   VGA_SYNC_n);

   logic [10:0]	   vga_hcount;
   logic [9:0]     vga_vcount;
   logic [8:0]	   hcount;
   logic [8:0]     vcount;
   logic [23:0]    vga_rgb;

   logic update_frame, update_vars;

   logic [8:0] tileRAM_addr;    //tile MAP rom address
   logic [8:0] tileRAM_raddr;   //tile MAP read address
   logic       tileRAM_wren;    //tile MAP write enable
   logic [7:0] tileRAM_data;    //tile MAP read RAM data
   logic [7:0] tileRAM_wdata;   //tile MAP write RAM data
   
   logic [12:0] tileROM_addr;   //3bit/pixel tile artwork tile ROM address  
   logic [4:0] tileROM_pix;     //3bit tile ROM pixel output
   logic [1:0] tile_pal_id;
 
   //software interface vars
   logic [7:0] vscroll;//, vscroll_;
   logic [2:0] audio_cmd;
   logic audio_on;

   logic [7:0] fg_data; // 3bit unused + 3bit pxl + 2bit palID
   logic [5:0] fg_pix;
   logic [1:0] fg_palID;

   logic [23:0] sp_rgb_out;
   logic [23:0] tile_rgb_out;
   
   //instantiate modules.

   tileROM tileROM_1 (.clock(clk), .address(tileROM_addr),.q(tileROM_pix));

   tileRAM tileRAM_1 (.clock(clk),
                    .address(tileRAM_addr),
                    .wren(tileRAM_wren),
                    .data(tileRAM_wdata),
                    .q(tileRAM_data)); 
   
   
   vga_counters counters(.clk50(clk), .hcount(vga_hcount), .vcount(vga_vcount),.*);
   
   audio audio0(.clk(clk),.command(audio_cmd),.*);

   sprite_gen sprite_gen0(.clk(clk),
                          .reset(reset),
                          .chipselect(chipselect),
                          .address(address),
                          .writedata(writedata),
                          .write(write),
                          .hcount(hcount),
                          .vga_vcount(vga_vcount),
                          .vcount(vcount),
                          .fg_data(fg_data));
  
  // sprite_pal_rom sprite_pal0(
  //   .palID(fg_palID),
  //   .address(fg_pix),
  //   .rgb_out(sp_rgb_out)
  // );
 
colorPals colorPal_0(
    .palID(tile_pal_id),
    .address({1'b0, tileROM_pix}),
    .rgb_out(tile_rgb_out)
  );

  colorPals colorPal_1(
    .palID(fg_palID),
    .address(fg_pix),
    .rgb_out(sp_rgb_out)
  );

// Driver Read Write on posedge
   always_ff @(posedge clk) begin
   
     audio_cmd<=0;

     if (reset) begin
       //reset
        readdata<=0;
        //vscroll_<= 8'h0;
        vscroll<= 8'h0;

     end else if (chipselect && write)begin //WRITE from driver
      
      case (address)
        8'h0 :begin //Handled Below during negedge
        end

        8'h1:begin //Handled below during negedge
        end

       8'h2 : begin //PlayAudio {[2:0]audioClip_sel}
          audio_on<=writedata[0];
          audio_cmd<=writedata[3:1];
        end
        
	      8'h3 : begin //WriteVScroll {[7:0]}
         vscroll<=writedata[7:0];
        end

       endcase

     end else if (chipselect && read) begin //READ from driver

       case (address)
        8'h4:begin
          readdata[5:0]<={update_frame, GPIO[4:0]}; //TODO: readdata[5:0] = {update_frame,FIRE,RIGHT, LEFT, DOWN, UP}
        end

       endcase
     end
   end


   //Write TileMap from software
   always_ff @(negedge clk) begin

     if (chipselect && write && address==0)begin //WRITE from driver
	      //WriteTileMap {[8:0]tileMap_waddr, [4:0]tileID, [1:0]palID} 
          tileRAM_wdata <= writedata[6:0]; //{tileID, palID}
          tileRAM_addr <= writedata[16:7]; //{tileMAP_addr}
          tileRAM_wren<=1;
       end
       else begin
       //reset
        tileRAM_wdata<= 8'h0; //TODO: set to high impedance?
        tileRAM_addr <= tileRAM_raddr;
        tileRAM_wren<=0;
     end

  end

// Update frame: TODO: Not used at the moment. Comeback later.
   always_ff@(posedge clk) begin
     if(reset || (read && address==4 && update_frame))begin
       update_frame<=0;
     end else if(!update_frame && vga_vcount==480)begin
       update_frame<=1;
     end
   end

    /*//Update scrolling during vertical blanking
    always_ff@(posedge clk) begin

      if (reset) begin
        vscroll<=0;
      end
      else if (update_vars) begin
        vscroll<=vscroll_;
      end
        
    end*/

  //Tile scrolling
  always_comb begin
     if(hcount<320 && vcount<240)
     tileRAM_raddr = (hcount>>4) + (((vcount+vscroll)%256)>>4)*20;
     else tileRAM_raddr=0; 
   end

   always_comb begin
    if(hcount<320 && vcount<240)
      tileROM_addr = (tileRAM_data[7:2]<<8) + (hcount%16) + (((vcount+vscroll)%16)<<4);
    else tileROM_addr=0;
   end

assign fg_pix = fg_data[7:2];
assign fg_palID = fg_data[1:0];

assign tile_pal_id = tileRAM_data[1:0];

//FINAL VIDEO OUTPUT!!
//TODO: Use palatte to assign RGB
//VGA stuff. Draw RGB: 0,0,0 during blanking.
   always_comb begin
       vga_rgb = {8'h0, 8'h0, 8'h0};
      if (VGA_BLANK_n ) begin
          if(fg_pix != 6'd0 && (vga_hcount>1))
              vga_rgb = sp_rgb_out;
          else
              vga_rgb = tile_rgb_out;
    end
   end

assign {VGA_R, VGA_G, VGA_B} = vga_rgb;

assign update_vars = (vga_hcount==0 && vga_vcount == 480); //update anim variables
assign hcount = vga_hcount[10:2]; //divide by 4
assign vcount = vga_vcount[9:1]; //divide by 2
      
endmodule

/*
* sprite_pal_rom
*
*
*
*
*/
// module sprite_pal_rom (
//   input logic [1:0] palID,
//   input logic [2:0] address,
//   output logic [23:0] rgb_out
// );

// always_comb begin
  
//   rgb_out = 24'd0;
// if(palID==0)begin //ship
//   case(address)
//     0: rgb_out = 24'h000000; //transparent
//     1: rgb_out = 24'h888888; //dark gray
//     2: rgb_out = 24'h5B5B5B; //gray light
//     3: rgb_out = 24'h353535; //gray lighter
//     4: rgb_out = 24'h983939; //reddist 1
//     5: rgb_out = 24'h9B190A; //reddist 2
//     6: rgb_out = 24'hE84AFA; //magentaish
//     7: rgb_out = 24'hFFFFFF; //white
  
//   endcase
// end

// else if (palID==1)begin //explosion

//  case(address)
//     0: rgb_out = 24'h000000; //transparent
//     1: rgb_out = 24'h592A0A; //
//     2: rgb_out = 24'h6A370F; //
//     3: rgb_out = 24'h91601B; //
//     4: rgb_out = 24'hA88524; //
//     5: rgb_out = 24'h5A4C21; //
//     6: rgb_out = 24'h837927; //
//     7: rgb_out = 24'hB0B549; //
//   endcase

// end

// else if (palID==2)begin //TODO: modify and use for tiles
//  case(address)
//     0: rgb_out = 24'h000000; //transparent
//     1: rgb_out = 24'h888888; //dark gray
//     2: rgb_out = 24'h5B5B5B; //gray light
//     3: rgb_out = 24'h353535; //gray lighter
//     4: rgb_out = 24'h983939; //reddist 1
//     5: rgb_out = 24'h9B190A; //reddist 2
//     6: rgb_out = 24'hE84AFA; //magentaish
//     7: rgb_out = 24'hFFFFFF; //white
  
//   endcase

// end

// else if(palID==3)begin //TODO: modify and use for tiles
//  case(address)
//     0: rgb_out = 24'h000000; //transparent
//     1: rgb_out = 24'h888888; //dark gray
//     2: rgb_out = 24'h5B5B5B; //gray light
//     3: rgb_out = 24'h353535; //gray lighter
//     4: rgb_out = 24'h983939; //reddist 1
//     5: rgb_out = 24'h9B190A; //reddist 2
//     6: rgb_out = 24'hE84AFA; //magentaish
//     7: rgb_out = 24'hFFFFFF; //white
  
//   endcase
// end
// end

// endmodule


/*
* sprite_gen module
*
*
*
*
*
*/

module sprite_gen(
    input   logic         clk,
    input   logic         reset,
    input                 chipselect,
    input   logic [7:0]   address,
		input   logic [31:0]  writedata,
		input   logic         write,
    input   logic [8:0]	  hcount,
    input   logic [9:0]	  vga_vcount,
    input   logic [8:0]   vcount,
 
		output  logic [7:0]  fg_data);

   
    logic [4:0]  spriteRAM_addr;   //sprite MAP address (for 16 sprites)
    logic [4:0]  spriteRAM_raddr;  //sprite MAP read address (for 16 sprites)
    logic [4:0]  spriteRAM_waddr;

    logic        spriteRAM_wren;   //sprite MAP write enable
    logic [25:0] spriteRAM_data;   //sprite MAP read RAM data
    logic [25:0] spriteRAM_wdata;  //sprite MAP write RAM data

    logic [12:0] spriteROM_addr;   //3bit/pixel tile artwork tile ROM address

    logic [5:0]  spriteROM_pix;    //3bit sprite ROM pixel output

    logic [8:0] row0_addr, row1_addr, row_waddr,row_raddr;
    logic [8:0] row0_addr_b, row1_addr_b;
    logic       row0_wren, row1_wren, row_wren;
    logic       row0_wren_b,row1_wren_b;
    logic [7:0] row0_data,row1_data, row0_wdata,row1_wdata;
    logic drawpix;
    logic [8:0] row_next;

    //TODO:check width
    logic signed [8:0] sp_posY;
    logic signed [9:0] sp_posX;

    logic signed [8:0] sp_dy;
    logic signed [9:0] sp_dx;

    logic [1:0] sp_palID;
    logic [4:0] sp_ID;

    logic [3:0] spID_ptr;
    
   //Write sprite Map from software
   always_ff @(negedge clk) begin

     if (chipselect && write && address==1)begin //WRITE from driver
	      //WriteSpriteMap {[3:0]spriteMap_waddr, PosX[9:0], PosY[8:0], [4:0]spriteID, [1:0]palID} 
          spriteRAM_wdata <= writedata [25:0]; //{PosX, PosY, spriteID, palID}
          spriteRAM_waddr <= writedata  [30:26]; //{spriteMAP_addr - 5bits}
          spriteRAM_wren<=1;
       end
       else begin
       //reset
        spriteRAM_wdata<= 30'h0;
        spriteRAM_waddr<= spriteRAM_raddr;
        spriteRAM_wren<=0;
     end

  end
  //sprite ram read or write operation
  always_comb begin
    if(spriteRAM_wren)
     spriteRAM_addr =  spriteRAM_waddr;
     else
     spriteRAM_addr = spriteRAM_raddr; 
  end

    //sprite gen state machine

    enum {
        IDLE,       // await until hcount==0
        READ_WAIT,  // wait until drawpix
        DRAW        // draw
    } state;


//Row to write
  always_comb begin
    if(vcount==262) //end of vcount after it wraps up to 0
     row_next=0;
      else
     row_next=vcount+1;
  end

//Sprite generator FSM:

   always_ff @(negedge clk) begin
     if (reset)begin 
       state<=IDLE;
       spriteRAM_raddr<=0; //TODO: For testing only.
     end
     else

     case(state)

      IDLE:begin
        spriteRAM_raddr<=0;
        spID_ptr<=0;
        sp_dx<=0;
        if(hcount==0) state<=READ_WAIT;

      end

      READ_WAIT:begin

     if(drawpix)begin
        state<=DRAW;
      end else begin
        if (spriteRAM_raddr==15)
        state <=IDLE;
        else spriteRAM_raddr<=spriteRAM_raddr+1;
      end
      
      end
      
      DRAW:begin

        sp_dx<=sp_dx+1 ;

        if(sp_dx==15)begin
         sp_dx<=0;
         if (spriteRAM_raddr<15)begin
          spriteRAM_raddr<=spriteRAM_raddr+1;
          state<=READ_WAIT;
         end
         else
          state<=IDLE;
          
        end

      end

    endcase
   
   end
   
   //mux: select row0 or row1
   always_comb begin
    
    row0_wdata=0; row1_wdata=0;
    row1_wren = 0;  row0_wren = 0;
    row0_wren_b=0;row1_wren_b=0;
    //write to row 0
    //read from row 1
     if(vcount[0])begin
       row0_addr = row_waddr;
       row0_wren = row_wren && (spriteROM_pix!=0);
       row0_wdata = {spriteROM_pix,sp_palID[1:0]}; //3(unused)+3+2;
       row1_addr = row_raddr;
       fg_data = row1_data; //3bit final foreground pix. TODO: include palID in row
       if(vga_vcount[0]) row1_wren_b = 1;
     end
    //write to row 1
    //read from row 0
     else begin
       row1_addr = row_waddr;
       row1_wren = row_wren && (spriteROM_pix!=0);
       row1_wdata = {spriteROM_pix,sp_palID[1:0]}; //3(unused)+3+2;
       row0_addr = row_raddr;
       fg_data = row0_data;    //3bit final foreground pix. TODO: include palID in row
       if(vga_vcount[0]) row0_wren_b = 1;
  
     end

   end

   assign row0_addr_b = row0_addr -1;
   assign row1_addr_b = row1_addr -1;
   
   //Wrap around
   always_comb begin
    if (hcount<320)
    row_raddr = hcount;
    else row_raddr = 0;  
   end
 
  //One ff delay to match lines
 //assign row_waddr = sp_posX + sp_dx;
 //assign row_wren =  (state==DRAW && drawpix)? 1:0; 
  always_ff @(negedge clk)begin
    row_waddr <= sp_posX + sp_dx;
    row_wren  <=  (state==DRAW && drawpix)? 1:0; 
  end
  
 assign {sp_posX[9:0],sp_posY[8:0],sp_ID[4:0],sp_palID[1:0]} = spriteRAM_data;
 assign sp_dy = (row_next - sp_posY);
 assign spriteROM_addr = (sp_ID<<8) + sp_dx + (sp_dy<<4);
 assign drawpix = (sp_dy>=0 && sp_dy<15 && sp_dx<16);


   rowRAM2 row0(  .clock(clk),
	               .data_a(row0_wdata),
                 .address_a(row0_addr),
                 .wren_a(row0_wren),
	               .q_a(row0_data),
                 .data_b(8'd0),
                 .address_b(row0_addr_b),
                 .wren_b(row0_wren_b),
	               .q_b()
	               );

   rowRAM2 row1(  .clock(clk),
	               .data_a(row1_wdata),
                 .address_a(row1_addr),
                 .wren_a(row1_wren),
	               .q_a(row1_data),
                 .data_b(8'd0),
                 .address_b(row1_addr_b),
                 .wren_b(row1_wren_b),
	               .q_b()
	               );



   spriteROM spriteROM_1 (.clock(~clk),.address(spriteROM_addr),.q(spriteROM_pix));

   spriteRAM spriteRAM_1 (.clock(clk),
                       .address(spriteRAM_addr),
                       .wren(spriteRAM_wren),
                       .data(spriteRAM_wdata),
                       .q(spriteRAM_data));

	       
endmodule



/*
* vga_counters module:
* generates video signal
*
*
*/
module vga_counters(
 input logic 	     clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;
       
   logic endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule

/*
* audio module:
*
* uses 8bit audio clips
*/
module audio (
		input clk, // 50MHz
		input reset,
    input audio_on, //run audio + flying sound
    input logic [2:0]command, //0:fire, 1:fuel, 2:explode
		input left_chan_ready,
		input right_chan_ready,
		output logic [15:0] sample_data_l,
		output logic sample_valid_l,
		output logic [15:0] sample_data_r,
		output logic sample_valid_r
	);

    parameter DAC_RATE = 50000000/16000;
  //Define audio sample sizes
  //TODO: fill sample sizes appropriately 
    parameter AR_FLY_ADDR_ST=0;
    parameter AR_FIRE_ADDR_ST=0;
    parameter AR_FUEL_ADDR_ST=0;
    parameter AR_EXPLODE_ADDR_ST=0;

    parameter AR_FLY_ADDR_END=23213;
    parameter AR_FIRE_ADDR_END=1599;
    parameter AR_FUEL_ADDR_END=2039;
    parameter AR_EXPLODE_ADDR_END=4095;

  logic [11:0] dac_ctr;

  logic dac_ready;
  
  logic	[15:0]  audio_out;

  logic ar_fire_go;
  logic ar_fuel_go;
  logic ar_explode_go;
  
  logic ar_fly_play;
  logic [7:0] ar_fly_q;
  logic [15:0] ar_fly_dout;
  logic [14:0] ar_fly_rom_addr;

  logic ar_fire_play;
  logic [7:0] ar_fire_q;
  logic [15:0] ar_fire_dout;
  logic [10:0] ar_fire_rom_addr;

  logic ar_fuel_play;
  logic [7:0] ar_fuel_q;
  logic [15:0] ar_fuel_dout;
  logic [10:0] ar_fuel_rom_addr;

  logic ar_explode_play;
  logic [7:0] ar_explode_q;
  logic [15:0] ar_explode_dout;
  logic [11:0] ar_explode_rom_addr;


  //TODO: this could be done using a single ROM.
  audioROM_fly ar_fly(.address(ar_fly_rom_addr), .clock(clk), .q(ar_fly_q));
  audioROM_fire ar_fire(.address(ar_fire_rom_addr), .clock(clk), .q(ar_fire_q));
  audioROM_fuel ar_fuel(.address(ar_fuel_rom_addr), .clock(clk), .q(ar_fuel_q));
  audioROM_explode ar_explode(.address(ar_explode_rom_addr), .clock(clk), .q(ar_explode_q));


//Handle audio buffering and also background (flying) sound
  always_ff @(posedge clk) begin

    if(reset || !audio_on) begin
      dac_ctr <=0;
      sample_valid_l <= 0; sample_valid_r <= 0;
      audio_out<=0;
      ar_fly_rom_addr<=AR_FLY_ADDR_ST;
    end

    else if(audio_on && dac_ready && dac_ctr<DAC_RATE) begin

      sample_valid_l <= 0; sample_valid_r <= 0;
      dac_ctr<=dac_ctr+1;
    
    end else if(audio_on && dac_ready && dac_ctr==DAC_RATE)begin //send DAC output
        dac_ctr<=0;
        ar_fly_rom_addr<=ar_fly_rom_addr+1;
        
        if(ar_fly_rom_addr==AR_FLY_ADDR_END)begin
            ar_fly_rom_addr<=AR_FLY_ADDR_ST;
        end

        //final output registers
        audio_out <= ar_fly_dout + ar_fire_dout + ar_fuel_dout + ar_explode_dout;
        sample_valid_l <= 1; sample_valid_r <= 1;
    
    end 

  end


//AUDIO - FIRE
  //fire audio rom
  always_ff @(posedge clk) begin

      if (reset || !audio_on)begin
          ar_fire_play<=0;
      end else if (ar_fire_go) begin
          ar_fire_play<=1;
          ar_fire_rom_addr<=AR_FIRE_ADDR_ST;
      end
      else if (!ar_fire_go && ar_fire_play && dac_ready && dac_ctr==DAC_RATE) begin
          
          ar_fire_rom_addr<= ar_fire_rom_addr+1;

          if (ar_fire_rom_addr==AR_FIRE_ADDR_END) begin
              ar_fire_play<=0;
          end
      end
  end

  //AUDIO - FUEL
  //fuel audio rom
  always_ff @(posedge clk) begin

      if (reset || !audio_on)begin
          ar_fuel_play<=0;
      end else if (ar_fuel_go) begin
          ar_fuel_play<=1;
          ar_fuel_rom_addr<=AR_FUEL_ADDR_ST;
      end
      else if (!ar_fire_go && ar_fuel_play && dac_ready && dac_ctr==DAC_RATE) begin
         
          ar_fuel_rom_addr<= ar_fuel_rom_addr+1;
         
          if (ar_fuel_rom_addr==AR_FUEL_ADDR_END) begin
              ar_fuel_play<=0;
          end
      end
  end

  //AUDIO - EXPLODE
  //explode audio rom
  always_ff @(posedge clk) begin

      if (reset || !audio_on)begin
          ar_explode_play<=0;
      end else if (ar_explode_go) begin
          ar_explode_play<=1;
          ar_explode_rom_addr<=AR_EXPLODE_ADDR_ST;
      end
      else if (!ar_fire_go && ar_explode_play && dac_ready && dac_ctr==DAC_RATE) begin
          ar_explode_rom_addr<= ar_explode_rom_addr+1;
          if (ar_explode_rom_addr==AR_EXPLODE_ADDR_END) begin
              ar_explode_play<=0;
          end
      end
  end

  assign ar_fly_dout= {8'h0, ar_fly_q};
  assign ar_fire_dout= {8'h0,ar_fire_q};//(ar_fire_play)? {8'h0,ar_fire_q}:0;
  assign ar_fuel_dout= {8'h0,ar_fuel_q};//(ar_fuel_play)? {8'h0,ar_fuel_q}:0;
  assign ar_explode_dout= {8'h0,ar_explode_q};//(ar_explode_play)? {8'h0,ar_explode_q}:0;

  assign ar_fire_go = command[0];
  assign ar_fuel_go = command[1];
  assign ar_explode_go = command[2];

  assign dac_ready = (left_chan_ready && right_chan_ready);
  assign sample_data_l = audio_out<<5;
  assign sample_data_r = audio_out<<5;

 endmodule