/* * Device driver for the variable fir module
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod fir.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree fir.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "fir.h"

#define DRIVER_NAME "fir"

/* Device registers */


int z=0;
/*
 * Information about our device
 */
struct fir_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
        fir_param_t x;
} dev;

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
static void write_fir_param(fir_param_t *x)
{
	for (z=0;z<500;z++){
	iowrite16(x->fircoeff[z], (dev.virtbase+2*z) );
	}
    dev.x = *x;

}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long fir_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	fir_arg_t vla;

	switch (cmd) {
	
	
	case FIR_WRITE_PARAM:
		if (copy_from_user(&vla, (fir_arg_t *) arg, sizeof(fir_arg_t)))
			return -EACCES;
		write_fir_param(&vla.firparam);
		break;
		
		
	case FIR_READ_PARAM:
	  	vla.firparam = dev.x;
		if (copy_to_user((fir_arg_t *) arg, &vla, sizeof(fir_arg_t)))
			return -EACCES;
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations fir_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = fir_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice fir_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &fir_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init fir_probe(struct platform_device *pdev)
{
        fir_param_t beige =    {{  15369, 32767, 26902, 13680, 8056, 3137, -4996, -15891, -22587, -15494, -4256, 586, 349, -1217, 2072, 4979, 1068, -922, -386, 892, 7679, 10773, 3222, -2272, -1960, -2079, -2618, -4738, -5895, -2462, 1068, 3137, 2055, -1784, 539, 6166, 4087, -335, -3707, -7377, -6319, -3771, -3812, -2984, 1224, 3459, 362, -3826, -5345, -5193, -2845, 1512, 3096, 3385, 4847, 3235, -2737, -6434, -4799, -1370, 464, 681, 244, -1173, -3039, -3626, -4104, -4986, -4178, -2408, -1563, -410, 817, 1492, 1380, -23, -1048, -1926, -3127, -3012, -2978, -4067, -4148, -3982, -4033, -2781, -1153, 88, 1414, 2177, 2174, 1624, 325, -546, -613, -1685, -3408, -4182, -3497, -1811, -240, 1122, 2120, 1316, -766, -2120, -3073, -3602, -2442, -831, 142, 546, 593, 447, 10, -461, -78, 437, -179, -1180, -1882, -2309, -2242, -1465, -362, 651, 1373, 1302, 264, -576, -491, 27, 603, 437, -457, -756, -661, -254, 851, 1794, 2255, 2557, 2235, 1455, 759, 27, -471, -739, -956, -973, -854, -885, -858, -861, -1075, -671, 284, 1082, 1845, 2428, 2672, 2666, 2062, 841, -267, -807, -864, -773, -736, -583, -223, 247, 691, 1092, 1536, 2045, 2306, 2014, 1288, 753, 708, 688, 427, -88, -644, -800, -586, -142, 539, 1092, 1187, 953, 552, 596, 1197, 1567, 1543, 1282, 949, 695, 335, -217, -508, -447, -240, 61, 176, 186, 237, 274, 437, 841, 1326, 1784, 1930, 1655, 1451, 1322, 1085, 824, 451, 128, 47, 67, 227, 454, 651, 970, 1177, 1156, 1122, 888, 518, 251, 40, -57, 30, 189, 457, 644, 586, 675, 797, 725, 593, 315, 13, 40, 291, 542, 637, 596, 705, 915, 1027, 942, 824, 915, 1071, 1058, 868, 603, 329, 135, 61, 23, -30, -6, 169, 400, 546, 627, 790, 956, 987, 820, 430, 98, 20, 47, 37, -20, -98, -145, -179, -156, 108, 535, 888, 1136, 1265, 1126, 786, 485, 237, 40, -40, -115, -122, 47, 88, -37, 37, 139, 91, 142, 281, 376, 451, 424, 220, 33, 54, 125, 13, -189, -257, -203, -206, -172, -16, 125, 156, -3, -278, -345, -142, 30, 105, 162, 271, 447, 379, 10, -257, -281, -301, -349, -383, -376, -322, -312, -220, 67, 301, 356, 410, 362, 98, -179, -342, -410, -535, -746, -820, -746, -603, -400, -267, -210, -64, 30, -64, -196, -206, -101, -16, 10, 57, 115, 78, 0, -108, -230, -274, -261, -267, -308, -312, -234, -105, -71, -159, -234, -200, -101, -132, -322, -508, -610, -641, -691, -773, -790, -708, -525, -247, -61, 10, 81, 145, 172, 105, -30, -101, -122, -234, -413, -593, -719, -753, -695, -542, -383, -244, -61, 122, 230, 230, 91, -84, -203, -339, -468, -518, -549, -596, -675, -776, -837, -783, -593, -349, -179, -78, 23, 108, 115, 44, -47, -54, -6, -13, -98, -237, -376, -481, -556, -580, -539, -417, -234, -67, 6, -6, -57, -101, -81, -71, -132, -244, -366, -447, -474, -505, -451, -284, -172, -169, -220, -298, -325, -271, -176, -94, -47, -50, -78, -111, -111, -50, -13, -20, -40, -108, -240, -335, -356, -359, -373, -383, -390, -373, -318, -251, -183, -71, 54, 81, 13}};

	int ret;

	/* Register ourselves as a misc device: creates /dev/fir */
	ret = misc_register(&fir_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	/* Set an initial color */
        write_fir_param(&beige);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&fir_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int fir_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&fir_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id fir_of_match[] = {
	{ .compatible = "fir,fir-21.1" },
	{},
};
MODULE_DEVICE_TABLE(of, fir_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver fir_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(fir_of_match),
	},
	.remove	= __exit_p(fir_remove),
};

/* Called when the module is loaded: set things up */
static int __init fir_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&fir_driver, fir_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit fir_exit(void)
{
	platform_driver_unregister(&fir_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(fir_init);
module_exit(fir_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("LILONGYINIUBI, Columbia University");
MODULE_DESCRIPTION("fir driver");
