/*
 * Avalon memory-mapped peripheral that generates VGA
 *
 * Stephen A. Edwards
 * Columbia University
 */

module vga_ball(input logic        clk,
	        input logic 	   reset,
		
		//		input logic bypass,
		input logic [15:0]  writedata,
		input logic 	   write,
		input 		   chipselect,
		input logic [2:0]  address,

		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output logic 	   VGA_CLK, VGA_HS, VGA_VS,
		                   VGA_BLANK_n,
		output logic 	   VGA_SYNC_n);

   logic [10:0]	   hcount;
   logic [9:0]     vcount;
	
	
	logic      bypass;
	logic [9:0] sel_x=40;//select for FIR
	logic [9:0] sel_y;//config by sw
	
	logic [9:0] sw_x=373;//
	logic [9:0] sw_y=139;//
	
	logic [9:0] slider1_x=187;//slider for clipping
	logic [9:0] slider1_y;//config by sw

	logic [9:0] slider2_x=278;//slider  
	logic [9:0] slider2_y;//config by sw

	logic [9:0] slider3_x=324;//slider  
	logic [9:0] slider3_y;//config by sw


	logic [9:0] slider4_x=464;//slider  
	logic [9:0] slider4_y;//config by sw

	logic [9:0] slider5_x=510;//slider  
	logic [9:0] slider5_y;//config by sw

	logic [9:0] slider6_x=556;//slider 
	logic [9:0] slider6_y;//config by sw


	
	
   vga_counters counters(.clk50(clk), .*);


   always_ff @(posedge clk)
     if (reset) begin
	sel_y <= 132;
	slider1_y <= 232;
	slider2_y <= 289;
	slider3_y <= 255;
	slider4_y <= 150;
	slider5_y <= 255;
	slider6_y <= 150;
	bypass<=1;
     end else if (chipselect && write)
       case (address)
	 3'h0 : bypass <= writedata[0];
	 3'h1 : sel_y <= writedata[9:0];
	 3'h2 : slider1_y <= writedata[9:0];
	 3'h3 : slider2_y <= writedata[9:0];
	 3'h4 : slider3_y <= writedata[9:0];
	 3'h5 : slider4_y <= writedata[9:0];
	 3'h6 : slider5_y <= writedata[9:0];
	 3'h7 : slider6_y <= writedata[9:0];

       endcase

logic [9:0]hcnt;
logic [18:0] bgaddr;
logic [7:0] BG,SWON,SWOFF;
logic [7:0] seladdr;
logic [7:0] S1;
always_comb  begin
hcnt=hcount[10:1];
bgaddr=vcount*640+hcnt;
end
BG	B1 (
	.address ( bgaddr ),
	.clock ( clk ),
	.q (BG  )
	);

SEL	SEL1 (
	.address ( (hcnt-sel_x)+(vcount-sel_y)*13 +1),
	.clock ( clk ),
	.q (S1  )
	);
	


SWON	SW1 (
	.address ( (hcnt-sw_x)+(vcount-sw_y)*38 ),
	.clock ( clk ),
	.q (SWON  )
	);
	
SWOFF	SW2 (
	.address ( (hcnt-sw_x)+(vcount-sw_y)*38 ),
	.clock ( clk ),
	.q (SWOFF  )
	);
	


   always_ff @(posedge clk) begin
	if(reset) begin
					

	end
	else begin
		if (VGA_BLANK_n ) begin
		if(hcnt>=slider1_x&&hcnt<slider1_x+36&&vcount>=slider1_y&&vcount<slider1_y+17)begin
			VGA_R<=255;
			VGA_G<=255;
			VGA_B<=255;


		end
		else if(hcnt>=slider2_x&&hcnt<slider2_x+36&&vcount>=slider2_y&&vcount<slider2_y+17)begin
			VGA_R<=255;
			VGA_G<=255;
			VGA_B<=255;
		end
		else if(hcnt>=slider3_x&&hcnt<slider3_x+36&&vcount>=slider3_y&&vcount<slider3_y+17)begin
			VGA_R<=255;
			VGA_G<=255;
			VGA_B<=255;

		end
		else if(hcnt>=slider4_x&&hcnt<slider4_x+36&&vcount>=slider4_y&&vcount<slider4_y+17)begin
			VGA_R<=255;
			VGA_G<=255;
			VGA_B<=255;

		end
		else	if(hcnt>=slider5_x&&hcnt<slider5_x+36&&vcount>=slider5_y&&vcount<slider5_y+17)begin
			VGA_R<=255;
			VGA_G<=255;
			VGA_B<=255;

		end
		else if(hcnt>=slider6_x&&hcnt<slider6_x+36&&vcount>=slider6_y&&vcount<slider6_y+17)begin
			VGA_R<=255;
			VGA_G<=255;
			VGA_B<=255;

		end
		else if(hcnt>=sel_x&&hcnt<sel_x+13&&vcount>=sel_y&&vcount<sel_y+13)begin
		VGA_R[7:5]<=S1[2:0];
		VGA_G[7:5]<=S1[5:3];
		VGA_B[7:6]<=S1[7:6];
		end
		else if(hcnt>=sw_x&&hcnt<sw_x+38&&vcount>=sw_y&&vcount<sw_y+20)begin
		if (bypass)begin
			VGA_R[7:5]<=SWOFF[2:0];
			VGA_G[7:5]<=SWOFF[5:3];
			VGA_B[7:6]<=SWOFF[7:6];
			end else begin
			VGA_R[7:5]<=SWON[2:0];
			VGA_G[7:5]<=SWON[5:3];
			VGA_B[7:6]<=SWON[7:6];
			end

		end
		else begin
		VGA_R[7:5]<=BG[2:0];
		VGA_R[4:0]<=5'b0;

		VGA_G[7:5]<=BG[5:3];
		VGA_G[4:0]<=5'b0;

		VGA_B[7:6]<=BG[7:6];
		VGA_B[5:0]<=6'b0;
		end
		end
	  end
   end
	       
endmodule

module vga_counters(
 input logic 	     clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;
       
   logic endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule

