module readOCM(
	input	logic						clk, reset,

	// ------------ Read weight and bias ------------
	// Input from HPS
	input	logic						start,
	input	logic	[15: 0]				read_length,
	input	logic	[5: 0]				read_data_dim,

	// Output to pipeline
	output	logic						in_data_ready,
	output	logic	[(3*3+1)*16-1: 0]	weight_bias,
	// ----------------------------------------------
	
	// ------------ Read feature map 3x3 ------------
	// Input from pipeline
	input	logic						start_fm,
	input	logic	[15: 0]				conv_idx,

	// Output to pipeline
	output	logic						finish_fm,
	output	logic	[3*3*16-1: 0]		feat_map_in,
	// ----------------------------------------------
	

	// On-Chip RAM 0 s2 (read)
	input	logic	[ 7: 0]				ocm0_readdata,
	output	logic	[16: 0]				ocm0_addr,
	output	logic						ocm0_chip,
	output	logic						ocm0_clk_enab,

	// Debug
	output	logic	[ 2: 0]				debug_state
);

	// FSM
	logic [ 2: 0]	state;
	assign debug_state = state;
	parameter S0 = 0, S1 = 1, S2 = 2, S3 = 3, S4 = 4, S5 = 5, S6 = 6, S7 = 7;

	// For OCM Read
	logic [16: 0]	addr = 0;
	logic [15: 0]	read_counter = 0;


	// FM Request from conv
	// 0 Padding checks
	logic [2:0]			zero_padding_mask;
	logic				left_col_padding;
	logic				right_col_padding;

	logic [8:0][15:0]	req_fm_addr;
	logic [3:0]			fm_save_idx;

    always_comb begin
        case (read_data_dim)
            32: 
                begin
                    zero_padding_mask = 5;
                    left_col_padding = (conv_idx[0 +: 5] == 0)? 1 : 0;
                    right_col_padding = (conv_idx[0 +: 5] == read_data_dim-1)? 1 : 0;
                end
            16: 
                begin
                    zero_padding_mask = 4;
                    left_col_padding = (conv_idx[0 +: 4] == 0)? 1 : 0;
                    right_col_padding = (conv_idx[0 +: 4] == read_data_dim-1)? 1 : 0;
                end
            8: 
                begin
                    zero_padding_mask = 3;
                    left_col_padding = (conv_idx[0 +: 3] == 0)? 1 : 0;
                    right_col_padding = (conv_idx[0 +: 3] == read_data_dim-1)? 1 : 0;
                end
            4: 
                begin
                    zero_padding_mask = 2;
                    left_col_padding = (conv_idx[0 +: 2] == 0)? 1 : 0;
                    right_col_padding = (conv_idx[0 +: 2] == read_data_dim-1)? 1 : 0;
                end
            2: 
                begin
                    zero_padding_mask = 1;
                    left_col_padding = (conv_idx[0 +: 1] == 0)? 1 : 0;
                    right_col_padding = (conv_idx[0 +: 1] == read_data_dim-1)? 1 : 0;
                end
            default:
                begin
                    // ERROR
                    zero_padding_mask = 0;
                    left_col_padding = 0;
                    right_col_padding = 0;
                end
		endcase
    end


	always @ (posedge clk) begin
		case (state)
			S0:	// reset
				begin
					addr = 0;
					ocm0_clk_enab = 0;
					in_data_ready = 0;
				end
			S1:	// prepare to read
				begin
					read_counter = 0;
					
					ocm0_addr = addr;
					ocm0_chip = 1;
					ocm0_clk_enab = 1;
					
					addr = addr + 1;
				end
			S2:	// Read
				begin
					ocm0_addr = addr;
					ocm0_chip = 1;
					ocm0_clk_enab = 1;

					weight_bias[(read_counter)*8 +: 8] = ocm0_readdata;
					
					addr = addr + 1;
					read_counter = read_counter + 1;
				end
			S3:	// Finished reading weight bias. Wait for FM request
				begin
					in_data_ready = 1;
					finish_fm = 0;
					ocm0_clk_enab = 0;
				end
			S4: // FM request, prepare read
				begin
					fm_save_idx = 0;
					if (conv_idx == 0)
                        begin
                            req_fm_addr[0] = 20+conv_idx*2+read_data_dim*2+2;
                            req_fm_addr[1] = 20+conv_idx*2+read_data_dim*2;
                            req_fm_addr[2] = 16'b0;
                            req_fm_addr[3] = 20+conv_idx*2+2;
                            req_fm_addr[4] = 20+conv_idx*2;
                            req_fm_addr[5] = 16'b0;
                            req_fm_addr[6] = 16'b0;
                            req_fm_addr[7] = 16'b0;
                            req_fm_addr[8] = 16'b0;
                        end
                    else if (conv_idx == read_data_dim-1)
                        begin
                            req_fm_addr[0] = 16'b0;
                            req_fm_addr[1] = 20+conv_idx*2+read_data_dim*2;
                            req_fm_addr[2] = 20+conv_idx*2+read_data_dim*2-2;
                            req_fm_addr[3] = 16'b0;
                            req_fm_addr[4] = 20+conv_idx*2;
                            req_fm_addr[5] = 20+conv_idx*2-2;
                            req_fm_addr[6] = 16'b0;
                            req_fm_addr[7] = 16'b0;
                            req_fm_addr[8] = 16'b0;
                        end
                    else if (conv_idx == read_data_dim*(read_data_dim-1))
                        begin
                            req_fm_addr[0] = 16'b0;
                            req_fm_addr[1] = 16'b0;
                            req_fm_addr[2] = 16'b0;
                            req_fm_addr[3] = 20+conv_idx*2+2;
                            req_fm_addr[4] = 20+conv_idx*2;
                            req_fm_addr[5] = 16'b0;
                            req_fm_addr[6] = 20+conv_idx*2-read_data_dim*2+2;
                            req_fm_addr[7] = 20+conv_idx*2-read_data_dim*2;
                            req_fm_addr[8] = 16'b0;
                        end
                    else if (conv_idx == read_data_dim*read_data_dim-1)
                        begin
                            req_fm_addr[0] = 16'b0;
                            req_fm_addr[1] = 16'b0;
                            req_fm_addr[2] = 16'b0;
                            req_fm_addr[3] = 16'b0;
                            req_fm_addr[4] = 20+conv_idx*2;
                            req_fm_addr[5] = 20+conv_idx*2-2;
                            req_fm_addr[6] = 16'b0;
                            req_fm_addr[7] = 20+conv_idx*2-read_data_dim*2;
                            req_fm_addr[8] = 20+conv_idx*2-read_data_dim*2-2;
                        end
                    else if (left_col_padding)
                        begin
                            req_fm_addr[0] = 20+conv_idx*2+read_data_dim*2+2;
                            req_fm_addr[1] = 20+conv_idx*2+read_data_dim*2;
                            req_fm_addr[2] = 16'b0;
                            req_fm_addr[3] = 20+conv_idx*2+2;
                            req_fm_addr[4] = 20+conv_idx*2;
                            req_fm_addr[5] = 16'b0;
                            req_fm_addr[6] = 20+conv_idx*2-read_data_dim*2+2;
                            req_fm_addr[7] = 20+conv_idx*2-read_data_dim*2;
                            req_fm_addr[8] = 16'b0;
                        end
                    else if (right_col_padding)
                        begin
                            req_fm_addr[0] = 16'b0;
                            req_fm_addr[1] = 20+conv_idx*2+read_data_dim*2;
                            req_fm_addr[2] = 20+conv_idx*2+read_data_dim*2-2;
                            req_fm_addr[3] = 16'b0;
                            req_fm_addr[4] = 20+conv_idx*2;
                            req_fm_addr[5] = 20+conv_idx*2-2;
                            req_fm_addr[6] = 16'b0;
                            req_fm_addr[7] = 20+conv_idx*2-read_data_dim*2;
                            req_fm_addr[8] = 20+conv_idx*2-read_data_dim*2-2;
                        end
                    else if (conv_idx < read_data_dim-1)
                        begin
                            req_fm_addr[0] = 20+conv_idx*2+read_data_dim*2+2;
                            req_fm_addr[1] = 20+conv_idx*2+read_data_dim*2;
                            req_fm_addr[2] = 20+conv_idx*2+read_data_dim*2-2;
                            req_fm_addr[3] = 20+conv_idx*2+2;
                            req_fm_addr[4] = 20+conv_idx*2;
                            req_fm_addr[5] = 20+conv_idx*2-2;
                            req_fm_addr[6] = 16'b0;
                            req_fm_addr[7] = 16'b0;
                            req_fm_addr[8] = 16'b0;
                        end
                    else if (conv_idx > read_data_dim*(read_data_dim-1))
                        begin
                            req_fm_addr[0] = 16'b0;
                            req_fm_addr[1] = 16'b0;
                            req_fm_addr[2] = 16'b0;
                            req_fm_addr[3] = 20+conv_idx*2+2;
                            req_fm_addr[4] = 20+conv_idx*2;
                            req_fm_addr[5] = 20+conv_idx*2-2;
                            req_fm_addr[6] = 20+conv_idx*2-read_data_dim*2+2;
                            req_fm_addr[7] = 20+conv_idx*2-read_data_dim*2;
                            req_fm_addr[8] = 20+conv_idx*2-read_data_dim*2-2;
                        end
                    else
                        begin
                            req_fm_addr[0] = 20+conv_idx*2+read_data_dim*2+2;
                            req_fm_addr[1] = 20+conv_idx*2+read_data_dim*2;
                            req_fm_addr[2] = 20+conv_idx*2+read_data_dim*2-2;
                            req_fm_addr[3] = 20+conv_idx*2+2;
                            req_fm_addr[4] = 20+conv_idx*2;
                            req_fm_addr[5] = 20+conv_idx*2-2;
                            req_fm_addr[6] = 20+conv_idx*2-read_data_dim*2+2;
                            req_fm_addr[7] = 20+conv_idx*2-read_data_dim*2;
                            req_fm_addr[8] = 20+conv_idx*2-read_data_dim*2-2;
                        end

					ocm0_addr = req_fm_addr[fm_save_idx];
					ocm0_chip = 1;
					ocm0_clk_enab = 1;

					fm_save_idx = fm_save_idx + 1;
				end
			S5:  // Read FM 8 LSB
				begin
					if (req_fm_addr[fm_save_idx-1] != 0) begin
						feat_map_in[(fm_save_idx-1)*16 +: 8] = ocm0_readdata;
						ocm0_addr = ocm0_addr+1;
					end
				end
			S6:  // Read FM 8 MSB
				begin
					if (req_fm_addr[fm_save_idx-1] != 0) begin
						feat_map_in[(fm_save_idx-1)*16+8 +: 8] = ocm0_readdata;
					end
					ocm0_addr = req_fm_addr[fm_save_idx];
                    fm_save_idx = fm_save_idx + 1;
				end
			S7:  // Finished FM reading
				begin
					finish_fm = 1;
				end
		endcase
	end

	// Determine the next state
	always @ (posedge clk or posedge reset or negedge start) begin
		if (reset || ~start)
			begin
				state <= S0;
			end
		else
			case (state)
				S0:
					if(start)
						state <= S1;
				S1:
					state <= S2;
				S2:
					begin
						if(addr >= 20)
							state <= S3;
					end
				S3:
					if(start_fm)
						state <= S4;
				S4:
					state <= S5;
				S5:
					state <= S6;
				S6:
					if(fm_save_idx >= 9)
						state <= S7;
					else
						state <= S5;
				S7:
					if(start_fm == 0)
						state <= S3;
			endcase
	end

endmodule
