/*
 * Userspace program that communicates with the vga_ball device driver
 * through ioctls
 *
 * Stephen A. Edwards
 * Columbia University
 */

#include <stdio.h>
#include "vga_ball.h"
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <math.h>

struct rect {
    int x, y;
    int width, height;
    int border_radius;
    int r, g, b;
};

struct line {
    int start_col, end_col;
    int r, g, b;
};

#define NUM_RECTS 24

struct rect rectangles[NUM_RECTS] = {
   // head
    {.x = 200, .y = 10,
     .width = 200, .height = 200,
     .border_radius = 100,
     .r = 210, .g = 106, .b = 69},

   // left ear
    {.x = 200, .y = 0,
     .width = 50, .height = 50,
     .border_radius = 20,
     .r = 210, .g = 106, .b = 69},

   // right ear
    {.x = 350, .y = 0,
     .width = 50, .height = 50,
     .border_radius = 20,
     .r = 210, .g = 106, .b = 69},

   // left inner ear
    {.x = 206, .y = 7,
     .width = 30, .height = 30,
     .border_radius = 13,
     .r = 246, .g = 196, .b = 167},

   // right inner ear
    {.x = 362, .y = 7,
     .width = 30, .height = 30,
     .border_radius = 13,
     .r = 246, .g = 196, .b = 167},

   // left eye
    {.x = 276, .y = 120,
     .width = 10, .height = 10,
     .border_radius = 5,
     .r = 50, .g = 50, .b = 50},

   // right eye
    {.x = 320, .y = 120,
     .width = 10, .height = 10,
     .border_radius = 5,
     .r = 50, .g = 50, .b = 50},

   // nose
    {.x = 292, .y = 119,
     .width = 26, .height = 26,
     .border_radius = 13,
     .r = 160, .g = 81, .b = 53},

   // left moustache
    {.x = 280, .y = 130,
     .width = 30, .height = 30,
     .border_radius = 13,
     .r = 255, .g = 255, .b = 255},

   // right moustache
    {.x = 300, .y = 130,
     .width = 30, .height = 30,
     .border_radius = 13,
     .r = 255, .g = 255, .b = 255},

   // first left stripe
    {.x = 202, .y = 110,
     .width = 37, .height = 8,
     .border_radius = 4,
     .r = 0, .g = 0, .b = 0},

   // first right stripe
    {.x = 362, .y = 110,
     .width = 37, .height = 8,
     .border_radius = 4,
     .r = 0, .g = 0, .b = 0},

   // second left stripe
    {.x = 205, .y = 130,
     .width = 30, .height = 9,
     .border_radius = 4,
     .r = 0, .g = 0, .b = 0},

   // second right stripe
    {.x = 366, .y = 130,
     .width = 30, .height = 9,
     .border_radius = 4,
     .r = 0, .g = 0, .b = 0},

   // first top stripe
    {.x = 274, .y = 13,
     .width = 10, .height = 40,
     .border_radius = 4,
     .r = 0, .g = 0, .b = 0},

   // second top stripe
    {.x = 295, .y = 11,
     .width = 10, .height = 40,
     .border_radius = 4,
     .r = 0, .g = 0, .b = 0},

   // third top stripe
    {.x = 316, .y = 13,
     .width = 10, .height = 40,
     .border_radius = 4,
     .r = 0, .g = 0, .b = 0},

   // left newspaper
    {.x = 160, .y = 180,
     .width = 150, .height = 200,
     .border_radius = 0,
     .r = 204, .g = 204, .b = 204},

   // right newspaper
    {.x = 310, .y = 180,
     .width = 150, .height = 200,
     .border_radius = 0,
     .r = 187, .g = 187, .b = 187},

   // top headline
    {.x = 170, .y = 210,
     .width = 120, .height = 10,
     .border_radius = 0,
     .r = 174, .g = 174, .b = 174},

   // bottom headline
    {.x = 185, .y = 230,
     .width = 120, .height = 10,
     .border_radius = 0,
     .r = 174, .g = 174, .b = 174},

   // newspaper picture
    {.x = 185, .y = 256,
     .width = 100, .height = 60,
     .border_radius = 0,
     .r = 174, .g = 174, .b = 174},

   // left hand
    {.x = 150, .y = 270,
     .width = 30, .height = 30,
     .border_radius = 15,
     .r = 210, .g = 106, .b = 69},

   // right hand
    {.x = 440, .y = 270,
     .width = 30, .height = 30,
     .border_radius = 15,
     .r = 210, .g = 106, .b = 69},

};

struct line rows[480][8];
int row_counters[480];

void initialize_rows() {
    for (int i = 0; i < 480; i++) {
        for (int num = 0; num < 8; num++) {
            rows[i][num].start_col = 0;
            rows[i][num].end_col = 0;
            rows[i][num].r = 0;
            rows[i][num].g = 0;
            rows[i][num].b = 0;
        }
        row_counters[i] = 0;
    }
}

int rect_start_col(struct rect *obj, int i) {
    if (i < obj->y + obj->border_radius) {
        int height = obj->y + obj->border_radius - i;
        int delta = (int) sqrt(pow(obj->border_radius, 2) - pow(height, 2));
        return obj->x + obj->border_radius - delta;
    } else if (i > obj->y + obj->height - obj->border_radius) {
        int height = i - (obj->y + obj->height - obj->border_radius);
        int delta = (int) sqrt(pow(obj->border_radius, 2) - pow(height, 2));
        return obj->x + obj->border_radius - delta;
    } else {
        return obj->x;
    }
}

int rect_end_col(struct rect *obj, int i) {
    if (i < obj->y + obj->border_radius) {
        int height = obj->y + obj->border_radius - i;
        int delta = (int) sqrt(pow(obj->border_radius, 2) - pow(height, 2));
        return obj->x + obj->width - (obj->border_radius - delta);
    } else if (i > obj->y + obj->height - obj->border_radius) {
        int height = i - (obj->y + obj->height - obj->border_radius);
        int delta = (int) sqrt(pow(obj->border_radius, 2) - pow(height, 2));
        return obj->x + obj->width - (obj->border_radius - delta);
    } else {
        return obj->x + obj->width;
    }
}

void add_rect(struct rect *obj) {
    for (int i = obj->y; i < obj->y + obj->height; i++) {
        if (row_counters[i] > 7) {
            printf("too many overlapping objects on row %d\n", i);
            return;
        }
        struct line curr = {
            .start_col = rect_start_col(obj, i),
            .end_col = rect_end_col(obj, i),
            .r = obj->r, .g = obj->g, .b = obj->b
        };
        rows[i][row_counters[i]++] = curr;
    }
}

void add_all_rects() {
    for (int j = 0; j < NUM_RECTS; j++) {
        add_rect(rectangles + j);
    }
}

int vga_ball_fd;

void write_rows() {
    for (int i = 0; i < 480; i++) {
        for (int num = 0; num < 8; num++) {
            struct line curr = rows[i][num];
            vga_ball_arg_t vla;
            vla.obj.row = i;
            vla.obj.num = num;
            vla.obj.line_r = curr.r;
            vla.obj.line_g = curr.g;
            vla.obj.line_b = curr.b;
            vla.obj.start_col = curr.start_col;
            vla.obj.end_col = curr.end_col;
            if (ioctl(vga_ball_fd, VGA_BALL_WRITE_OBJ, &vla)) {
                perror("ioctl(VGA_BALL_WRITE_OBJ) failed");
                return;
            }
        }
    }
}

int main()
{
  vga_ball_arg_t vla;
  int i;
  static const char filename[] = "/dev/vga_ball";

  printf("VGA ball Userspace program started\n");

  if ( (vga_ball_fd = open(filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename);
    return -1;
  }

  initialize_rows();
  add_all_rects();
  write_rows();

}

