/*
 * Avalon memory-mapped peripheral that generates VGA
 *
 * Stephen A. Edwards
 * Columbia University
 */

module vga_ball(input logic        clk,
	        input logic 	   reset,
		input logic [7:0]  writedata,
		input logic 	   write,
		input 		   chipselect,
		input logic [15:0]  address,

		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output logic 	   VGA_CLK, VGA_HS, VGA_VS,
		                   VGA_BLANK_n,
		output logic 	   VGA_SYNC_n);

    // vga params
    logic [10:0] hcount;
    logic [9:0] vcount;

    // displaylist
    logic [9:0] a;
    logic [6:0][7:0][7:0] din, dout;
    logic we;
    displaylist m( .* );

    // write displaylist
    logic [7:0][7:0] w_start_col_1, w_start_col_2, w_end_col_1, w_end_col_2;
    logic [7:0][7:0] w_line_r, w_line_g, w_line_b;

    always_ff @(posedge clk)
        if (chipselect && write) begin
            we <= 0;
            case (address[2:0])
                3'd0 : w_start_col_1[address[5:3]] <= writedata;
                3'd1 : w_start_col_2[address[5:3]] <= writedata;
                3'd2 : w_end_col_1[address[5:3]] <= writedata;
                3'd3 : w_end_col_2[address[5:3]] <= writedata;
                3'd4 : w_line_r[address[5:3]] <= writedata;
                3'd5 : w_line_g[address[5:3]] <= writedata;
                3'd6 : w_line_b[address[5:3]] <= writedata;
            endcase
            if (address[5:3] == 3'd7 && address[2:0] == 3'd6) begin
                we <= 1;
                din <= {w_start_col_1, w_start_col_2, w_end_col_1, w_end_col_2, w_line_r, w_line_g, w_line_b};
                a <= address[15:6];
            end
        end else begin
            we <= 0;
            a <= vcount;
        end

    // read displaylist
    logic [7:0][7:0] start_col_1, start_col_2, end_col_1, end_col_2;
    logic [7:0][9:0] start_col, end_col;
    logic [7:0][7:0] line_r, line_g, line_b;

    always_comb begin
        {start_col_1, start_col_2, end_col_1, end_col_2, line_r, line_g, line_b} = dout;
    end

    genvar i;
    generate
        for (i=0; i<8; i++) begin: generated
            always_comb begin
                start_col[i] = {start_col_1[i], start_col_2[i][1:0]};
                end_col[i] = {end_col_1[i], end_col_2[i][1:0]};
            end
        end
    endgenerate

    // set vga inputs
    vga_counters counters(.clk50(clk), .*);

    always_comb begin
        if (hcount[10:1] >= start_col[7] && hcount[10:1] < end_col[7])
            {VGA_R, VGA_G, VGA_B} = {line_r[7], line_g[7], line_b[7]};
        else if (hcount[10:1] >= start_col[6] && hcount[10:1] < end_col[6])
            {VGA_R, VGA_G, VGA_B} = {line_r[6], line_g[6], line_b[6]};
        else if (hcount[10:1] >= start_col[5] && hcount[10:1] < end_col[5])
            {VGA_R, VGA_G, VGA_B} = {line_r[5], line_g[5], line_b[5]};
        else if (hcount[10:1] >= start_col[4] && hcount[10:1] < end_col[4])
            {VGA_R, VGA_G, VGA_B} = {line_r[4], line_g[4], line_b[4]};
        else if (hcount[10:1] >= start_col[3] && hcount[10:1] < end_col[3])
            {VGA_R, VGA_G, VGA_B} = {line_r[3], line_g[3], line_b[3]};
        else if (hcount[10:1] >= start_col[2] && hcount[10:1] < end_col[2])
            {VGA_R, VGA_G, VGA_B} = {line_r[2], line_g[2], line_b[2]};
        else if (hcount[10:1] >= start_col[1] && hcount[10:1] < end_col[1])
            {VGA_R, VGA_G, VGA_B} = {line_r[1], line_g[1], line_b[1]};
        else if (hcount[10:1] >= start_col[0] && hcount[10:1] < end_col[0])
            {VGA_R, VGA_G, VGA_B} = {line_r[0], line_g[0], line_b[0]};
        else
            {VGA_R, VGA_G, VGA_B} = {8'd255, 8'd255, 8'd255};
    end

endmodule

module displaylist(input logic    clk,
              input logic [9:0]   a,
              input logic [6:0][7:0][7:0]  din,
              input logic         we,
              output logic [6:0][7:0][7:0] dout);

              logic [6:0][7:0][7:0] mem [1023:0];

              always_ff @(posedge clk) begin
              if (we) mem[a] <= din;
                 dout <= mem[a];
              end
endmodule

module vga_counters(
 input logic 	     clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;
       
   logic endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule
