/*
 * Avalon memory-mapped peripheral that generates VGA
 *
 * Stephen A. Edwards
 * Columbia University
 */

module vga_ball(input logic        clk,
	        input logic 	   reset,
		input logic [15:0]  writedata,
		input logic 	   write,
		input 		   chipselect,
		input logic [4:0]  address,

		input left_chan_ready,
		input right_chan_ready,
		output logic [15:0] sample_data_l,
		output logic sample_valid_l,
		output logic [15:0] sample_data_r,
		output logic sample_valid_r,


		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output logic 	   VGA_CLK, VGA_HS, VGA_VS,
		                   VGA_BLANK_n,
		output logic 	   VGA_SYNC_n);

   logic [10:0]	   hcount;
   logic [9:0]     vcount;
   logic [10:0]	   hcount_data;
   logic [10:0]	   hcount_data2;
   logic [9:0]     vcount_data;
   logic [9:0]     vcount_data2;
   logic [7:0] 	   background_r, background_g, background_b,ball_x1,ball_x2,ball_y1,ball_y2;
   logic [15:0]    wall1,wall2,tank1_p,tank1_d,enemy1_p,enemy1_d,enemy2_p,enemy2_d,enemy3_p,enemy3_d,enemy4_p,enemy4_d;
   logic [15:0]    bullet1_p,bullet2_p,bullet3_p,bullet4_p,bullet5_p,bullet_data,game_data;
   logic [10:0]  x;
   logic [8:0]   y;

	reg [11:0] counter;
	logic flag1;
	logic flag2;
	logic flag3;

   vga_counters counters(.clk50(clk), .*);
   reg [255:0] image;
 
   always_ff @(posedge clk) begin
     if (reset) begin
	background_r <= 8'hff;
	background_g <= 8'hff;
	background_b <= 8'hff;
        tank1_d <= 16'b0111001100100000;
	tank1_p <= 16'hA079;
        wall1 <= 16'h0000;
        wall2 <= 16'h0000;
        enemy1_p <= 16'h5018;
        enemy1_d <= 16'h4201;
        enemy2_p <= 16'h0000;
        enemy2_d <= 16'h0000;
        enemy3_p <= 16'h0000;
        enemy3_d <= 16'h0000;
        enemy4_p <= 16'h0000;
        enemy4_d <= 16'h0000;
        bullet1_p <= 16'h7880;
        bullet2_p <= 16'h0000;
        bullet3_p <= 16'h0000;
        bullet4_p <= 16'h0000;
        bullet5_p <= 16'h0000;
        bullet_data <= 16'h0001;
        game_data <= 16'h0003;
     end else if (chipselect && write)
       case (address)
	 5'h0 : wall1 <= writedata;
	 5'h1 : wall2 <= writedata;
         5'h2 : tank1_p <= writedata;
         5'h3 : tank1_d <= writedata;
         5'h4 : enemy1_p <= writedata;
         5'h5 : enemy1_d <= writedata;
         5'h6 : enemy2_p <= writedata;
         5'h7 : enemy2_d <= writedata;
         5'h8 : enemy3_p <= writedata;
         5'h9 : enemy3_d <= writedata;
         5'ha : enemy4_p <= writedata;
         5'hb : enemy4_d <= writedata;
         5'hc : bullet1_p <= writedata;
         5'hd : bullet2_p <= writedata;
         5'he : bullet3_p <= writedata;
         5'hf : bullet4_p <= writedata;
         5'h10 : bullet5_p <= writedata;
         5'h11: bullet_data <= writedata;
         5'h12: game_data <= writedata;
       endcase
end


	reg	[10:0]  address1;
	wire	[15:0]  q1;

	shoot audio1(.address(address1), .clock(clk), .q(q1)); //1653

	
	reg	[10:0]  address2;
	wire	[15:0]  q2;
	hit  audio2(.address(address2), .clock(clk), .q(q2));//1235

	reg	[12:0]  address3;
	wire	[15:0]  q3;
	bomb audio3(.address(address3), .clock(clk), .q(q3));//5832
	
	always_ff @(posedge clk) begin
		if(reset) begin
			counter <= 0;
			sample_valid_l <= 0; sample_valid_r <= 0;
		end

		else if(left_chan_ready == 1 && right_chan_ready == 1 && counter < 3125) begin
			counter <= counter + 1;
			sample_valid_l <= 0; sample_valid_r <= 0;
		end
		else if(left_chan_ready == 1 && right_chan_ready == 1 && counter == 3125) begin
			counter <= 0;
			sample_valid_l <= 1; sample_valid_r <= 1;

			if (game_data[1:0]==3'b11 || flag3 ==1'b0) begin
				if (address3 < 13'd5832) begin
					address3 <= address3+1;
					flag3 <= 1'b0;
				end
				else begin
					address3 <=0;
					flag3 <= 1'b1;
				end
				sample_data_l <= q3;
				sample_data_r <= q3;
			end


			else if (game_data[1:0]==3'b01 || flag1 ==1'b0) begin
				if (address1 < 11'd1653) begin
					address1 <= address1+1;
					flag1 <= 1'b0;
				end
				else begin
					address1 <=0;
					flag1 <= 1'b1;
				end
				sample_data_l <= q1;
				sample_data_r <= q1;
			end

			else if (game_data[1:0]==3'b10 || flag2 ==1'b0) begin
				if (address2 < 11'd1235) begin
					address2 <= address2+1;
					flag2 <= 1'b0;
				end
				else begin
					address2 <=0;
					flag2 <= 1'b1;
				end
				sample_data_l <= q2;
				sample_data_r <= q2;
			end

			else begin
				sample_data_l <= 0;
				sample_data_r <= 0;
			end	
		end
		
		else begin
		sample_valid_l <= 0; sample_valid_r <= 0;
		end
	end


  //---------------------- RAM map.v --------------------------------------
   logic map_en;
   logic [9:0] map_address;
   logic [4:0] map_data;
   logic [4:0] map_output;
   map map_unit(.address(map_address),.clock(clk),.data(map_data),.wren(map_en),.q(map_output));

   always_ff @(posedge clk) begin
      if(wall1[0]) begin
       map_en <=1;
       map_address <= wall1[15:11] + wall1[10:6]*32;
       map_data <= wall1[5:1];
      end
      else begin
       map_en <=0;
       map_address <= hcount_data2[10:5]-2+vcount_data2[8:4]*32-32;
      end 
   end
   //--------------------- ROM wall.v -------------------------------------
   //------------------ store the image of wall ---------------------------
   logic [7:0] wall_address;
   logic [15:0] wall_output;
   wall wall_unit(.address(wall_address),.clock(clk),.q(wall_output));
   always_ff @(posedge clk) begin
       wall_address <= map_output*8 + vcount_data2[3:1];
     // end
   end

   // -------------------------ROM tank.v --------------------------
   logic [9:0] tank_address;
   logic [31:0] tank_output;
   tank tank_unit(.address(tank_address),.clock(clk),.q(tank_output));
   always_ff @(posedge clk) begin
   if (hcount_data <(tank1_p[15:8]*4+64) && vcount_data <(tank1_p[7:0]*2+32) && hcount_data>=(tank1_p[15:8]*4) && vcount_data >=(tank1_p[7:0]*2))
       tank_address <= vcount_data[8:1]-tank1_p[7:0]+tank1_d[15:12]*8*16+tank1_d[5:4]*32+(tank1_p[8]^tank1_p[0])*16; //display player1 tank
 
   else if (enemy1_d[0] && hcount_data <(enemy1_p[15:8]*4+64) && vcount_data <(enemy1_p[7:0]*2+32) && hcount_data>=(enemy1_p[15:8]*4) && vcount_data >=(enemy1_p[7:0]*2))
       tank_address <= vcount_data[8:1]-enemy1_p[7:0]+enemy1_d[15:12]*8*16+enemy1_d[5:4]*32+(enemy1_p[8]^enemy1_p[0])*16; //display enemy1 tank

   else if (enemy2_d[0] && hcount_data <(enemy2_p[15:8]*4+64) && vcount_data <(enemy2_p[7:0]*2+32) && hcount_data>=(enemy2_p[15:8]*4) && vcount_data >=(enemy2_p[7:0]*2))
       tank_address <= vcount_data[8:1]-enemy2_p[7:0]+enemy2_d[15:12]*8*16+enemy2_d[5:4]*32+(enemy2_p[8]^enemy2_p[0])*16; //display enemy2 tank

   else if (enemy3_d[0] && hcount_data <(enemy3_p[15:8]*4+64) && vcount_data <(enemy3_p[7:0]*2+32) && hcount_data>=(enemy3_p[15:8]*4) && vcount_data >=(enemy3_p[7:0]*2))
       tank_address <= vcount_data[8:1]-enemy3_p[7:0]+enemy3_d[15:12]*8*16+enemy3_d[5:4]*32+(enemy3_p[8]^enemy3_p[0])*16; //display enemy3 tank

   else if (enemy4_d[0] && hcount_data <(enemy4_p[15:8]*4+64) && vcount_data <(enemy4_p[7:0]*2+32) && hcount_data>=(enemy4_p[15:8]*4) && vcount_data >=(enemy4_p[7:0]*2))
       tank_address <= vcount_data[8:1]-enemy4_p[7:0]+enemy4_d[15:12]*8*16+enemy4_d[5:4]*32+(enemy4_p[8]^enemy4_p[0])*16; //display enemy4 tank


end

  //------------------------ ROM colormap.v ----------------------------
  //------------------------ store tank color -----------------
   logic [4:0] colormap_address;
   logic [23:0] colormap_output;
   colormap colormap_unit(.address(colormap_address),.clock(clk),.q(colormap_output));
   always_ff @(posedge clk) begin  
   if (hcount_data <(tank1_p[15:8]*4+64) && vcount_data <(tank1_p[7:0]*2+32) && hcount_data>=(tank1_p[15:8]*4) && vcount_data >=(tank1_p[7:0]*2))
       colormap_address  <= tank_output>>(30-2*(hcount_data[9:2]-tank1_p[15:8]))&32'd3|(tank1_d[9:6]&4'b1100);   // display player1 tank

   else if (enemy1_d[0] && hcount_data <(enemy1_p[15:8]*4+64) && vcount_data <(enemy1_p[7:0]*2+32) && hcount_data>=(enemy1_p[15:8]*4) && vcount_data >=(enemy1_p[7:0]*2))
       colormap_address  <= tank_output>>(30-2*(hcount_data[9:2]-enemy1_p[15:8]))&32'd3|(enemy1_d[9:6]&4'b1100);   // display enemy1 tank

   else if (enemy2_d[0] && hcount_data <(enemy2_p[15:8]*4+64) && vcount_data <(enemy2_p[7:0]*2+32) && hcount_data>=(enemy2_p[15:8]*4) && vcount_data >=(enemy2_p[7:0]*2))
       colormap_address  <= tank_output>>(30-2*(hcount_data[9:2]-enemy2_p[15:8]))&32'd3|(enemy2_d[9:6]&4'b1100);   // display enemy2 tank

   else if (enemy3_d[0] && hcount_data <(enemy3_p[15:8]*4+64) && vcount_data <(enemy3_p[7:0]*2+32) && hcount_data>=(enemy3_p[15:8]*4) && vcount_data >=(enemy3_p[7:0]*2))
       colormap_address  <= tank_output>>(30-2*(hcount_data[9:2]-enemy3_p[15:8]))&32'd3|(enemy3_d[9:6]&4'b1100);   // display enemy3 tank

   else if (enemy4_d[0] && hcount_data <(enemy4_p[15:8]*4+64) && vcount_data <(enemy4_p[7:0]*2+32) && hcount_data>=(enemy4_p[15:8]*4) && vcount_data >=(enemy4_p[7:0]*2))
       colormap_address  <= tank_output>>(30-2*(hcount_data[9:2]-enemy4_p[15:8]))&32'd3|(enemy4_d[9:6]&4'b1100);   // display enemy4 tank


   end

  // ---------------------------ROM wallcolor.v ------------------------

   logic [4:0] wallcolor_address;
   logic [23:0] wallcolor_output;
   wallcolor wallcolor_unit(.address(wallcolor_address),.clock(clk),.q(wallcolor_output));
   always_ff @(posedge clk) begin  
   if (hcount_data>=64 && hcount_data <1024 && vcount_data>=16 && vcount_data< 432) begin
       case (map_output)
       4'b0000: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b00000); // blank
       4'b0001: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b00000); // wall
       4'b0010: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b00000); // wall
       4'b0011: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b00000); // wall
       4'b0100: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b00000); // wall
       4'b0101: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b00000); // wall
       4'b0110: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b00100); // brick
       4'b0111: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b00100); // ice
       4'b1000: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b01000); // grass
       4'b1001: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b01100); // water
       4'b1010: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b01100); // water
       4'b1011: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b01100); // water

       default: wallcolor_address  <= wall_output>>(14-2*(hcount_data[4:2]))&32'd3|(5'b10000); // other
       endcase
   end
end

always_comb begin
    	{VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0};
    	if(hcount==1278) hcount_data = 1; 
    	else if(hcount==1279) hcount_data = 2; 
    	else if(hcount==1277) hcount_data =0;
   	else hcount_data = hcount+3;
     	hcount_data2 = hcount +8; //????????????????????????
    	vcount_data = vcount;
    	vcount_data2 = vcount;

	//----------------------- display on VGA --------------------------------------
	if (VGA_BLANK_n ) begin
	// display bullet   
	// bullet 1
	if ((bullet_data[0]!=0) && hcount>= (bullet1_p[15:8]*4) && hcount<= (bullet1_p[15:8]*4+11) &&  vcount >=(bullet1_p[7:0]*2) &&  vcount <=(bullet1_p[7:0]*2+5)) 
		{VGA_R, VGA_G, VGA_B} = {8'hAD, 8'hAD, 8'hAD};
	// bullet 2
	else if ((bullet_data[1]!=0) && hcount>= (bullet2_p[15:8]*4) && hcount<= (bullet2_p[15:8]*4+11) &&  vcount >=(bullet2_p[7:0]*2) &&  vcount <=(bullet2_p[7:0]*2+5)) 
		{VGA_R, VGA_G, VGA_B} = {8'hAD, 8'hAD, 8'hAD};
	// bullet 3
	else if ((bullet_data[2]!=0) && hcount>= (bullet3_p[15:8]*4) && hcount<= (bullet3_p[15:8]*4+11) &&  vcount >=(bullet3_p[7:0]*2) &&  vcount <=(bullet3_p[7:0]*2+5))
		 {VGA_R, VGA_G, VGA_B} = {8'hAD, 8'hAD, 8'hAD};
	// bullet 4
	else if ((bullet_data[3]!=0) && hcount>= (bullet4_p[15:8]*4) && hcount<= (bullet4_p[15:8]*4+11) &&  vcount >=(bullet4_p[7:0]*2) &&  vcount <=(bullet4_p[7:0]*2+5))
		 {VGA_R, VGA_G, VGA_B} = {8'hAD, 8'hAD, 8'hAD};
	// bullet 5
	else if ((bullet_data[4]!=0) && hcount>= (bullet5_p[15:8]*4) && hcount<= (bullet5_p[15:8]*4+11) &&  vcount >=(bullet5_p[7:0]*2) &&  vcount <=(bullet5_p[7:0]*2+5))
		 {VGA_R, VGA_G, VGA_B} = {8'hAD, 8'hAD, 8'hAD};
	// map
    	else if (hcount>=64 && hcount<1024 && vcount>=16 && vcount<432 && wallcolor_output)
		{VGA_R, VGA_G, VGA_B} = wallcolor_output;
     	// player tank
	else if ((tank1_d[0]!=0) && hcount <(tank1_p[15:8]*4+64) && vcount <(tank1_p[7:0]*2+32) && hcount>=(tank1_p[15:8]*4) && vcount >=(tank1_p[7:0]*2))
		{VGA_R, VGA_G, VGA_B} = colormap_output;
    	// enemy tank1
        else if ((enemy1_d[0]!=0) && hcount <(enemy1_p[15:8]*4+64) && vcount <(enemy1_p[7:0]*2+32) && hcount>=(enemy1_p[15:8]*4) && vcount >=(enemy1_p[7:0]*2)) 
		{VGA_R, VGA_G, VGA_B} = colormap_output;
    	// enemy tank2
        else if ((enemy2_d[0]!=0) && hcount <(enemy2_p[15:8]*4+64) && vcount <(enemy2_p[7:0]*2+32) && hcount>=(enemy2_p[15:8]*4) && vcount >=(enemy2_p[7:0]*2))	
		{VGA_R, VGA_G, VGA_B} = colormap_output;
     	// enemy tank3
        else if ((enemy3_d[0]!=0) && hcount <(enemy3_p[15:8]*4+64) && vcount <(enemy3_p[7:0]*2+32) && hcount>=(enemy3_p[15:8]*4) && vcount >=(enemy3_p[7:0]*2)) 
		{VGA_R, VGA_G, VGA_B} = colormap_output;
     	// enemy tank4
        else if ((enemy4_d[0]!=0) && hcount <(enemy4_p[15:8]*4+64) && vcount <(enemy4_p[7:0]*2+32) && hcount>=(enemy4_p[15:8]*4) && vcount >=(enemy4_p[7:0]*2))	
		{VGA_R, VGA_G, VGA_B} = colormap_output;
   	// background
	else if (hcount>=64 && hcount<896 && vcount>=16 && vcount<432)
	 	{VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0}; 
        else 
		{VGA_R, VGA_G, VGA_B} = {8'h63, 8'h63, 8'h63};
     
       end 
end
	       
endmodule

module vga_counters(
 input logic 	     clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;
       
   logic endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule
