use std::convert::TryInto;
use std::fs::File;
use std::io::{BufRead, BufReader, SeekFrom};
use std::io::prelude::*;
use std::io;
use std::os::unix::io::AsRawFd;

use clap::{App, Arg};
use hex;

const LOGO: &str = "
██╗   ██╗██╗  ████████╗██████╗  █████╗ ███╗   ██╗███████╗███████╗
██║   ██║██║  ╚══██╔══╝██╔══██╗██╔══██╗████╗  ██║██╔════╝██╔════╝
██║   ██║██║     ██║   ██████╔╝███████║██╔██╗ ██║█████╗  ███████╗
██║   ██║██║     ██║   ██╔══██╗██╔══██║██║╚██╗██║██╔══╝  ╚════██║
╚██████╔╝███████╗██║   ██║  ██║██║  ██║██║ ╚████║███████╗███████║
 ╚═════╝ ╚══════╝╚═╝   ╚═╝  ╚═╝╚═╝  ╚═╝╚═╝  ╚═══╝╚══════╝╚══════╝
";

extern { 
    fn get_addr(fd: i32) -> i32; 
    fn send(fd: i32, v: i32, a: i32) -> i32; 
}

#[derive(Clone, Copy)]
enum MemDest {
    CPU,
    PPU,
}

struct Device {
    #[allow(dead_code)]
    f: File,
    fd: i32,
}

impl Device {
    fn new(dev_name: &str) -> io::Result<Self> {
        let f = File::open(dev_name)?;
        let dev = f.as_raw_fd();
        Ok(Device { f, fd: dev })
    }

    fn read_addr(&self) -> i32 {
        unsafe { 
            get_addr(self.fd) 
        }
    }

    fn send_reset(&self, v: i32) {
        if v != 0 && v != 1 {
            println!("RESET {} FAIL", v);
        }
        unsafe {
            if send(self.fd, v, 3 << 15) != 0 {
                println!("RESET {} FAIL", v);
            }
        }
    }
    
    fn send_byte(&self, a: i32, b: u8, dest: MemDest) {
        print!("({:x}):{:x}\r", a, b);
        match dest {
            MemDest::CPU => unsafe { send(self.fd, b as i32, a); },
            MemDest::PPU => unsafe { send(self.fd, b as i32, (1 << 16) | a); },
        }
    }
    
    fn send_bytes(&self, f: File, dest: MemDest) {
        let reader = BufReader::new(f);
        let mut a = 0;
    
        // assert CPU reset
        self.send_reset(1);
        for l in reader.lines() {
            for i in l.unwrap().split_whitespace() {
                let b = hex::decode(i).unwrap()[0];
                print!("({:x}):{:x}\r", a, b);
                self.send_byte(a, b, dest);
                a += 1;
            }
        }
        println!();
        // de-assert CPU reset
        self.send_reset(0);
    }

    /// loads into CPU $8000 - $bfff
    /// (16k)
    fn load_pgr_low(&self, rom: &str, seek: u64) -> io::Result<()> {
        let mut f = File::open(rom)?;
        f.seek(SeekFrom::Start(seek))?;

        // 16k buffer
        let mut buf = [0; 16384];
        f.read(&mut buf)?;

        let mut a = 0x8000;
        for b in buf.iter() {
            self.send_byte(a, *b, MemDest::CPU);
            a += 1;
        }
        println!();

        Ok(())
    }

    /// loads into CPU $c000 - $ffff
    /// (16k)
    fn load_pgr_high(&self, rom: &str, seek: u64) -> io::Result<()> {
        let mut f = File::open(rom)?;
        f.seek(SeekFrom::Start(seek))?;

        // 16k buffer
        let mut buf = [0; 16384];
        f.read(&mut buf)?;

        let mut a = 0xc000;
        for b in buf.iter() {
            self.send_byte(a, *b, MemDest::CPU);
            a += 1;
        }
        println!();

        Ok(())
    }

    /// loads into PPU $0000 - $1fff
    /// (8k)
    fn load_chr(&self, rom: &str, seek: u64) -> io::Result<()> {
        let mut f = File::open(rom)?;
        f.seek(SeekFrom::Start(seek))?;

        // 8k buffer
        let mut buf = [0; 8192];
        f.read(&mut buf)?;

        let mut a = 0;
        for b in buf.iter() {
            self.send_byte(a, *b, MemDest::PPU);
            a += 1;
        }
        println!();

        Ok(())
    }
    
    fn load_mario(&self, mario_path: &str) -> io::Result<()> {
        // mario has two sets of PGR (16k each)
        // and one set of CHR (8k)
        // and 16 byte header (ignored)
        //
        // TODO need to handle flags:
        // 1: vertical (horizontal arrangement) (CIRAM A10 = PPU A10)
       
        self.send_reset(1);
        self.load_pgr_low(mario_path, 16)?;
        self.load_pgr_high(mario_path, 16 + 16384)?;

        self.load_chr(mario_path, 16 + (16384 * 2))?;
        self.send_reset(0);

        Ok(())
    }
    
    fn load_dk(&self, dk_path: &str) -> io::Result<()> {
        // donkey kong has one set of PGR (16k)
        // and one set of CHR (8k)
        // and 16 byte header (ignored)
    
        self.send_reset(1);
        self.load_pgr_low(dk_path, 16)?;
        // duplicate since DK only has one
        self.load_pgr_high(dk_path, 16)?;

        self.load_chr(dk_path, 16+16384)?;
        self.send_reset(0);

        Ok(())
    }
}

fn main() -> io::Result<()> {
    let dev = Device::new("/dev/ultranes");
    let dev = match dev {
        Ok(d) => d,
        Err(e) => {
            println!("Unable to create ultranes device: {}", e);
            println!("Did you insert the device driver module?");
            return Err(e);
        }
    };

    let matches = App::new("ultranes")
        .version("0.1.0")
        .author("Zach Schuermann")
        .about(LOGO)
        .subcommand(
            App::new("reset")
                .about("set reset high/low")
                .arg(
                    Arg::with_name("signal")
                        .help("signal (0/1)")
                        .index(1)
                        .required(true)
                )
        )
        .subcommand(
            App::new("write")
                .about("write value to address")
                .arg(
                    Arg::with_name("value")
                        .help("value")
                        .index(1)
                        .required(true)
                )
                .arg(
                    Arg::with_name("addr")
                        .help("address")
                        .index(2)
                        .required(true)
                )
        )
        .subcommand(
            App::new("load")
                .about("load ROM")
                .arg(
                    Arg::with_name("ROM")
                        .help("Name of ROM to load (currently supports 'dk' or 'mario'")
                        .index(1)
                )
                .arg(
                    Arg::with_name("ROM path")
                        .help("Path of (.nes) ROM to load")
                        .index(2)
                )
                .subcommand(
                    App::new("cpu")
                        .about("load to CPU RAM directly")
                        .arg(
                            Arg::with_name("file")
                            .help("Name of file to load")
                            .index(1)
                            .required(true)
                        )
                )
        )
        .get_matches();

    if let Some(ref matches) = matches.subcommand_matches("reset") {
        let reset: u8 = matches.value_of_t("signal").unwrap();
        if reset == 1 {
            dev.send_reset(1);
        } else if reset == 0 {
            dev.send_reset(0);
        }
    } else if let Some(ref matches) = matches.subcommand_matches("write") {
        let val: i32 = matches.value_of_t("value").unwrap();
        let addr: i32 = matches.value_of_t("addr").unwrap();
        println!("write: {}:{}", val, addr);
        match matches.value_of("dest") {
            Some("cpu") => dev.send_byte(val, addr.try_into().unwrap(), MemDest::CPU),
            Some("ppu") => dev.send_byte(val, addr.try_into().unwrap(), MemDest::CPU),
            _ => println!("ERROR: invalid destination")
        }
    } else if let Some(ref matches) = matches.subcommand_matches("load") {
        if let Some(ref matches) = matches.subcommand_matches("cpu") {
            let file = matches.value_of("file").unwrap();
            let file = File::open(file);
            match file {
                Ok(f) => dev.send_bytes(f, MemDest::CPU),
                Err(e) => println!("Could not load file: {}", e),
            }
        } else if let Some(ref rom_path) = matches.value_of("ROM path") {
            match matches.value_of("ROM") {
                Some("dk") => dev.load_dk(rom_path)?,
                Some("mario") => dev.load_mario(rom_path)?,
                _ => println!("ERROR: could not parse ROM")
            }
        }
    } else {
        // default behavior: read address
        println!("${:x}", dev.read_addr());
    }

    Ok(())
}
