/* * Device driver for the VGA video generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_ball.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree vga_ball.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "vga_ball.h"

#define DRIVER_NAME "vga_ball"

/* Device registers */
#define def_hcount(base)        (base)
#define def_vcount(base)        (base+1)
#define def_carx(base)          (base+2)
#define def_score_t(base)       (base+3)
#define def_score_u(base)       (base+4)
#define def_spaceship(base)     (base+5)
#define def_bullet_gnd(base, i) (base+6+(i))

/*
 * Information about our device
 */
struct vga_ball_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
	
	vga_ball_bullet_gnd   bullet_gnd;
	vga_ball_hcount       hcount;
	vga_ball_vcount       vcount;
	vga_ball_carx         carx;
	vga_ball_score_t      score_t;
	vga_ball_score_u      score_u;
    vga_ball_spaceship    spaceship;
} dev;


static void write_bullet_gnd(vga_ball_bullet_gnd *bullet_gnd){
	unsigned int i = 0;
	for(i =0; i < GND_SZ+1; i++){
		iowrite8(bullet_gnd->bullet_gnd[i], def_bullet_gnd(dev.virtbase, i));
	}
	dev.bullet_gnd = *bullet_gnd;
}

static void write_hc(vga_ball_hcount* hcount){
	iowrite8(hcount->hcount, def_hcount(dev.virtbase));
	dev.hcount = *hcount;
}

static void write_vc(vga_ball_vcount* vcount){
	iowrite8(vcount->vcount, def_vcount(dev.virtbase));
	dev.vcount = *vcount;
}

static void write_carx(vga_ball_carx* carx){
	iowrite8(carx->carx, def_carx(dev.virtbase));
	dev.carx = *carx;
}

static void write_score_t(vga_ball_score_t* score_t){
	iowrite8(score_t->score_t, def_score_t(dev.virtbase));
	dev.score_t = *score_t;
}

static void write_score_u(vga_ball_score_u* score_u){
	iowrite8(score_u->score_u, def_score_u(dev.virtbase));
	dev.score_u = *score_u;
}

static void write_spaceship(vga_ball_spaceship* spaceship){
	iowrite8(spaceship->spaceship, def_spaceship(dev.virtbase));
	dev.spaceship = *spaceship;
}

static long vga_ball_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	vga_ball_bullet_gnd  bullet_gnd;
	vga_ball_hcount      hcount;
	vga_ball_vcount      vcount;
	vga_ball_carx        carx;
	vga_ball_score_t     score_t;
	vga_ball_score_u     score_u;
    vga_ball_spaceship   spaceship;

	switch (cmd) {
	case VGA_BALL_WRITE_HCOUNT:
		if (copy_from_user(&hcount, (vga_ball_hcount *) arg,
				   sizeof(vga_ball_hcount)))
			return -EACCES;
		write_hc(&hcount);
		break;
		
	case VGA_BALL_WRITE_VCOUNT:
		if (copy_from_user(&vcount, (vga_ball_vcount *) arg,
				   sizeof(vga_ball_vcount)))
			return -EACCES;
		write_vc(&vcount);
		break;
	
	case VGA_BALL_WRITE_CARX:
		if (copy_from_user(&carx, (vga_ball_carx *) arg,
				   sizeof(vga_ball_carx)))
			return -EACCES;
		write_carx(&carx);
		break;	

	case VGA_BALL_WRITE_SCORE_T:
		if (copy_from_user(&score_t, (vga_ball_score_t *) arg,
				   sizeof(vga_ball_score_t)))
			return -EACCES;
		write_score_t(&score_t);
		break;	
	
	case VGA_BALL_WRITE_SCORE_U:
		if (copy_from_user(&score_u, (vga_ball_score_u*) arg,
				   sizeof(vga_ball_score_u)))
			return -EACCES;
		write_score_u(&score_u);
		break;	
	
	case VGA_BALL_WRITE_SPACESHIP:
		if (copy_from_user(&spaceship, (vga_ball_spaceship*) arg,
				   sizeof(vga_ball_spaceship)))
			return -EACCES;
		write_spaceship(&spaceship);
		break;	    
    
	case VGA_BALL_WRITE_BULLET_GND:
		if (copy_from_user(&bullet_gnd, (vga_ball_bullet_gnd *) arg,
				   sizeof(vga_ball_bullet_gnd)))
			return -EACCES;
		write_bullet_gnd(&bullet_gnd);
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations vga_ball_fops = {
	.owner		    = THIS_MODULE,
	.unlocked_ioctl = vga_ball_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice vga_ball_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &vga_ball_fops,
};

static int __init vga_ball_probe(struct platform_device *pdev)
{
    //vga_ball_pos_t start = { 1, 1};
	int ret;

	/* Register ourselves as a misc device: creates /dev/vga_ball */
	ret = misc_register(&vga_ball_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	/* Set an initial color */
     //   write_pos(&start);

	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&vga_ball_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int vga_ball_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&vga_ball_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id vga_ball_of_match[] = {
	{ .compatible = "csee4840,vga_ball-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, vga_ball_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver vga_ball_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(vga_ball_of_match),
	},
	.remove	= __exit_p(vga_ball_remove),
};

/* Called when the module is loaded: set things up */
static int __init vga_ball_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&vga_ball_driver, vga_ball_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit vga_ball_exit(void)
{
	platform_driver_unregister(&vga_ball_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(vga_ball_init);
module_exit(vga_ball_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("VGA ball driver");