import math
import numpy as np
from PIL import Image
import os
from scipy import stats
import click
import sys

MSG_RESIZE = 'requested size in pixels as tuple (int, int)'
MSG_MINLUMA = 'low brightness threshold (int) 1:white (highest) 0:black (lowest)'
MSG_MAXLUMA = 'high brightness threshold (int) 1:white (highest) 0:black (lowest)'

@click.command()
@click.option('--resize', default=[None]*2, type=click.Tuple([int, int]), help=MSG_RESIZE)
@click.option('--minluma', type=float, help=MSG_MINLUMA)
@click.option('--maxluma', type=float, help=MSG_MAXLUMA)
@click.option('--inverted_color', is_flag=True, type=bool)
@click.option('-s', '--save', is_flag=True, type=bool)
@click.argument('path', type=click.Path(exists=True))
def run(path, resize, inverted_color, minluma, maxluma, save):
	ImageProcess.set_luma_thres(minluma, maxluma, inverted_color)
	im = ImageProcess.binarize(ImageProcess.load(path, resize))
	with open('%s.sv' % (os.path.splitext(path)[0]+'_'+str(im.shape[1])+'x'+str(im.shape[0])), "w") as fd:
		fd.write('/*\n\tGenerated using img2bin_bw.py \n\n')
		
		fd.write("\tImage Stats...\n")
		fd.write("\tHeight: %d px\n" % im.shape[0])
		fd.write("\tWidth: %d px\n"% im.shape[1])
		fd.write("\tTotal size = %d px\n" % (im.shape[0]*im.shape[1]))
		id, count = np.unique(im, return_counts=True)
		for i in range(id.shape[0]):
			fd.write("\tid=%d\t color=%s \t freq=%d\n" % (id[i], ImageProcess.MAP_ID_CLR[id[i]], count[i]))
		fd.write("*/\n\n")
		
		reg_nm = os.path.splitext(os.path.basename(path))[0]
		for r in range(len(im)):
			fd.write('%s[%d][0:%d] <=' % (reg_nm, r, im.shape[1]-1))
			str_list = ''.join([str(e) for e in im[r][:]])
			fd.write('\t%d\'b%s;\n' % (im.shape[1], str_list))
	arr, img = ImageProcess.revert(im)
	if save:
		img.save(os.path.splitext(path)[0]+'_'+str(arr.shape[1])+
			'x'+str(arr.shape[0])+'_bw.png')
	img.show()

class ImageProcess:
	__DEFAULT_LUMA_THRESHOLD_MAX = 0.7
	__DEFAULT_LUMA_THRESHOLD_MIN = 0
	__INVERTED_LUMA_THRESHOLD_MAX = 1
	__INVERTED_LUMA_THRESHOLD_MIN = 0.7
	__LUMA_THRESHOLD_MAX = __DEFAULT_LUMA_THRESHOLD_MAX
	__LUMA_THRESHOLD_MIN = __DEFAULT_LUMA_THRESHOLD_MIN
	DEFAULT_TEMP_DIR  = "../templates/"
	IMG_EXTS = ['.png', '.jpg', '.gif']
	MAP_ID_CLR = {0:'White', 1:'Black'}
	__ERR_LUMAVAL = 'minluma and maxluma must be within [0,1]\n default values will be used'
	
	def binarize(img):
		new_img = np.zeros([img.shape[0], img.shape[1]])
		img = ImageProcess.rgb2luma(img)
		for r in range(img.shape[0]):
			for c in range(img.shape[1]):
				if ImageProcess.__LUMA_THRESHOLD_MAX > img[r,c] and \
					img[r,c] > ImageProcess.__LUMA_THRESHOLD_MIN:
					new_img[r,c] = 1
		return new_img.astype(int)
			
	def rgb2luma(im):
		R_FACTOR = 0.2126
		G_FACTOR = 0.7152
		B_FACTOR = 0.0722
		bw_im = np.zeros([im.shape[0], im.shape[1]])
		for r in range(im.shape[0]):
			for c in range(im.shape[1]):
				bw_im[r,c] = (R_FACTOR*im[r,c,0]+G_FACTOR*im[r,c,1]+B_FACTOR*im[r,c,2])/255
		return bw_im	
	
	def set_luma_thres(min=None, max=None, inverted=False):
		if min == None:
			min = ImageProcess.__DEFAULT_LUMA_THRESHOLD_MIN
		if max == None:
			max = ImageProcess.__DEFAULT_LUMA_THRESHOLD_MAX
		if min >= 0 and max <= 1:
			if inverted:
				ImageProcess.__LUMA_THRESHOLD_MIN = ImageProcess.__INVERTED_LUMA_THRESHOLD_MIN
				ImageProcess.__LUMA_THRESHOLD_MAX = ImageProcess.__INVERTED_LUMA_THRESHOLD_MAX
			else:
				ImageProcess.__LUMA_THRESHOLD_MIN = min
				ImageProcess.__LUMA_THRESHOLD_MAX = max
		else:
			print(ImageProcess.__ERR_LUMAVAL, file=sys.stderr)
		
	def get_imgs(dir=DEFAULT_TEMP_DIR, ext=IMG_EXTS):
		files = []
		for f in os.listdir(dir):
			for e in ext:
				if f.endswith(e):
					files.append(os.path.join(dir, f))
		return files
	
	def revert(im):
		im = 255*(1-im)
		return im.astype(np.uint8), Image.fromarray(im.astype(np.uint8))
	
	def load(path, size):
		img = Image.open(path)
		if size[0] and size[1]:
			img = img.resize(size)
		return np.array(img)

if __name__=="__main__":
	run()