import numpy as np
from PIL import Image, ImageColor
import os
import click
from scipy import stats

MSG_RESIZE = 'requested size in pixels as tuple (int, int)'

@click.command()
@click.option('--resize', default=[None]*2, type=click.Tuple([int, int]), help=MSG_RESIZE)
@click.argument('path', type=click.Path(exists=True))
def run(path, resize):
	im = ImageProcess.binarize(ImageProcess.load(path, resize))
	im = ImageProcess.compress(im)
	with open('%s.sv' % os.path.splitext(path)[0], "w") as fd:
		fd.write('/*\n\tGenerated using img2bin.py \n\n')
		
		fd.write("\tImage Stats...\n")
		fd.write("\tHeight: %d px\n" % im.shape[0])
		fd.write("\tWidth: %d px\n"% im.shape[1])
		fd.write("\tTotal size = %d px\n" % (im.shape[0]*im.shape[1]))
		id, count = np.unique(im, return_counts=True)
		for i in range(id.shape[0]):
			fd.write("\tid=%d\t color=%s \t freq=%d\n" % (id[i], ImageProcess.MAP_ID_CLR[id[i]], count[i]))
		fd.write("*/\n\n")
		
		fd.write('%s [%d][%d] =\n' % (os.path.splitext(os.path.basename(path))[0], im.shape[0], im.shape[1]))
		for r in range(len(im)):
			str_list = ''.join([str(e) for e in im[r][:]])
			fd.write('\t%d\'d%s;\n' % (im.shape[1], str_list))
	arr, img = ImageProcess.revert(im)
	img.show()

class ImageProcess:
	__PINK = ["#FFC0CB", "#FFB6C1", "#FF69B4", "#FF1493", "#DB7093", "#C71585"]
	__MAGENTA = ["#E6E6FA", "#D8BFD8", "#DDA0DD", "#DA70D6", "#EE82EE", "#FF00FF", 
				"#BA55D3", "#9932CC", "#9400D3", "#8A2BE2", "#8B008B", "#800080", "#9370DB", "#7B68EE",
				"#6A5ACD", "#483D8B", "#663399", "#4B0082"] + __PINK
	__RED = ["#FA8072", "#E9967A", "#F08080", "#CD5C5C", "#DC143C", "#FF0000", "#B22222",
			"#8B0000", "#BE0000", "#EE1515", "#D20000", "#EE1717"]
	__ORANGE = ["#FFA500", "#FF8C00", "#FF7F50", "#FF6347", "#FF4500"]
	__YELLOW = ["#FFD700", "#FFFF00", "#FFFFE0", "#FFFACD", "#FAFAD2", "#FFEFD5", "#FFE4B5", 
		"#FFDAB9", "#EEE8AA", "#F0E68C", "#BDB76B", "#B3D400", "#AECF00"]
	__GREY = 	["#DCDCDC", "#D3D3D3", "#C0C0C0", "#A9A9A9", "#708090", "#778899"]
	__GREEN = ["#ADFF2F", "#7FFF00", "#00FF00", "#32CD32", "#98FB98", "#90EE90", "#00FA9A", 
				"#00FF7F", "#3CB371", "#2E8B57", "#228B22", "#008000", "#006400", "#9ACD32", "#6B8E23", 
				"#556B2F", "#66CDAA", "#8FBC8F", "#20B2AA", "#008B8B", "#008080"] + __GREY
	__CYAN = ["#40B1F8", "#00FFFF", "#E0FFFF", "#7FFFD4", "#40E0D0", "#48D1CC", "#00CED1"]
	__BLUE = ["#4682B4", "#5F9EA0",  "#B0C4DE", "#ADD8E6", "#B0E0E6", "#87CEFA", "#87CEEB", 
				"#6495ED", "#00BFFF", "#1E90FF", "#4169E1", "#0000FF", "#0000CD", "#00008B", "#000080", "#191970"]
	__BROWN = ["#DEB887", "#D2B48C", "#BC8F8F", "#DAA520", "#B8860B", "#CD853F", "#D2691E", "#8B4513", "#A0522D", 
				"#800000", "#A52A2A", "#808000"]
	__WHITE = ["#FFFFFF", "#FFFAFA", "#F0FFF0", "#F5FFFA", "#F0FFFF", "#F0F8FF", "#F8F8FF", 
				"#F5F5F5", "#FFF5EE", "#F5F5DC", "#FDF5E6", "#FFFAF0", "#FFFFF0", "#0FAEBD", "#FAF0E6", 
				"#FFF0F5", "#FFE4E1"]
	__BLACK = 	["#020202", "#050505", "#080808", "#0a0a0a", "#141414", "#1f1f1f", "#292929", "#303030", "#202020", "#383838"]
	__COLORS = __PINK + __MAGENTA + __RED + __ORANGE + __YELLOW + __GREEN + __CYAN + __BLUE + \
		__BROWN + __WHITE + __BLACK 
	MAP_ID_CLR = {0:'White', 1:'Black', 2:'Red', 3:'Green', 4:'Blue', 
		5:'Yellow', 6:'Magenta', 8:'Cyan', 7:'Brown', 9:'Orange'}
	MAP_CLR_ID = {'White':0, 'Black':1, 'Red':2, 'Green':3, 'Blue':4, 
		'Yellow':5, 'Magenta':6, 'Cyan':8, 'Brown':7, 'Orange':9}

	def load(path, size=None):
		img = Image.open(path)
		if size[0] and size[1]:
			img = img.resize(size)
		return np.array(img)
		
	def map2group(clr):
		group = None
		if clr in ImageProcess.__MAGENTA:
			group = 'Magenta'
		elif clr in ImageProcess.__RED:
			group = 'Red'
		elif clr in ImageProcess.__ORANGE:
			group = 'Orange'
		elif clr in ImageProcess.__YELLOW:
			group = 'Yellow'
		elif clr in ImageProcess.__GREEN:
			group = 'Green'
		elif clr in ImageProcess.__BLUE:
			group = 'Blue'
		elif clr in ImageProcess.__CYAN:
			group = 'Cyan'
		elif clr in ImageProcess.__BROWN:
			group = 'Brown'
		elif clr in ImageProcess.__WHITE:
			group = 'White'
		elif clr in ImageProcess.__BLACK:
			group = 'Black'
		return group
	
	def rgb_dist(LHS, RHS):
		return ((LHS[0]-RHS[0])**2) + ((LHS[1]-RHS[1])**2) + ((LHS[2]-RHS[2])**2)
		
	def closest(val, list):
		dists = []
		for e in list:
			dists.append(ImageProcess.rgb_dist(val, ImageColor.getrgb(e)))
		return list[dists.index(min(dists))]
		
	def remap(bin_im):
		cpy_im = bin_im
		id_new = 0
		while len(cpy_im):
			id_old = int(stats.mode(cpy_im, axis=None)[0])
			if id_new < id_old:
				clr_new = ImageProcess.MAP_ID_CLR[id_new]
				ImageProcess.MAP_ID_CLR[id_new] = ImageProcess.MAP_ID_CLR[id_old]
				ImageProcess.MAP_ID_CLR[id_old] = clr_new
				ImageProcess.MAP_CLR_ID[ImageProcess.MAP_ID_CLR[id_old]] = id_old
				ImageProcess.MAP_CLR_ID[ImageProcess.MAP_ID_CLR[id_new]] = id_new
				bin_im[bin_im == id_old] = id_new
				id_new += 1
			cpy_im = cpy_im[cpy_im != id_old]
		return bin_im
	
	def binarize(im):
		bin_im = np.zeros([im.shape[0], im.shape[1]])
		for r in range(im.shape[0]):
			for c in range(im.shape[1]):
				trgt = ImageProcess.closest(im[r,c,:], ImageProcess.__COLORS)
				bin_im[r,c] = ImageProcess.MAP_CLR_ID[ImageProcess.map2group(trgt)]
		return bin_im.astype(int)
		
	def revert(bin_im):
		im = np.zeros([bin_im.shape[0], bin_im.shape[1], 3])
		for r in range(bin_im.shape[0]):
			for c in range(bin_im.shape[1]):
				id = bin_im[r,c]
				rgb_tuple = ImageColor.getrgb(ImageProcess.MAP_ID_CLR[id])
				im[r,c,:] = rgb_tuple
		return im.astype(np.uint8), Image.fromarray(im.astype(np.uint8))

	def compress(bin_im):
		num_px = bin_im.shape[0]*bin_im.shape[1]
		id, count = np.unique(bin_im, return_counts=True)
		for i in range(count.shape[0]):
			if count[i] < (0.005*num_px):
				bin_im[bin_im == id[i]] = ImageProcess.MAP_CLR_ID['Black']
		return ImageProcess.remap(bin_im)
		
if __name__=="__main__":
	run()