/*
 * Userspace program that communicates with the vga_ball device driver
 * through ioctls
 *
 * Stephen A. Edwards
 * Columbia University
 */

#include <stdio.h>
#include "vga_ball.h"
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <time.h>

#define TICK_GND       0.5
#define TICK_VCOUNT    2
#define TICK_HCOUNT    2
#define TICK_CARX      1
#define MIN_Y          5
#define MAX_Y          4
#define MAX_X          19
#define MAX_CARX       2
#define OBSTACLE       8
#define TICK_SCORE     2
#define TICK_BULLET    1
#define BULLET_MAX     8
#define TICK_SPACESHIP 1
#define INVISIBLE      -1
#define HIT_SPACESHIP  10
#define HIT_BULLET     2
#define TICK_CHECK     0.5

int vga_ball_fd;

void set_bullet_gnd(const vga_ball_bullet_gnd *c)
{
  vga_ball_bullet_gnd bullet_gnd;
  bullet_gnd = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_BULLET_GND, &bullet_gnd)) {
      perror("ioctl(VGA_BALL_WRITE_BULLET_GND) failed");
      return;
  }
}

void set_hc(const vga_ball_hcount *c)
{
  vga_ball_hcount hcount;
  hcount = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_HCOUNT, &hcount)) {
      perror("ioctl(VGA_BALL_WRITE_HCOUNT) failed");
      return;
  }
}

void set_carx(const vga_ball_carx *c)
{
  vga_ball_carx carx;
  carx = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_CARX, &carx)) {
      perror("ioctl(VGA_BALL_WRITE_CARX) failed");
      return;
  }
}

void set_score_t(const vga_ball_score_t *c)
{
  vga_ball_score_t score_t;
  score_t = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_SCORE_T, &score_t)) {
      perror("ioctl(VGA_BALL_WRITE_SCORE_T) failed");
      return;
  }
}

void set_score_u(const vga_ball_score_u *c)
{
  vga_ball_score_u score_u;
  score_u = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_SCORE_U, &score_u)) {
      perror("ioctl(VGA_BALL_WRITE_SCORE_U) failed");
      return;
  }
}

void set_vc(const vga_ball_vcount *c)
{
  vga_ball_vcount vcount;
  vcount = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_VCOUNT, &vcount)) {
      perror("ioctl(VGA_BALL_SET_POS) failed");
      return;
  }
}

void set_spaceship(const vga_ball_spaceship *c)
{
  vga_ball_spaceship spaceship;
  spaceship = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_SPACESHIP, &spaceship)) {
      perror("ioctl(VGA_BALL_SET_SPACESHIP) failed");
      return;
  }
}

void init_arr(short lhs[], short rhs[], unsigned int sz){
	unsigned int i = 0;
	for(i=0; i<sz; i++){
		lhs[i] = rhs[i];
	}
}

void update_score(vga_ball_score_t* t, vga_ball_score_u* u){
    if( u->score_u < 9 ){
        u->score_u++;
        set_score_u(u);
    }
    else if(t->score_t < 9){
        t->score_t++;
        set_score_t(t);
        u->score_u = 0;
        set_score_u(u);
    }
}

void update_ui(){
	short bullet_gnd[GND_SZ+1] = {-1,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18};
	vga_ball_carx new_carx;
	new_carx.carx = 0;
    set_carx(&new_carx);
    
	vga_ball_score_t new_score_t;
	new_score_t.score_t = 0;
    set_score_t(&new_score_t);
    
	vga_ball_score_u new_score_u;
	new_score_u.score_u = 0;
    set_score_u(&new_score_u);
    
    vga_ball_spaceship new_spaceship;
    new_spaceship.spaceship = INVISIBLE;
    set_spaceship(&new_spaceship);
    
	vga_ball_bullet_gnd new_bullet_gnd;
    init_arr(new_bullet_gnd.bullet_gnd, bullet_gnd, GND_SZ+1);
    set_bullet_gnd(&new_bullet_gnd);
    
	vga_ball_hcount new_hcount;
	new_hcount.hcount = MAX_X;
    set_hc(&new_hcount);
    
	vga_ball_vcount new_vcount;
	new_vcount.vcount = MIN_Y;
    set_vc(&new_vcount);
	
	clock_t last_update_gnd       = clock();
	clock_t last_update_hc        = clock();
	clock_t last_update_vc        = clock();
	clock_t last_update_carx      = clock();
	clock_t last_update_score     = clock();
    clock_t last_update_bullet    = clock();
    clock_t last_update_spaceship = clock();
    clock_t last_update_check     = clock();
	
	for(;;){
		if(((double)(clock()-last_update_score)/ CLOCKS_PER_SEC > TICK_SCORE) && 
			new_hcount.hcount == OBSTACLE && 
            new_carx.carx == MAX_CARX){
			if(new_vcount.vcount == MAX_Y){
				update_score(&new_score_t, &new_score_u);
				if(new_score_t.score_t == 2 && new_score_u.score_u == 5){
					fprintf(stderr, "You Won!");
					break;
				}
				last_update_score = clock();
			}
        }
        if( new_vcount.vcount == MIN_Y && new_hcount.hcount == OBSTACLE && 
            new_carx.carx == MAX_CARX && ((double)(clock()-last_update_check)/ CLOCKS_PER_SEC > TICK_CHECK)){
              fprintf(stderr, "hcount=%d, vcount=%d\n", new_hcount.hcount, new_vcount.vcount);
			  fprintf(stderr, "GameOver!");
			  break;
        }
        if( new_spaceship.spaceship == HIT_SPACESHIP && 
            new_bullet_gnd.bullet_gnd[0] == HIT_BULLET ){
            update_score(&new_score_t, &new_score_u);
            new_spaceship.spaceship = INVISIBLE;
            new_bullet_gnd.bullet_gnd[0] = INVISIBLE;
            set_spaceship(&new_spaceship);
            set_bullet_gnd(&new_bullet_gnd);
        }
		if ((double)(clock()-last_update_gnd)/ CLOCKS_PER_SEC > TICK_GND){
			last_update_gnd = clock();
			unsigned int i=0;
			for(i=1; i<GND_SZ+1; i++){
				if(! (--new_bullet_gnd.bullet_gnd[i] >= 0)){
					new_bullet_gnd.bullet_gnd[i] = GND_SZ;
				}
			}
			set_bullet_gnd(&new_bullet_gnd);
		}
		if ((double)(clock()-last_update_vc)/ CLOCKS_PER_SEC > TICK_VCOUNT){
			last_update_vc = clock();
			if(new_vcount.vcount == MIN_Y){
				new_vcount.vcount = MAX_Y;
			}
			else{
				new_vcount.vcount = MIN_Y;
			}
			set_vc(&new_vcount);
		}
        if ((double)(clock()-last_update_bullet)/ CLOCKS_PER_SEC > TICK_BULLET){
			last_update_bullet = clock();
			if(new_bullet_gnd.bullet_gnd[0] == INVISIBLE){
				new_bullet_gnd.bullet_gnd[0] = BULLET_MAX;
			}
            else{
                new_bullet_gnd.bullet_gnd[0]--;
            }
			set_bullet_gnd(&new_bullet_gnd);
		}
        if ((double)(clock()-last_update_spaceship)/ CLOCKS_PER_SEC > TICK_SPACESHIP){
			last_update_spaceship = clock();
			if(new_spaceship.spaceship == MAX_X){
				new_spaceship.spaceship = INVISIBLE;
			}
            else{
                new_spaceship.spaceship++;
            }
			set_spaceship(&new_spaceship);
		}
		if ((double)(clock()-last_update_hc)/ CLOCKS_PER_SEC > TICK_HCOUNT){
			last_update_hc = clock();
			new_hcount.hcount--;
			if(new_hcount.hcount < 0){
				new_hcount.hcount = MAX_X;
			}
			set_hc(&new_hcount);
		}
		if (new_carx.carx < 2 &&
			(double)(clock()-last_update_carx)/ CLOCKS_PER_SEC > TICK_CARX){
			last_update_carx = clock();
			new_carx.carx++;
			set_carx(&new_carx);
		}
	}
}

int main()
{
  //vga_ball_arg_t vla;
  int i;
  static const char filename[] = "/dev/vga_ball";

  printf("VGA ball Userspace program started\n");

  if ( (vga_ball_fd = open(filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename);
    return -1;
  }
  
  update_ui();
  
  printf("VGA BALL Userspace program terminating\n");
  return 0;
}
