/*
 * Avalon memory-mapped peripheral that generates VGA
 *
 *
 * 	Name / UNI
 * 		Daniel Mesko / dpm2153
 * 		Cansu Cabuk / cc4455
 *   	Alan Armero / aa3938
 */

module vga_display(input logic        clk,
	        input logic 	   reset,
		input logic [7:0]  writedata,
		input logic 	   write,
		input 		   chipselect,
		input logic [3:0]  address,

		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output logic 	   VGA_CLK, VGA_HS, VGA_VS,
		                   VGA_BLANK_n,
		output logic 	   VGA_SYNC_n);

   logic [10:0]	   hcount;
   logic [9:0]     vcount;

	
   logic [7:0]	   p1_x, p1_y, p2_x, p2_y, p1_health, p2_health;


   vga_counters counters(.clk50(clk), .*);

   always_ff @(posedge clk)
     if (reset) begin
		p1_x <= 8'd10;
		p1_y <= 8'd70;
		p2_x <= 8'd80;
		p2_y <= 8'd70;
		p1_health <= 8'd3;
		p2_health <= 8'd3;
     end else if (chipselect && write)
       case (address)
	 3'h0 : p1_x <= writedata;
	 3'h1 : p1_y <= writedata;
	 3'h2 : p2_x <= writedata;
	 3'h3 : p2_y <= writedata;
	 3'h4 : p1_health <= writedata;
	 3'h5 : p2_health <= writedata;
       endcase


   always_comb begin
	if (((hcount[10:3] >= p1_x - 3) && (hcount[10:3] <= p1_x + 3)) &&
		((vcount[9:2] >= p1_y -3 ) && (vcount[9:2] <= p1_y + 3)) )
	  {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'h00};
	else if (((hcount[10:3] >= p2_x - 3) && (hcount[10:3] <= p2_x + 3)) &&
		((vcount[9:2] >= p2_y -3 ) && (vcount[9:2] <= p2_y + 3)) )
	  {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'hff};
	else if ((((hcount[10:3] >= 15 - 2) && (hcount[10:3] <= 15 + 2)) &&
		((vcount[9:2] >= 10 -3 ) && (vcount[9:2] <= 10 + 3))) && (p1_health > 0) ) 
		 {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
	else if ((((hcount[10:3] >= 25 - 2) && (hcount[10:3] <= 25+ 2)) &&
		((vcount[9:2] >= 10 -3 ) && (vcount[9:2] <= 10 + 3))) && (p1_health > 1) )
		 {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
	else if ((((hcount[10:3] >= 35 - 2) && (hcount[10:3] <= 35 + 2)) &&
		((vcount[9:2] >= 10 -3 ) && (vcount[9:2] <= 10 + 3))) && (p1_health > 2) ) 
		 {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
	else if ((((hcount[10:3] >= 100 - 2) && (hcount[10:3] <= 100 + 2)) &&
		((vcount[9:2] >= 10 -3 ) && (vcount[9:2] <= 10 + 3))) && (p2_health > 0) ) 
		 {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
	else if ((((hcount[10:3] >= 110 - 2) && (hcount[10:3] <= 110 + 2)) &&
		((vcount[9:2] >= 10 -3 ) && (vcount[9:2] <= 10 + 3))) && (p2_health > 1) ) 
		 {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
	else if ((((hcount[10:3] >= 120 - 2) && (hcount[10:3] <= 120 + 2)) &&
		((vcount[9:2] >= 10 -3 ) && (vcount[9:2] <= 10 + 3))) && (p2_health > 2) ) 
		 {VGA_R, VGA_G, VGA_B} = {8'hff, 8'h00, 8'hff};
	else	
		{VGA_R, VGA_G, VGA_B} =
             {8'h00, 8'h00, 8'h00};
   end
	       
endmodule

module vga_counters(
 input logic 	     clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row
 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;
       
   logic endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule
