/*
 * Avalon memory-mapped peripheral that generates VGA
 *
 * Stephen A. Edwards
 * Columbia University
 */

module vga_ball(input logic        clk,
	        input logic 	   reset,
		input logic [15:0] writedata,
		input logic 	   write,
		input 		   chipselect,
		input logic [4:0]  address,

		output logic [7:0] VGA_R, VGA_G, VGA_B,
		output logic 	   VGA_CLK, VGA_HS, VGA_VS,
		                   VGA_BLANK_n,
		output logic 	   VGA_SYNC_n);

   logic [10:0]	   hcount;
   logic [9:0]     vcount;
   logic           tik;
   
   logic [15:0]    x0;
   logic [15:0]    y0;
   logic [15:0]    x1;
   logic [15:0]    y1;
   logic [15:0]    x2;
   logic [15:0]    y2;
   logic [15:0]    x3;
   logic [15:0]    y3;    

   logic [15:0]    frequency_0;
   logic [15:0]    size_0;
   logic [15:0]    frequency_1;
   logic [15:0]    size_1;
   logic [15:0]    frequency_2;
   logic [15:0]    size_2;
   logic [15:0]    frequency_3;
   logic [15:0]    size_3;
   logic [15:0]    frequency_4;
   logic [15:0]    size_4;
   
   logic [47:0]    freq1;
   logic [47:0]    size1;
   logic [47:0]    freq2;
   logic [47:0]    size2;
   logic [47:0]    freq3;
   logic [47:0]    size3;
   logic [47:0]    freq4;
   logic [47:0]    size4;
   logic [47:0]    freq5;
   logic [47:0]    size5;
/*
   logic en1;
   logic en2;
   logic en3;
   logic en4;
   logic en5;
   logic en6;
   logic en7;
   logic en8;
   logic en9;
   logic en10;
*/  
   logic [9:0]      en;
   
	
   vga_counters counters(.clk50(clk), .*);   
   pulse  pulse(.clk50(clk), .tiktok(tik), .reset(reset));

   shift_register_sub register_sub1(.clkreg(clk), .din(frequency_0), .dout(freq1), .en(en[0]), .reset(reset), .tiktok(tik));  //clkreg, din, dout, en
   shift_register_add register_add1(.clkreg(clk), .din(size_0), .dout(size1), .en(en[1]), .reset(reset), .tiktok(tik));

   shift_register_sub register_sub2(.clkreg(clk), .din(frequency_1), .dout(freq2), .en(en[2]), .reset(reset), .tiktok(tik));
   shift_register_add register_add2(.clkreg(clk), .din(size_1), .dout(size2), .en(en[3]), .reset(reset), .tiktok(tik));

   shift_register_sub register_sub3(.clkreg(clk), .din(frequency_2), .dout(freq3), .en(en[4]), .reset(reset), .tiktok(tik));
   shift_register_add register_add3(.clkreg(clk), .din(size_2), .dout(size3), .en(en[5]), .reset(reset), .tiktok(tik));

   shift_register_sub register_sub4(.clkreg(clk), .din(frequency_3), .dout(freq4), .en(en[6]), .reset(reset), .tiktok(tik));
   shift_register_add register_add4(.clkreg(clk), .din(size_3), .dout(size4), .en(en[7]), .reset(reset), .tiktok(tik));

   //shift_register_sub register_sub5(.clkreg(clk), .din(frequency_4), .dout(freq5), .en(en[8]), .reset(reset), .tiktok(tik));
   //shift_register_add register_add5(.clkreg(clk), .din(size_4), .dout(size5), .en(en[9]), .reset(reset), .tiktok(tik));

   always_ff @(posedge clk)
     if (reset) 
       begin
	en <= 10'd0;
	//tik <= 1'd0;
	frequency_0 <= 16'd120;   size_0 <= 16'd10;	
        frequency_1 <= 16'd220;   size_1 <= 16'd20;
	frequency_2 <= 16'd320;   size_2 <= 16'd30;
	frequency_3 <= 16'd420;   size_3 <= 16'd40;
	//frequency_4 <= 16'd520;   size_4 <= 16'd50;
	x0 <= 16'd300; y0 <= 16'd100;
	x1 <= 16'd800; y1 <= 16'd100;
	x2 <= 16'd300; y2 <= 16'd300;
	x3 <= 16'd800; y3 <= 16'd300;
	
       end 
     else if (chipselect && write)
       case (address)
	 4'h0 : begin frequency_0 <= writedata; en <= 10'b0000000001; end
         4'h1 : begin size_0 <= writedata; en <= 10'b0000000010; end

	 4'h2 : begin frequency_1 <= writedata; en <= 10'b0000000100; end
         4'h3 : begin size_1 <= writedata; en <= 10'b0000001000; end

	 4'h4 : begin frequency_2 <= writedata; en <= 10'b0000010000; end
         4'h5 : begin size_2 <= writedata; en <= 10'b0000100000; end

	 4'h6 : begin frequency_3 <= writedata; en <= 10'b0001000000; end
         4'h7 : begin size_3 <= writedata; en <= 10'b0010000000; end

	 4'h8  : x0 <= writedata;
         4'h9  : y0 <= writedata;

	 4'hA : x1 <= writedata;
         4'hB : y1 <= writedata;
	 4'hC : x2 <= writedata;
         4'hD : y2 <= writedata;
	 4'hE : x3 <= writedata;
         4'hF : y3 <= writedata;
	 
	 
       endcase
      else
	 en <= 0; 

   integer back_r0, back_g0, back_b0;
   integer rx_0, ry_0;
   integer back_r1, back_g1, back_b1;
   integer rx_1, ry_1;
   integer back_r2, back_g2, back_b2;
   integer rx_2, ry_2;
   integer back_r3, back_g3, back_b3;
   integer rx_3, ry_3;
   integer back_r4, back_g4, back_b4;
   integer rx_4, ry_4;
   integer back_r5, back_g5, back_b5;
   integer rx_5, ry_5;
   integer back_r6, back_g6, back_b6;
   integer rx_6, ry_6;
   integer back_r7, back_g7, back_b7;
   integer rx_7, ry_7;
   integer back_r8, back_g8, back_b8;
   integer rx_8, ry_8;
   integer back_r9, back_g9, back_b9;
   integer rx_9, ry_9;
   integer back_r10, back_g10, back_b10;
   integer rx_10, ry_10;
   integer back_r11, back_g11, back_b11;
   integer rx_11, ry_11;
   /*integer back_r12, back_g12, back_b12;
   integer rx_12, ry_12;
   integer back_r13, back_g13, back_b13;
   integer rx_13, ry_13;
   integer back_r14, back_g14, back_b14;
   integer rx_14, ry_14;
*/
   always_comb begin
     {VGA_R, VGA_G, VGA_B} = {8'h0, 8'h0, 8'h0};
     rx_0 = hcount - x0;
     ry_0 = vcount - y0;

     rx_1 = hcount - x0;
     ry_1 = vcount - y0;

     rx_2 = hcount - x0;
     ry_2 = vcount - y0;

     rx_3 = hcount - x1;
     ry_3 = vcount - y1;

     rx_4 = hcount - x1;
     ry_4 = vcount - y1;

     rx_5 = hcount - x1;
     ry_5 = vcount - y1;

     rx_6 = hcount - x2;
     ry_6 = vcount - y2;

     rx_7 = hcount - x2;
     ry_7 = vcount - y2;

     rx_8 = hcount - x2;
     ry_8 = vcount - y2;

     rx_9 = hcount - x3;
     ry_9 = vcount - y3;

     rx_10 = hcount - x3;
     ry_10 = vcount - y3;

     rx_11 = hcount - x3;
     ry_11 = vcount - y3;

/////////////////////////////////////////////////
     if (freq1[15:0] >= 16'd255)
	back_r0 = 8'd255;
     else
	back_r0 = freq1[15:0];

     if (freq1[15:0] >= 16'd510 && freq1[15:0] < 16'd765)
	back_g0 = 8'd255;
     else if (freq1[15:0] < 16'd510 && freq1[15:0] >= 8'd255)
	back_g0 = freq1[15:0] - 8'd255;
     else
	back_g0 = 0;

     if (freq1[15:0] >= 16'd510)
	back_b0 = freq1[15:0] -16'd510;
     else
	back_b0 = 0;

    //////////////////////////////////////////////

     if (freq1[31:16] >= 16'd255)
	back_r1 = 8'd255;
     else
	back_r1 = freq1[31:16];

     if (freq1[31:16] >= 16'd510 && freq1[31:16] < 16'd765)
	back_g1 = 8'd255;
     else if (freq1[31:16] < 16'd510 && freq1[31:16] >= 8'd255)
	back_g1 = freq1[31:16] - 8'd255;
     else
	back_g1 = 0;

     if (freq1[31:16] >= 16'd510)
	back_b1 = freq1[31:16] - 16'd510;
     else
	back_b1 = 0; 

///////////////////////////////////////////////////////////

     if (freq1[47:32] >= 16'd255)
	back_r2 = 8'd255;
     else
	back_r2 = freq1[47:32];

     if (freq1[47:32] >= 16'd510 && freq1[47:32] < 16'd765)
	back_g2 = 8'd255;
     else if (freq1[47:32] < 16'd510 && freq1[47:32] >= 8'd255)
	back_g2 = freq1[47:32] - 8'd255;
     else
	back_g2 = 0;

     if (freq1[47:32] >= 16'd510)
	back_b2 = freq1[47:32] - 16'd510;
     else
	back_b2 = 0; 

/////////////////////////////////////////////////////////////////

     if (freq2[15:0] >= 16'd255)
	back_g3 = 8'd255;
     else
	back_g3 = freq2[15:0];

     if (freq2[15:0] >= 16'd510 && freq2[15:0] < 16'd765)
	back_r3 = 8'd255;
     else if (freq2[15:0] < 16'd510 && freq2[15:0] >= 8'd255)
	back_r3 = freq2[15:0] - 8'd255;
     else
	back_r3 = 0;

     if (freq2[15:0] >= 16'd510)
	back_b3 = freq2[15:0] -16'd510;
     else
	back_b3 = 0;

    //////////////////////////////////////////////

     if (freq2[31:16] >= 16'd255)
	back_g4 = 8'd255;
     else
	back_g4 = freq2[31:16];

     if (freq2[31:16] >= 16'd510 && freq2[31:16] < 16'd765)
	back_r4 = 8'd255;
     else if (freq2[31:16] < 16'd510 && freq2[31:16] >= 8'd255)
	back_r4 = freq2[31:16] - 8'd255;
     else
	back_r4 = 0;

     if (freq2[31:16] >= 16'd510)
	back_b4 = freq2[31:16] - 16'd510;
     else
	back_b4 = 0; 

///////////////////////////////////////////////////////////

     if (freq2[47:32] >= 16'd255)
	back_g5 = 8'd255;
     else
	back_g5 = freq2[47:32];

     if (freq2[47:32] >= 16'd510 && freq2[47:32] < 16'd765)
	back_r5 = 8'd255;
     else if (freq2[47:32] < 16'd510 && freq2[47:32] >= 8'd255)
	back_r5 = freq2[47:32] - 8'd255;
     else
	back_r5 = 0;

     if (freq2[47:32] >= 16'd510)
	back_b5 = freq2[47:32] - 16'd510;
     else
	back_b5 = 0; 

/////////////////////////////////////////////////////////////////

     if (freq3[15:0] >= 16'd255)
	back_b6 = 8'd255;
     else
	back_b6 = freq3[15:0];

     if (freq3[15:0] >= 16'd510 && freq3[15:0] < 16'd765)
	back_g6 = 8'd255;
     else if (freq3[15:0] < 16'd510 && freq3[15:0] >= 8'd255)
	back_g6 = freq3[15:0] - 8'd255;
     else
	back_g6 = 0;

     if (freq3[15:0] >= 16'd510)
	back_r6 = freq3[15:0] -16'd510;
     else
	back_r6 = 0;

    //////////////////////////////////////////////

     if (freq3[31:16] >= 16'd255)
	back_b7 = 8'd255;
     else
	back_b7 = freq3[31:16];

     if (freq3[31:16] >= 16'd510 && freq3[31:16] < 16'd765)
	back_g7 = 8'd255;
     else if (freq3[31:16] < 16'd510 && freq3[31:16] >= 8'd255)
	back_g7 = freq3[31:16] - 8'd255;
     else
	back_g7 = 0;

     if (freq3[31:16] >= 16'd510)
	back_r7 = freq3[31:16] - 16'd510;
     else
	back_r7 = 0; 

///////////////////////////////////////////////////////////

     if (freq3[47:32] >= 16'd255)
	back_b8 = 8'd255;
     else
	back_b8 = freq3[47:32];

     if (freq3[47:32] >= 16'd510 && freq3[47:32] < 16'd765)
	back_g8 = 8'd255;
     else if (freq3[47:32] < 16'd510 && freq3[47:32] >= 8'd255)
	back_g8 = freq3[47:32] - 8'd255;
     else
	back_g8 = 0;

     if (freq3[47:32] >= 16'd510)
	back_r8 = freq3[47:32] - 16'd510;
     else
	back_r8 = 0; 

/////////////////////////////////////////////////////////////////

     if (freq4[15:0] >= 16'd255)
	back_b9 = 8'd255;
     else
	back_b9 = freq4[15:0];

     if (freq4[15:0] >= 16'd510 && freq4[15:0] < 16'd765)
	back_r9 = 8'd255;
     else if (freq4[15:0] < 16'd510 && freq4[15:0] >= 8'd255)
	back_r9 = freq4[15:0] - 8'd255;
     else
	back_r9 = 0;

     if (freq4[15:0] >= 16'd510)
	back_g9 = freq4[15:0] -16'd510;
     else
	back_g9 = 0;

    //////////////////////////////////////////////

     if (freq4[31:16] >= 16'd255)
	back_b10 = 8'd255;
     else
	back_b10 = freq4[31:16];

     if (freq4[31:16] >= 16'd510 && freq4[31:16] < 16'd765)
	back_r10 = 8'd255;
     else if (freq4[31:16] < 16'd510 && freq4[31:16] >= 8'd255)
	back_r10 = freq4[31:16] - 8'd255;
     else
	back_r10 = 0;

     if (freq4[31:16] >= 16'd510)
	back_g10 = freq4[31:16] - 16'd510;
     else
	back_g10 = 0; 

///////////////////////////////////////////////////////////

     if (freq4[47:32] >= 16'd255)
	back_b11 = 8'd255;
     else
	back_b11 = freq4[47:32];

     if (freq4[47:32] >= 16'd510 && freq4[47:32] < 16'd765)
	back_r11 = 8'd255;
     else if (freq4[47:32] < 16'd510 && freq4[47:32] >= 8'd255)
	back_r11 = freq4[47:32] - 8'd255;
     else
	back_r11 = 0;

     if (freq4[47:32] >= 16'd510)
	back_g11 = freq4[47:32] - 16'd510;
     else
	back_g11 = 0; 

/////////////////////////////////////////////////////////////////

     /*if (freq5[15:0] >= 16'd255)
	back_r12 = 8'd255;
     else
	back_r12 = freq5[15:0];

     if (freq5[15:0] >= 16'd510 && freq5[15:0] < 16'd765)
	back_g12 = 8'd255;
     else if (freq5[15:0] < 16'd510 && freq5[15:0] >= 8'd255)
	back_g12 = freq5[15:0] - 8'd255;
     else
	back_g12 = 0;

     if (freq5[15:0] >= 16'd510)
	back_b12 = freq5[15:0] -16'd510;
     else
	back_b12 = 0;

    //////////////////////////////////////////////

     if (freq5[31:16] >= 16'd255)
	back_r13 = 8'd255;
     else
	back_r13 = freq5[31:16];

     if (freq5[31:16] >= 16'd510 && freq5[31:16] < 16'd765)
	back_g13 = 8'd255;
     else if (freq5[31:16] < 16'd510 && freq5[31:16] >= 8'd255)
	back_g13 = freq5[31:16] - 8'd255;
     else
	back_g13 = 0;

     if (freq5[31:16] >= 16'd510)
	back_b13 = freq5[31:16] - 16'd510;
     else
	back_b13 = 0; 

///////////////////////////////////////////////////////////

     if (freq5[47:32] >= 16'd255)
	back_r14 = 8'd255;
     else
	back_r14 = freq5[47:32];

     if (freq5[47:32] >= 16'd510 && freq5[47:32] < 16'd765)
	back_g14 = 8'd255;
     else if (freq5[47:32] < 16'd510 && freq5[47:32] >= 8'd255)
	back_g14 = freq5[47:32] - 8'd255;
     else
	back_g14 = 0;

     if (freq5[47:32] >= 16'd510)
	back_b14 = freq5[47:32] - 16'd510;
     else
	back_b14 = 0; 
*/
/////////////////////////////////////////////////////////////////
     if (VGA_BLANK_n)
	     if (rx_0*rx_0/4 + ry_0*ry_0 >= size1[15:0]*size1[15:0] && rx_0*rx_0/4 + ry_0*ry_0 <= (size1[15:0]+1)*(size1[15:0]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r0[7:0], back_g0[7:0], back_b0[7:0]};

	     else if (rx_1*rx_1/4 + ry_1*ry_1 >= size1[31:16]*size1[31:16] && rx_1*rx_1/4 + ry_1*ry_1 <= (size1[31:16]+1)*(size1[31:16]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r1[7:0], back_g1[7:0], back_b1[7:0]};

	     else if (rx_2*rx_2/4 + ry_2*ry_2 >= size1[47:32]*size1[47:32] && rx_2*rx_2/4 + ry_2*ry_2 <= (size1[47:32]+1)*(size1[47:32]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r2[7:0], back_g2[7:0], back_b2[7:0]};


	     else if (rx_3*rx_3/4 + ry_3*ry_3 >= size2[15:0]*size2[15:0] && rx_3*rx_3/4 + ry_3*ry_3 <= (size2[15:0]+1)*(size2[15:0]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r3[7:0], back_g3[7:0], back_b3[7:0]};

	     else if (rx_4*rx_4/4 + ry_4*ry_4 >= size2[31:16]*size2[31:16] && rx_4*rx_4/4 + ry_4*ry_4 <= (size2[31:16]+1)*(size2[31:16]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r4[7:0], back_g4[7:0], back_b4[7:0]};

	     else if (rx_5*rx_5/4 + ry_5*ry_5 >= size2[47:32]*size2[47:32] && rx_5*rx_5/4 + ry_5*ry_5 <= (size2[47:32]+1)*(size2[47:32]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r5[7:0], back_g5[7:0], back_b5[7:0]};


	     else if (rx_6*rx_6/4 + ry_6*ry_6 >= size3[15:0]*size3[15:0] && rx_6*rx_6/4 + ry_6*ry_6 <= (size3[15:0]+1)*(size3[15:0]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r6[7:0], back_g6[7:0], back_b6[7:0]};

	     else if (rx_7*rx_7/4 + ry_7*ry_7 >= size3[31:16]*size3[31:16] && rx_7*rx_7/4 + ry_7*ry_7 <= (size3[31:16]+1)*(size3[31:16]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r7[7:0], back_g7[7:0], back_b7[7:0]};

	     else if (rx_8*rx_8/4 + ry_8*ry_8 >= size3[47:32]*size3[47:32] && rx_8*rx_8/4 + ry_8*ry_8 <= (size3[47:32]+1)*(size3[47:32]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r8[7:0], back_g8[7:0], back_b8[7:0]};


	     else if (rx_9*rx_9/4 + ry_9*ry_9 >= size4[15:0]*size4[15:0] && rx_9*rx_9/4 + ry_9*ry_9 <= (size4[15:0]+1)*(size4[15:0]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r9[7:0], back_g9[7:0], back_b9[7:0]};

	     else if (rx_10*rx_10/4 + ry_10*ry_10 >= size4[31:16]*size4[31:16] && rx_10*rx_10/4 + ry_10*ry_10 <= (size4[31:16]+1)*(size4[31:16]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r10[7:0], back_g10[7:0], back_b10[7:0]};

	     else if (rx_11*rx_11/4 + ry_11*ry_11 >= size4[47:32]*size4[47:32] && rx_11*rx_11/4 + ry_11*ry_11 <= (size4[47:32]+1)*(size4[47:32]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r11[7:0], back_g11[7:0], back_b11[7:0]};

	     /*
	     else if (rx_12*rx_12/4 + ry_12*ry_9 >= size5[15:0]*size5[15:0] && rx_12*rx_12/4 + ry_12*ry_12 <= (size5[15:0]+1)*(size5[15:0]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r12[7:0], back_g12[7:0], back_b12[7:0]};

	     else if (rx_13*rx_13/4 + ry_13*ry_13 >= size5[31:16]*size5[31:16] && rx_13*rx_13/4 + ry_13*ry_13 <= (size5[31:16]+1)*(size5[31:16]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r13[7:0], back_g13[7:0], back_b13[7:0]};

	     else if (rx_14*rx_14/4 + ry_14*ry_14 >= size5[47:32]*size5[47:32] && rx_14*rx_14/4 + ry_14*ry_14 <= (size5[47:32]+1)*(size5[47:32]+1))
		  {VGA_R, VGA_G, VGA_B} = {back_r14[7:0], back_g14[7:0], back_b14[7:0]}; */

	     else
		  {VGA_R, VGA_G, VGA_B} ={8'h0, 8'h0, 8'h0};


   end
	       
endmodule





module vga_counters(
 input logic 	     clk50, reset,
 output logic [10:0] hcount,  // hcount[10:1] is pixel column
 output logic [9:0]  vcount,  // vcount[9:0] is pixel row

 output logic 	     VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC +
                            HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC +
                            VBACK_PORCH; // 525

   logic endOfLine;  

   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;
       
   logic endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) &
		      !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1'b0; // For putting sync on the green signal; unused
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: rising edge sensitive
   
endmodule






module shift_register_sub(clkreg, din, dout, en, reset, tiktok);

	input clkreg;
	input en;
	input reset;
	input tiktok;
	input [15:0] din;
	output [47:0] dout;

	wire [47:0] dout;
	reg [47:0] qtemp;
	always @ (posedge clkreg)
	   begin
		if (reset)
		  begin
			qtemp[15:0]  <= 220;
			qtemp[31:16] <= 400;
			qtemp[47:32] <= 600;
		  end
		else if (en == 1)
	   		qtemp <= {qtemp[31:0], din};
			
		else if (tiktok)
		  begin
			if(qtemp[15:0]>3)
				qtemp[15:0]  <= qtemp[15:0]  - 2'd2;
			//else
			//	qtemp[15:0]  <= qtemp[15:0];

			if(qtemp[31:16]>3)
				qtemp[31:16] <= qtemp[31:16] - 2'd2;
			//else
			//	qtemp[31:16] <= qtemp[31:16];

			if(qtemp[47:32]>3)
				qtemp[47:32] <= qtemp[47:32] - 2'd2;
			//else
			//	qtemp[47:31] <= qtemp[47:31];

		  end
	   end
	assign dout = qtemp;
endmodule

module shift_register_add(clkreg, din, dout, en, reset, tiktok);

	input clkreg;
	input en;
	input reset;
	input tiktok;
	input [15:0] din;
	output [47:0] dout;
	wire [47:0] dout;
	reg [47:0] qtemp;
	always @ (posedge clkreg)
	   begin
		if (reset)
		    begin
			qtemp[15:0]  <= 20;
			qtemp[31:16] <= 20;
			qtemp[47:32] <= 20;
		    end
		else if (en == 1)
	   		qtemp <= {qtemp[31:0], din};
		else if (tiktok)
		    begin
			if(qtemp[15:0]<765)
				qtemp[15:0]  <= qtemp[15:0]  + 1'd1;
			//else
			//	qtemp[15:0]  <= qtemp[15:0];

			if(qtemp[31:16]<765)
				qtemp[31:16] <= qtemp[31:16] + 1'd1;
			//else
			//	qtemp[31:16] <= qtemp[31:16];

			if(qtemp[47:32]<765)
				qtemp[47:32] <= qtemp[47:32] + 1'd1;
			//else
			//	qtemp[47:31] <= qtemp[47:31];
		    end
	   end
	
	assign dout = qtemp;
endmodule

module pulse(clk50, tiktok, reset);
	input clk50;
	input reset;
	output tiktok;
	reg [23:0] count;
	
	always @ (posedge clk50)
	  begin
		if(reset)
			count <= 1'd0;
		else if (count > 20'd300000)
		    begin
			tiktok <= 1;
			count <= 0;
		    end
		else
		    begin
			count <= count + 1'd1;
			tiktok <= 0;
		    end	
	  end
endmodule
	 
