/*
 * Userspace program that communicates with the hardware driver
 * through ioctls
 *
 * Final project
 * CSEE 4840 Embedded System
 * Columbia University
 */

#include <stdio.h>
#include <stdlib.h>
#include "vga_ball.h"
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

#include <inttypes.h>
#include <alsa/asoundlib.h>
#include <math.h>
#include <fftw3.h>
#include <stdint.h>

#define N 16384
#define COLORS 9

int vga_ball_fd;

/* send info to hardware */
void set_background_color(const vga_ball_color_t *c)
{
  vga_ball_arg_t vla;
  vla.background = *c;
  if (ioctl(vga_ball_fd, VGA_BALL_WRITE_BACKGROUND, &vla)) {
      perror("ioctl(VGA_BALL_SET_BACKGROUND) failed");
      return;
  }
}


int main(int argc, char *argv[])
{
  /* set parameters for audio input ------------------------------------------------------*/ 
  int i,j;
  int o;
  int err;
  char *buffer;
  int buffer_frames = 16384;
  unsigned int rate = 44100;
  snd_pcm_t *capture_handle;
  snd_pcm_hw_params_t *hw_params;
  snd_pcm_format_t format = SND_PCM_FORMAT_S16_LE;//SND_PCM_FORMAT_S16_LE;
  /*--------------------------------------------------------------------------------------*/

  /* set up audio input interfaces -------------------------------------------------------*/
  if ((err = snd_pcm_open (&capture_handle, argv[1], SND_PCM_STREAM_CAPTURE, 0)) < 0) {
    fprintf (stderr, "cannot open audio device %s (%s)\n", 
             argv[1],
             snd_strerror (err));
    exit (1);
  }

  fprintf(stdout, "audio interface opened\n");
       
  if ((err = snd_pcm_hw_params_malloc (&hw_params)) < 0) {
    fprintf (stderr, "cannot allocate hardware parameter structure (%s)\n",
             snd_strerror (err));
    exit (1);
  }

  fprintf(stdout, "hw_params allocated\n");
         
  if ((err = snd_pcm_hw_params_any (capture_handle, hw_params)) < 0) {
    fprintf (stderr, "cannot initialize hardware parameter structure (%s)\n",
             snd_strerror (err));
    exit (1);
  }

  fprintf(stdout, "hw_params initialized\n");
  
  if ((err = snd_pcm_hw_params_set_access (capture_handle, hw_params, SND_PCM_ACCESS_RW_INTERLEAVED)) < 0) {
    fprintf (stderr, "cannot set access type (%s)\n",
             snd_strerror (err));
    exit (1);
  }

  fprintf(stdout, "hw_params access setted\n");
  
  if ((err = snd_pcm_hw_params_set_format (capture_handle, hw_params, format)) < 0) {
    fprintf (stderr, "cannot set sample format (%s)\n",
             snd_strerror (err));
    exit (1);
  }

  fprintf(stdout, "hw_params format setted\n");
  
  if ((err = snd_pcm_hw_params_set_rate_near (capture_handle, hw_params, &rate, 0)) < 0) {
    fprintf (stderr, "cannot set sample rate (%s)\n",
             snd_strerror (err));
    exit (1);
  }
  
  fprintf(stdout, "hw_params rate setted\n");

  if ((err = snd_pcm_hw_params_set_channels (capture_handle, hw_params, 1)) < 0) {
    fprintf (stderr, "cannot set channel count (%s)\n",
             snd_strerror (err));
    exit (1);
  }

  fprintf(stdout, "hw_params channels setted\n");
  
  if ((err = snd_pcm_hw_params (capture_handle, hw_params)) < 0) {
    fprintf (stderr, "cannot set parameters (%s)\n",
             snd_strerror (err));
    exit (1);
  }

  fprintf(stdout, "hw_params setted\n");
  
  snd_pcm_hw_params_free (hw_params);

  fprintf(stdout, "hw_params freed\n");
  
  if ((err = snd_pcm_prepare (capture_handle)) < 0) {
    fprintf (stderr, "cannot prepare audio interface for use (%s)\n",
             snd_strerror (err));
    exit (1);
  }

  fprintf(stdout, "audio interface prepared\n");
  /*--------------------------------------------------------------------------------------*/
  
  /* set parameters for FFT --------------------------------------------------------------*/
  short dis;
  double in[N];
  double output[N][2];
  double output_m[N/2];
  fftw_plan     p1;

  double max1[3];
  double max2[3];
  double max3[3];
  double max4[3];
  //double max5[3];

  short a_pow, a_fre, b_pow, b_fre, c_pow, c_fre, d_pow, d_fre, e_pow, e_fre;
  /*--------------------------------------------------------------------------------------*/

  /* set parameters for circle movement --------------------------------------------------------------*/
  float angle1 = 1.57;
  float angle2 = 3.14;
  float angle3 = 4.71;
  float angle4 = 0.0;

  float r = 120;

  short x1 = (640 / 2 + cos(angle1) * r) *2;
  short y1 = 480 / 2 + sin(angle1) * r;
  short x2 = (640 / 2 + cos(angle2) * r) *2;
  short y2 = 480 / 2 + sin(angle2) * r;
  short x3 = (640 / 2 + cos(angle3) * r) *2;
  short y3 = 480 / 2 + sin(angle3) * r;
  short x4 = (640 / 2 + cos(angle4) * r) *2;
  short y4 = 480 / 2 + sin(angle4) * r;
  /*--------------------------------------------------------------------------------------*/
  
  /* set parameters for hardware ---------------------------------------------------------*/
  vga_ball_arg_t vla;
  static const char filename[] = "/dev/vga_ball";
  /*--------------------------------------------------------------------------------------*/

  /* start hardware ----------------------------------------------------------------------*/
  if ( (vga_ball_fd = open(filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename);
    return -1;
  }
  printf("initial state: ");
  /*--------------------------------------------------------------------------------------*/


  while(1){
    
    /* allocate buffer for audio input */
    buffer = malloc(16384 * snd_pcm_format_width(format) / 8 * 2);
    fprintf(stdout, "buffer allocated\n");

    /* read and store the audio input */
    if ((err = snd_pcm_readi (capture_handle, buffer, buffer_frames)) != buffer_frames) 
         fprintf (stderr, "read from audio interface failed (%s)\n",
                  snd_strerror (err));

    /* input buffer decoding */
    for (int n=0, j=0; n<N; n++,j+=2){
        dis = buffer[j+1];
        dis <<= 8;
        dis += buffer[j];
        in[n] = (double)dis;
    }

    /* excute FFT */
    p1 = fftw_plan_dft_r2c_1d(N, in, output, FFTW_ESTIMATE);    //p1 = fftw_plan_dft_1d(N, in, output, FFTW_FORWARD, FFTW_PATIENT);
    fftw_execute(p1);

    /* calculate amplitude / power */
    for (int i = 60; i < 8192; i++) {
        output_m[i] = sqrt(pow(output[i][0],2) + pow(output[i][1], 2));
    }

    ///* FFT output display */
    // for (int i = 0; i < N/2; i++) {
    //       printf("%4d %15.10f %15.10f\n", i, in[i], output_m[i]);
    // }

    /* select the max amplitude value in five frequency range ------------------------------*/
    for (i = 0; i<400; i++){
        if(output_m[i]>max1[1]){
            max1[0] = i;
            max1[1] = output_m[i];
            max1[2] = 44100/16384 * i;
        }
    }
    for (i = 400; i<1000; i++){
        if(output_m[i]>max2[1]){
            max2[0] = i;
            max2[1] = output_m[i];
            max2[2] = 44100/16384 * i;
        }
    }

    for (i = 1000; i<2200; i++){
        if(output_m[i]>max3[1]){
            max3[0] = i;
            max3[1] = output_m[i];
            max3[2] = 44100/16384 * i;
        }
    }

    for (i = 2200; i<4096; i++){
        if(output_m[i]>max4[1]){
            max4[0] = i;
            max4[1] = output_m[i];
            max4[2] = 44100/16384 * i;
        }
    }

    // for (i = 2454; i<4096; i++){
    //     if(output_m[i]>max5[1]){
    //         max5[0] = i;
    //         max5[1] = output_m[i];
    //         max5[2] = 44100/16384 * i;
    //     }
    // }
    /*--------------------------------------------------------------------------------------*/

    ///* max value selection display */
    // printf("%4.2f, %15.10f, %15.10f\n", max1[0], max1[1], max1[2]);
    // printf("%4.2f, %15.10f, %15.10f\n", max2[0], max2[1], max2[2]);
    // printf("%4.2f, %15.10f, %15.10f\n", max3[0], max3[1], max3[2]);
    // printf("%4.2f, %15.10f, %15.10f\n", max4[0], max4[1], max4[2]);
    // printf("%4.2f, %15.10f, %15.10f\n", max5[0], max5[1], max5[2]);
    
    /* max value and frequency modulation */
    a_pow = (short)(log(max1[1]) / log(2) * 6 - 60 - 20);
    a_fre = (short)(max1[0] / 22 + 450);
    b_pow = (short)(log(max2[1]) / log(2) * 6 - 60 - 7);
    b_fre = (short)(max2[0] / 22 + 450);
    c_pow = (short)(log(max3[1]) / log(2) * 6 - 60 - 5);
    c_fre = (short)(max3[0] / 22 + 450);
    d_pow = (short)(log(max4[1]) / log(2) * 6 - 60 - 5);
    d_fre = (short)(max4[0] / 22 + 450);
    // e_pow = (short)(log((short)max5[1]) / log(2) *3);
    // e_fre = (short)(max5[0] / 22 + 350);

    /* modulation display */
    printf("%d, %d\n", a_pow, a_fre);
    printf("%d, %d\n", b_pow, b_fre);
    printf("%d, %d\n", c_pow, c_fre);
    printf("%d, %d\n", d_pow, d_fre);
    // printf("%d, %d\n", e_pow, e_fre);

    /* free resources */
    fftw_destroy_plan(p1);
    free(buffer);
    fprintf(stdout, "buffer freed\n");

    /* circle movement */
    if(angle1 < 6.28)
    	angle1 = angle1 + 0.05;
    else
        angle1 = 0;

    if(angle2 < 6.28)
    	angle2 = angle2 + 0.05;
    else
        angle2 = 0;

    if(angle3 < 6.28)
    	angle3 = angle3 + 0.05;
    else
        angle3 = 0;

    if(angle4 < 6.28)
    	angle4 = angle4 + 0.05;
    else
        angle4 = 0; 

    x1 = (short)(640 / 2 + cos(angle1) * r) *2;
    y1 = (short)(480 / 2 + sin(angle1) * r);
    x2 = (short)(640 / 2 + cos(angle2) * r) *2;
    y2 = (short)(480 / 2 + sin(angle2) * r);
    x3 = (short)(640 / 2 + cos(angle3) * r) *2;
    y3 = (short)(480 / 2 + sin(angle3) * r);
    x4 = (short)(640 / 2 + cos(angle4) * r) *2;
    y4 = (short)(480 / 2 + sin(angle4) * r);

    /* fill the hw-sw info format */
    vga_ball_color_t colors = 
        {a_fre, a_pow, b_fre, b_pow, c_fre, c_pow, d_fre, d_pow, 
         x1,    y1,    x2,    y2,    x3,    y3,    x4,    y4   };
    printf("VGA ball Userspace program started\n");
    
    /* print the real-time positions of the 4 circles */
    printf("%d %d\n", x1, y1);
	printf("%d %d\n", x2, y2);
	printf("%d %d\n", x3, y3);
	printf("%d %d\n", x4, y4);

    /* update hardware status */     
    set_background_color(&colors);
    
    /* reset */
    max1[1] = 0;
    max2[1] = 0;
    max3[1] = 0;
    max4[1] = 0;
    //max5[1] = 0;
  }

    
  snd_pcm_close (capture_handle);
  fprintf(stdout, "audio interface closed\n");
  printf("VGA BALL Userspace program terminating\n");
  return 0;
}
