#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "Vprocessor.h"
#include "verilated.h"
#include "verilator_test.h"

#define PROC(x) processor__DOT__ ## x
#define VCPU(x) processor__DOT__c__DOT__ ## x

#define S_FETCH 0
#define S_EXECUTE 1
#define S_EXECUTE2 2

void print_arr(SData *arr, int len) {
    for(int i = 0;i < len; ++i)
        printf("%04x ", arr[i]);
}

void print_mem(CData *arr, int len) {
    for(int i = 0;i < len; ++i)
        printf("%02x ", arr[i]);
}

void print_registers(Vprocessor *tb) {
    printf("Misc. registers: state=%02x counter=%02x, N=%02x X=%02x I=%02x D=%02x\n",
            tb->processor__DOT__c__DOT__state,
            tb->processor__DOT__c__DOT__clock_counter,
            tb->processor__DOT__c__DOT__N_register,
            tb->processor__DOT__c__DOT__X_register,
            tb->processor__DOT__c__DOT__I_register,
            tb->processor__DOT__c__DOT__D_register);

    printf("R_registers: ");
    print_arr(tb->processor__DOT__c__DOT__R_registers, 16);
}

void print_state(Vprocessor *tb) {
    print_registers(tb);
    printf("\nMemory: ");
    print_mem(tb->processor__DOT__m__DOT__mem, 32);
    printf("\n\n");
}

void clock_cycle(Vprocessor *tb) {
    tb->clk = 0;
    tb->eval();
    tb->clk = 1;
    tb->eval();
}

void machine_cycle(Vprocessor *tb) {
    for (int i = 0; i < 3; i++) {
        clock_cycle(tb);
    }
}

int main(int argc, char *argv[])
{
    const unsigned char prgrm[32] = {
        // Copy zero to R(4), R(6), R(7)
        0xF8, // LDI
        0x00,
        0xA4, // PLO D -> R(4)
        0xB4, // PHI D -> R(4)
        0xA6, // PLO D -> R(6)
        0xB6, // PHI D -> R(6)
        0xA7, // PLO D -> R(7)
        0xB7, // PHI D -> R(7)

        // Increment R(6) and copy to R(4)
        0x16, // INC R(6)
        0x86, // GLO R(6) -> D
        0xA4, // PLO D -> R(4)
        0x96, // GHI R(6) -> D
        0xB4, // PHI D -> R(4)

        // Keep incrementing until Q == 1
        0x39, // BNQ: Q == 0?
        0x08,

        // If Q == 1, start over
        0x7A, // REQ
        0x30, // BR
        0x00
    };

    Verilated::commandArgs(argc, argv);

    Vprocessor *tb = new Vprocessor;

    memcpy(tb->processor__DOT__m__DOT__mem, prgrm, 32);
    memset(tb->VCPU(R_registers), 0xCC, 32);

    tb->reset = 1;
    clock_cycle(tb);

    assert(tb->VCPU(R_registers[0]) == 0);
    assert(tb->VCPU(I_register) == 0);
    assert(tb->VCPU(N_register) == 0);
    assert(tb->VCPU(P_register) == 0);
    assert(tb->VCPU(Q) == 0);
    assert(tb->PROC(bus_from_cpu) == 0);
    assert(tb->PROC(ram_we) == 0);
    assert(tb->PROC(address_from_cpu) == 0);
    assert(tb->VCPU(clock_counter) == 0);
    assert(tb->VCPU(state) == S_FETCH);

    tb->reset = 0;
    assert(tb->VCPU(clock_counter) == 0);

    machine_cycle(tb);
    assert(tb->VCPU(state) == S_EXECUTE);
    // Just fetched first instruction and now ready to execute it

    machine_cycle(tb);
    assert(tb->VCPU(state) == S_FETCH);
    // Just executed first instruction, ready to fetch next one

    machine_cycle(tb);
    assert(tb->VCPU(state) == S_EXECUTE);
    assert(tb->VCPU(R_registers[0]) == 3);

    for (int i = 0; i < 6 * 2; i++) {
        machine_cycle(tb);
    }

    assert(tb->VCPU(state) == S_EXECUTE);
    assert(tb->VCPU(R_registers[0]) == 9);
    assert(tb->VCPU(I_register) == 0x1);
    assert(tb->VCPU(N_register) == 0x6);

    // Simulate the increment loop a few times and make sure CPU output matches
    int x = 0;
    while (x < 5) {
        assert(tb->VCPU(Q) == 0);

        x++;
        // Run the increment instructions
        for (int i = 0; i < 2 * 5; i++) {
            machine_cycle(tb);
        }

        assert(tb->VCPU(R_registers[0]) == 14);
        assert(tb->VCPU(R_registers[6]) == x);
        assert(tb->VCPU(R_registers[4]) == x);

        // Check the branch condition
        machine_cycle(tb);
        assert(tb->VCPU(R_registers[0]) == 8);
        machine_cycle(tb);
        assert(tb->VCPU(R_registers[0]) == 9);
    }

    // Set Q to 1 and make sure program restarts
    tb->VCPU(Q) = 1;

    for (int i = 0; i < 2 * 6; i++) {
        machine_cycle(tb);
    }

    assert(tb->VCPU(R_registers[0]) == 16);

    machine_cycle(tb);
    machine_cycle(tb);

    assert(tb->VCPU(Q) == 0);

    machine_cycle(tb);
    machine_cycle(tb);

    assert(tb->VCPU(R_registers[0]) == 1);

    for (int i = 0; i < 2 * 7; i++) {
        machine_cycle(tb);
    }

    assert(tb->VCPU(R_registers[4]) == 0);
    assert(tb->VCPU(R_registers[6]) == 0);
    assert(tb->VCPU(R_registers[7]) == 0);

    printf("Counter test passed\n");

    return 0;
}
